package buzz.getcoco.iot.android;

import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.common.collect.ImmutableList;

import java.util.Collection;
import java.util.List;

import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.ResourceCondition;
import buzz.getcoco.iot.Rule;
import buzz.getcoco.iot.ScheduleCondition;

public class RuleEx extends Rule {

  private transient Identifier identifier;

  private transient ImmutableList<Condition> conditions;
  private transient MutableLiveData<String> nameObservable;
  private transient MutableLiveData<List<Condition>> conditionsObservable;

  protected RuleEx(int id, Network parent) {
    super(id, parent);
  }

  @Override
  protected void internalClearAddResourceConditions(Collection<ResourceCondition> resourceConditions) {
    super.internalClearAddResourceConditions(resourceConditions);

    recompileAndPostConditions();
  }

  @Override
  protected void internalClearAddScheduleConditions(Collection<ScheduleCondition> scheduleConditions) {
    super.internalClearAddScheduleConditions(scheduleConditions);

    recompileAndPostConditions();
  }

  @Override
  protected void internalSetName(String name) {
    super.internalSetName(name);

    NetworkEx parent;

    if (null != (parent = (NetworkEx) getParent())) {
      parent.onRuleNameUpdated(this);
    }

    FactoryUtility.postValue(nameObservable, name);
  }

  private void recompileAndPostConditions() {
    if (null == conditions)
      return;

    FactoryUtility.postValue(conditionsObservable, getConditions());
  }

  public Identifier getIdentifier() {
    return (null != identifier) ?  identifier : (identifier = Identifier.getIdentifier(this));
  }

  public LiveData<String> getNameObservable() {
    return (null != nameObservable) ? nameObservable : (nameObservable = new MutableLiveData<>(getName()));
  }

  public LiveData<List<Condition>> getConditionsObservable() {
    return (null != conditionsObservable) ? conditionsObservable : (conditionsObservable = FactoryUtility.createLiveData(getConditions()));
  }

  // included scheduleConditions & resourceConditions
  public List<Condition> getConditions() {
    List<ScheduleCondition> scheduleConditions;
    List<ResourceCondition> resourceConditions;

    ImmutableList.Builder<Condition> conditionsBuilder = new ImmutableList.Builder<>();

    synchronized (scheduleConditions = getScheduleConditions()) {
      conditionsBuilder.addAll(scheduleConditions);
    }

    synchronized (resourceConditions = getResourceConditions()) {
      conditionsBuilder.addAll(resourceConditions);
    }

    return (conditions = conditionsBuilder.build());
  }

  private void trimMemory() {
    conditions = null;
    nameObservable = FactoryUtility.nullIfEmpty(nameObservable);
    conditionsObservable = FactoryUtility.nullIfEmpty(conditionsObservable);
  }
}
