package buzz.getcoco.iot.android;

import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import buzz.getcoco.iot.Attribute;
import buzz.getcoco.iot.Capability;
import buzz.getcoco.iot.CapabilityWrapper;
import buzz.getcoco.iot.Resource;

public class CapabilityWrapperEx implements CapabilityWrapper {

  private final Capability realCapability;

  private MutableLiveData<String> nameObservable;
  private MutableLiveData<Resource> parentResourceObservable;
  private final List<Attribute> attributeList;
  private MutableLiveData<List<Attribute>> attributeObservable;

  protected CapabilityWrapperEx(Capability realCapability) {
    this.realCapability = realCapability;
    attributeList = new ArrayList<>(10);
  }

  @Override
  public void internalSetName(String name) {
    FactoryUtility.postValue(nameObservable, name);
  }

  @Override
  public void internalSetParent(Resource parent) {
    FactoryUtility.postValue(parentResourceObservable, parent);
  }

  @Override
  public void internalAddAttribute(Attribute attribute) {
    attributeList.add(attribute);
    postCapabilityChanged();
    FactoryUtility.postValue(attributeObservable, attributeList);
  }

  @Override
  public void internalRemoveAttribute(int attributeId) {
    synchronized (attributeList) {
      Attribute attribute;
      Iterator<Attribute> attributeIterator = attributeList.listIterator();

      while (null != (attribute = attributeIterator.next())) {
        if (attribute.getId().getInt() == attributeId) {
          attributeIterator.remove();
          return;
        }
      }
    }

    postCapabilityChanged();
    FactoryUtility.postValue(attributeObservable, attributeList);
  }

  private void postCapabilityChanged() {
    ResourceEx resource = (ResourceEx) getRealCapability().getParent();

    if (null == resource)
      return;

    resource.onCapabilitiesChanged(this);
  }

  @Override
  public void clearAndSetStandardCommandSet(int[] commandsSupported) {
    // add LiveData if needed
  }

  @Override
  public void markAsReady() {
    for (Attribute a : getRealCapability()) {
      AttributeEx aEx = (AttributeEx) a;
      aEx.syncCurrentValue();
    }
  }

  @Override
  public Capability getRealCapability() {
    return realCapability;
  }

  public LiveData<String> getNameObservable() {
    return (null != nameObservable) ? nameObservable : (nameObservable = FactoryUtility.createLiveData(getRealCapability().getName()));
  }

  public LiveData<Resource> getParentResourceObservable() {
    return (null != parentResourceObservable) ? parentResourceObservable : (parentResourceObservable = FactoryUtility.createLiveData(getRealCapability().getParent()));
  }

  public LiveData<List<Attribute>> getAttributeObservable() {
    return (null != attributeObservable) ? attributeObservable : (attributeObservable = FactoryUtility.createLiveData(attributeList));
  }
}
