/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.opensocial.shindig.oauth;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;

import com.liferay.opensocial.model.OAuthConsumer;
import com.liferay.opensocial.model.OAuthConsumerConstants;
import com.liferay.opensocial.model.impl.OAuthConsumerImpl;
import com.liferay.opensocial.util.PortletPropsValues;
import com.liferay.petra.string.CharPool;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.util.FileUtil;
import com.liferay.portal.kernel.util.PropsKeys;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.kernel.util.StringUtil;

import org.apache.shindig.gadgets.oauth.OAuthStore;

/**
 * @author Dennis Ju
 */
@Singleton
public class LiferayOAuthStoreProvider implements Provider<OAuthStore> {

	@Inject
	public LiferayOAuthStoreProvider() {
		OAuthConsumer oAuthConsumer = _getOAuthConsumer(
			PortletPropsValues.SHINDIG_OAUTH_KEY_FILE_NAME,
			PortletPropsValues.SHINDIG_OAUTH_KEY_NAME);

		_oAuthStore = new LiferayOAuthStore(oAuthConsumer);
	}

	@Override
	public OAuthStore get() {
		return _oAuthStore;
	}

	private String _convertFromOpenSsl(String key) {
		key = key.replaceAll(_OPEN_SSL_A_Z, StringPool.BLANK);
		key = StringUtil.replace(key, CharPool.NEW_LINE, StringPool.BLANK);

		return key;
	}

	private OAuthConsumer _getOAuthConsumer(
		String keyFileName, String keyName) {

		OAuthConsumer oAuthConsumer = new OAuthConsumerImpl();

		oAuthConsumer.setServiceName(_DEFAULT_SERVICE_NAME);
		oAuthConsumer.setConsumerKey(_DEFAULT_CONSUMER_KEY);

		String consumerSecret = null;

		String liferayHome = PropsUtil.get(PropsKeys.LIFERAY_HOME);

		String path = liferayHome.concat(_KEY_DIR);

		path = StringUtil.removeChar(path, CharPool.QUOTE);

		keyFileName = path.concat(keyFileName);

		try {
			consumerSecret = FileUtil.read(keyFileName);
		}
		catch (Exception exception) {
		}
		finally {
			if (consumerSecret == null) {
				if (!FileUtil.exists(path)) {
					FileUtil.mkdirs(path);
				}

				if (_log.isWarnEnabled()) {
					_log.warn("Unable to load OAuth key from " + keyFileName);
				}

				return null;
			}
		}

		consumerSecret = _convertFromOpenSsl(consumerSecret);

		oAuthConsumer.setConsumerSecret(consumerSecret);

		oAuthConsumer.setKeyType(OAuthConsumerConstants.KEY_TYPE_RSA_PRIVATE);
		oAuthConsumer.setKeyName(keyName);

		return oAuthConsumer;
	}

	private static final String _DEFAULT_CONSUMER_KEY = "DEFAULT_CONSUMER_KEY";

	private static final String _DEFAULT_SERVICE_NAME = "LIFERAY";

	private static final String _KEY_DIR = "/data/opensocial/";

	private static final String _OPEN_SSL_A_Z = "-----[A-Z ]*-----";

	private static final Log _log = LogFactoryUtil.getLog(
		LiferayOAuthStoreProvider.class);

	private final OAuthStore _oAuthStore;

}