/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.opensocial.service.impl;

import com.liferay.opensocial.model.OAuthConsumer;
import com.liferay.opensocial.model.OAuthConsumerConstants;
import com.liferay.opensocial.service.base.OAuthConsumerLocalServiceBaseImpl;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.exception.PortalException;

import java.util.Date;
import java.util.List;

/**
 * @author Dennis Ju
 */
public class OAuthConsumerLocalServiceImpl
	extends OAuthConsumerLocalServiceBaseImpl {

	@Override
	public OAuthConsumer addOAuthConsumer(
		long companyId, String gadgetKey, String serviceName,
		String consumerKey, String consumerSecret, String keyType) {

		if (keyType.equals(OAuthConsumerConstants.KEY_TYPE_RSA_PRIVATE)) {
			consumerSecret = StringPool.BLANK;
		}

		Date now = new Date();

		long oAuthConsumerId = counterLocalService.increment();

		OAuthConsumer oAuthConsumer = oAuthConsumerPersistence.create(
			oAuthConsumerId);

		oAuthConsumer.setCompanyId(companyId);
		oAuthConsumer.setCreateDate(now);
		oAuthConsumer.setModifiedDate(now);
		oAuthConsumer.setGadgetKey(gadgetKey);
		oAuthConsumer.setServiceName(serviceName);
		oAuthConsumer.setConsumerKey(consumerKey);
		oAuthConsumer.setConsumerSecret(consumerSecret);
		oAuthConsumer.setKeyType(keyType);

		return oAuthConsumerPersistence.update(oAuthConsumer);
	}

	@Override
	public OAuthConsumer deleteOAuthConsumer(long oAuthConsumerId)
		throws PortalException {

		OAuthConsumer oAuthConsumer = oAuthConsumerPersistence.findByPrimaryKey(
			oAuthConsumerId);

		return deleteOAuthConsumer(oAuthConsumer);
	}

	@Override
	public OAuthConsumer deleteOAuthConsumer(OAuthConsumer oAuthConsumer) {

		// OAuth consumer

		oAuthConsumerPersistence.remove(oAuthConsumer);

		// OAuth tokens

		oAuthTokenLocalService.deleteOAuthTokens(
			oAuthConsumer.getGadgetKey(), oAuthConsumer.getServiceName());

		return oAuthConsumer;
	}

	@Override
	public void deleteOAuthConsumers(String gadgetKey) {
		List<OAuthConsumer> oAuthConsumers =
			oAuthConsumerPersistence.findByGadgetKey(gadgetKey);

		for (OAuthConsumer oAuthConsumer : oAuthConsumers) {
			deleteOAuthConsumer(oAuthConsumer);
		}
	}

	@Override
	public OAuthConsumer fetchOAuthConsumer(
		String gadgetKey, String serviceName) {

		return oAuthConsumerPersistence.fetchByG_S(gadgetKey, serviceName);
	}

	@Override
	public OAuthConsumer getOAuthConsumer(String gadgetKey, String serviceName)
		throws PortalException {

		return oAuthConsumerPersistence.findByG_S(gadgetKey, serviceName);
	}

	@Override
	public List<OAuthConsumer> getOAuthConsumers(String gadgetKey) {
		return oAuthConsumerPersistence.findByGadgetKey(gadgetKey);
	}

	@Override
	public List<OAuthConsumer> getOAuthConsumers(
		String gadgetKey, int start, int end) {

		return oAuthConsumerPersistence.findByGadgetKey(gadgetKey, start, end);
	}

	@Override
	public int getOAuthConsumersCount(String gadgetKey) {
		return oAuthConsumerPersistence.countByGadgetKey(gadgetKey);
	}

	@Override
	public OAuthConsumer updateOAuthConsumer(
			long oAuthConsumerId, String consumerKey, String consumerSecret,
			String keyType, String keyName, String callbackURL)
		throws PortalException {

		if (keyType.equals(OAuthConsumerConstants.KEY_TYPE_RSA_PRIVATE)) {
			consumerSecret = StringPool.BLANK;
		}

		OAuthConsumer oAuthConsumer = oAuthConsumerPersistence.findByPrimaryKey(
			oAuthConsumerId);

		oAuthConsumer.setConsumerKey(consumerKey);
		oAuthConsumer.setConsumerSecret(consumerSecret);
		oAuthConsumer.setKeyType(keyType);

		return oAuthConsumerPersistence.update(oAuthConsumer);
	}

}