/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.user.associated.data.exporter;

import com.liferay.petra.string.StringBundler;
import com.liferay.petra.string.StringPool;
import com.liferay.petra.string.StringUtil;
import com.liferay.portal.kernel.dao.orm.ActionableDynamicQuery;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.BaseModel;
import com.liferay.portal.kernel.util.SystemProperties;
import com.liferay.portal.kernel.util.Time;
import com.liferay.portal.kernel.xml.Document;
import com.liferay.portal.kernel.xml.SAXReaderUtil;
import com.liferay.portal.kernel.zip.ZipWriter;
import com.liferay.portal.kernel.zip.ZipWriterFactory;
import com.liferay.user.associated.data.util.UADDynamicQueryUtil;

import java.io.File;
import java.io.UnsupportedEncodingException;

/**
 * Provides the base implementation of {@link UADExporter} for entities
 * generated with Service Builder. The count and batch actions are based on the
 * {@code ActionableDynamicQuery}, which is available in the local service
 * generated by Service Builder.
 *
 * @author William Newbury
 */
public abstract class DynamicQueryUADExporter<T extends BaseModel>
	implements UADExporter<T> {

	@Override
	public long count(long userId) throws PortalException {
		ActionableDynamicQuery actionableDynamicQuery =
			getActionableDynamicQuery(userId);

		return actionableDynamicQuery.performCount();
	}

	@Override
	public byte[] export(T baseModel) throws PortalException {
		String xml = toXmlString(baseModel);

		xml = formatXML(xml);

		try {
			return xml.getBytes(StringPool.UTF8);
		}
		catch (UnsupportedEncodingException unsupportedEncodingException) {
			throw new PortalException(unsupportedEncodingException);
		}
	}

	@Override
	public File exportAll(long userId, ZipWriterFactory zipWriterFactory)
		throws PortalException {

		ActionableDynamicQuery actionableDynamicQuery =
			getActionableDynamicQuery(userId);

		ZipWriter zipWriter = getZipWriter(
			userId, getTypeKey(), zipWriterFactory);

		actionableDynamicQuery.setPerformActionMethod(
			(T baseModel) -> {
				try {
					writeToZip(baseModel, zipWriter);
				}
				catch (Exception exception) {
					_log.error(exception);
				}
			});

		actionableDynamicQuery.performActions();

		return zipWriter.getFile();
	}

	protected File createFolder(long userId) {
		File file = new File(
			StringBundler.concat(
				SystemProperties.get(SystemProperties.TMP_DIR), "/liferay/uad/",
				userId));

		file.mkdirs();

		return file;
	}

	/**
	 * Returns an {@code ActionableDynamicQuery} for type {@code T}. This can be
	 * retrieved from the service.
	 *
	 * @return an {@code ActionableDynamicQuery} for type {@code T}
	 */
	protected abstract ActionableDynamicQuery doGetActionableDynamicQuery();

	/**
	 * Returns names identifying fields on the type {@code T} entity that
	 * contain the primary key of a user.
	 *
	 * @return the fields that may contain the primary key of a user
	 */
	protected abstract String[] doGetUserIdFieldNames();

	protected String formatXML(String xml) {
		try {
			Document document = SAXReaderUtil.read(
				_escapeCDATAClosingCharacters(xml));

			return document.formattedString();
		}
		catch (Exception exception) {
			throw new SystemException(exception);
		}
	}

	/**
	 * Returns an {@code ActionableDynamicQuery} for type {@code T}. It should
	 * be populated with criteria and ready for use by the service.
	 *
	 * @param  userId the primary key of the user used to pre-filter the {@code
	 *         ActionableDynamicQuery}
	 * @return a pre-filtered {@code ActionableDynamicQuery}
	 */
	protected ActionableDynamicQuery getActionableDynamicQuery(long userId) {
		return UADDynamicQueryUtil.addActionableDynamicQueryCriteria(
			doGetActionableDynamicQuery(), doGetUserIdFieldNames(), userId);
	}

	/**
	 * Returns a {@code ZipWriter} to write the data to. Each individual type
	 * {@code T} entity is written as a file in the resulting ZIP file.
	 *
	 * @param  userId the the primary key of the user whose data to export
	 * @param  modelClassName the string representation of the model class name
	 * @return a {@code ZipWriter} where each piece of data is written
	 */
	protected ZipWriter getZipWriter(
		long userId, String modelClassName, ZipWriterFactory zipWriterFactory) {

		File file = createFolder(userId);

		return zipWriterFactory.getZipWriter(
			new File(
				StringBundler.concat(
					file.getAbsolutePath(), StringPool.SLASH, modelClassName,
					StringPool.UNDERLINE, Time.getShortTimestamp(), ".zip")));
	}

	/**
	 * Converts the type {@code T} base model to an XML string to be written to
	 * a file.
	 *
	 * @param  baseModel the base model to be converted into an XML string
	 * @return an XML string representation of the base model
	 */
	protected abstract String toXmlString(T baseModel);

	/**
	 * Converts the type {@code T} base model to a byte array and writes it to
	 * the {@code ZipWriter}.
	 *
	 * @param  baseModel the baseModel to write to the ZIP
	 * @param  zipWriter the {@code ZipWriter} to write to
	 * @throws Exception if an exception occurred
	 */
	protected void writeToZip(T baseModel, ZipWriter zipWriter)
		throws Exception {

		byte[] data = export(baseModel);

		zipWriter.addEntry(baseModel.getPrimaryKeyObj() + ".xml", data);
	}

	private String _escapeCDATAClosingCharacters(String xml) {

		// If the closing token of a CDATA container is found inside the CDATA
		// container, split the CDATA container into two separate CDATA
		// containers. This is generally accepted method of "escaping" for this
		// case since there is no real way to escape those characters. See
		// LPS-85393 for more information.

		xml = StringUtil.replace(xml, "]]><", "[$SPECIAL_CHARACTER$]");
		xml = StringUtil.replace(xml, "]]>", "]]]]><![CDATA[>");

		return StringUtil.replace(xml, "[$SPECIAL_CHARACTER$]", "]]><");
	}

	private static final Log _log = LogFactoryUtil.getLog(
		DynamicQueryUADExporter.class);

}