/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.layout.seo.internal;

import com.liferay.asset.display.page.portlet.AssetDisplayPageFriendlyURLProvider;
import com.liferay.layout.seo.canonical.url.LayoutSEOCanonicalURLProvider;
import com.liferay.layout.seo.internal.util.FriendlyURLMapperProvider;
import com.liferay.layout.seo.kernel.LayoutSEOLink;
import com.liferay.layout.seo.kernel.LayoutSEOLinkManager;
import com.liferay.layout.seo.model.LayoutSEOEntry;
import com.liferay.layout.seo.open.graph.OpenGraphConfiguration;
import com.liferay.layout.seo.service.LayoutSEOEntryLocalService;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.language.Language;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.model.Layout;
import com.liferay.portal.kernel.service.ClassNameLocalService;
import com.liferay.portal.kernel.service.ServiceContext;
import com.liferay.portal.kernel.service.ServiceContextThreadLocal;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.Html;
import com.liferay.portal.kernel.util.ListMergeable;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.util.Portal;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Cristina González
 */
@Component(service = LayoutSEOLinkManager.class)
public class LayoutSEOLinkManagerImpl implements LayoutSEOLinkManager {

	@Override
	public LayoutSEOLink getCanonicalLayoutSEOLink(
			Layout layout, Locale locale, String canonicalURL,
			Map<Locale, String> alternateURLs)
		throws PortalException {

		return new LayoutSEOLinkImpl(
			_html.escapeAttribute(
				_layoutSEOCanonicalURLProvider.getCanonicalURL(
					layout, locale, canonicalURL, alternateURLs)),
			null, LayoutSEOLink.Relationship.CANONICAL);
	}

	@Override
	public String getFullPageTitle(
			Layout layout, String portletId, String tilesTitle,
			ListMergeable<String> titleListMergeable,
			ListMergeable<String> subtitleListMergeable, String companyName,
			Locale locale)
		throws PortalException {

		String layoutTitle = _getPageTitle(
			layout, portletId, tilesTitle, titleListMergeable,
			subtitleListMergeable, locale);
		String siteAndCompanyName = _getPageTitleSuffix(layout, companyName);

		return _merge(layoutTitle, siteAndCompanyName);
	}

	@Override
	public List<LayoutSEOLink> getLocalizedLayoutSEOLinks(
			Layout layout, Locale locale, String canonicalURL,
			Map<Locale, String> alternateURLs)
		throws PortalException {

		List<LayoutSEOLink> layoutSEOLinks = new ArrayList<>(
			alternateURLs.size() + 2);

		layoutSEOLinks.add(
			getCanonicalLayoutSEOLink(
				layout, locale, canonicalURL, alternateURLs));

		ThemeDisplay themeDisplay = _getThemeDisplay();

		FriendlyURLMapperProvider.FriendlyURLMapper friendlyURLMapper =
			_friendlyURLMapperProvider.getFriendlyURLMapper(
				themeDisplay.getRequest());

		Map<Locale, String> mappedFriendlyURLs =
			friendlyURLMapper.getMappedFriendlyURLs(alternateURLs);

		mappedFriendlyURLs.forEach(
			(urlLocale, url) -> layoutSEOLinks.add(
				new LayoutSEOLinkImpl(
					_html.escapeAttribute(
						_getAlternateCustomCanonicalURL(
							layout, urlLocale, url)),
					LocaleUtil.toW3cLanguageId(urlLocale),
					LayoutSEOLink.Relationship.ALTERNATE)));

		String defaultLocaleURL = alternateURLs.get(
			_portal.getSiteDefaultLocale(layout.getGroupId()));

		if (defaultLocaleURL == null) {
			return layoutSEOLinks;
		}

		layoutSEOLinks.add(
			new LayoutSEOLinkImpl(
				_html.escapeAttribute(defaultLocaleURL), "x-default",
				LayoutSEOLink.Relationship.ALTERNATE));

		return layoutSEOLinks;
	}

	@Override
	public String getPageTitle(
			Layout layout, String portletId, String tilesTitle,
			ListMergeable<String> titleListMergeable,
			ListMergeable<String> subtitleListMergeable, Locale locale)
		throws PortalException {

		return _html.escape(
			_getPageTitle(
				layout, portletId, tilesTitle, titleListMergeable,
				subtitleListMergeable, locale));
	}

	@Override
	public String getPageTitleSuffix(Layout layout, String companyName)
		throws PortalException {

		return _html.escape(_getPageTitleSuffix(layout, companyName));
	}

	/**
	 * @deprecated As of Athanasius (7.3.x), replaced by {@link
	 *             OpenGraphConfiguration#isOpenGraphEnabled(Group)}
	 */
	@Deprecated
	@Override
	public boolean isOpenGraphEnabled(Layout layout) throws PortalException {
		return _openGraphConfiguration.isOpenGraphEnabled(layout.getGroup());
	}

	@Activate
	protected void activate() {
		_friendlyURLMapperProvider = new FriendlyURLMapperProvider(
			_assetDisplayPageFriendlyURLProvider, _classNameLocalService);
	}

	@Deactivate
	protected void deactivate() {
		_friendlyURLMapperProvider = null;
	}

	private String _getAlternateCustomCanonicalURL(
		Layout layout, Locale locale, String alternateURL) {

		LayoutSEOEntry layoutSEOEntry =
			_layoutSEOEntryLocalService.fetchLayoutSEOEntry(
				layout.getGroupId(), layout.isPrivateLayout(),
				layout.getLayoutId());

		if ((layoutSEOEntry == null) ||
			!layoutSEOEntry.isCanonicalURLEnabled()) {

			return alternateURL;
		}

		Locale siteDefaultLocale = LocaleUtil.getSiteDefault();

		try {
			siteDefaultLocale = _portal.getSiteDefaultLocale(
				layout.getGroupId());
		}
		catch (PortalException portalException) {
			_log.error(portalException, portalException);
		}

		String currentCanonicalURL = layoutSEOEntry.getCanonicalURL(
			siteDefaultLocale);
		String alternateCanonicalURL = layoutSEOEntry.getCanonicalURL(locale);

		if (currentCanonicalURL.equals(alternateCanonicalURL)) {
			return alternateURL;
		}

		return alternateCanonicalURL;
	}

	private String _getPageTitle(
			Layout layout, String portletId, String tilesTitle,
			ListMergeable<String> titleListMergeable,
			ListMergeable<String> subtitleListMergeable, Locale locale)
		throws PortalException {

		if (Validator.isNotNull(portletId) && layout.isSystem() &&
			!layout.isTypeControlPanel() &&
			StringUtil.equals(layout.getFriendlyURL(), "/manage")) {

			return _portal.getPortletTitle(portletId, locale);
		}

		if (Validator.isNotNull(tilesTitle)) {
			return _language.get(locale, tilesTitle);
		}

		if (subtitleListMergeable == null) {
			return _getTitle(layout, titleListMergeable, locale);
		}

		return _merge(
			subtitleListMergeable.mergeToString(StringPool.SPACE),
			_getTitle(layout, titleListMergeable, locale));
	}

	private String _getPageTitleSuffix(Layout layout, String companyName)
		throws PortalException {

		Group group = layout.getGroup();

		if (group.isControlPanel() || group.isLayoutPrototype() ||
			StringUtil.equals(companyName, group.getDescriptiveName())) {

			return companyName;
		}

		return _merge(group.getDescriptiveName(), companyName);
	}

	private ThemeDisplay _getThemeDisplay() {
		ServiceContext serviceContext =
			ServiceContextThreadLocal.getServiceContext();

		if (serviceContext != null) {
			return serviceContext.getThemeDisplay();
		}

		return null;
	}

	private String _getTitle(
			Layout layout, ListMergeable<String> titleListMergeable,
			Locale locale)
		throws PortalException {

		Group group = layout.getGroup();

		if (group.isLayoutPrototype()) {
			return group.getDescriptiveName(locale);
		}

		if (titleListMergeable != null) {
			return titleListMergeable.mergeToString(StringPool.SPACE);
		}

		return layout.getHTMLTitle(_language.getLanguageId(locale));
	}

	private String _merge(String... strings) {
		return StringUtil.merge(strings, " - ");
	}

	private static final Log _log = LogFactoryUtil.getLog(
		LayoutSEOLinkManagerImpl.class);

	@Reference
	private AssetDisplayPageFriendlyURLProvider
		_assetDisplayPageFriendlyURLProvider;

	@Reference
	private ClassNameLocalService _classNameLocalService;

	private FriendlyURLMapperProvider _friendlyURLMapperProvider;

	@Reference
	private Html _html;

	@Reference
	private Language _language;

	@Reference
	private LayoutSEOCanonicalURLProvider _layoutSEOCanonicalURLProvider;

	@Reference
	private LayoutSEOEntryLocalService _layoutSEOEntryLocalService;

	@Reference
	private OpenGraphConfiguration _openGraphConfiguration;

	@Reference
	private Portal _portal;

}