/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a resource data sync configuration, including its current status and last successful sync.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceDataSyncItem implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceDataSyncItem.Builder, ResourceDataSyncItem> {
    private static final SdkField<String> SYNC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SyncName").getter(getter(ResourceDataSyncItem::syncName)).setter(setter(Builder::syncName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncName").build()).build();

    private static final SdkField<String> SYNC_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SyncType").getter(getter(ResourceDataSyncItem::syncType)).setter(setter(Builder::syncType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncType").build()).build();

    private static final SdkField<ResourceDataSyncSourceWithState> SYNC_SOURCE_FIELD = SdkField
            .<ResourceDataSyncSourceWithState> builder(MarshallingType.SDK_POJO).memberName("SyncSource")
            .getter(getter(ResourceDataSyncItem::syncSource)).setter(setter(Builder::syncSource))
            .constructor(ResourceDataSyncSourceWithState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncSource").build()).build();

    private static final SdkField<ResourceDataSyncS3Destination> S3_DESTINATION_FIELD = SdkField
            .<ResourceDataSyncS3Destination> builder(MarshallingType.SDK_POJO).memberName("S3Destination")
            .getter(getter(ResourceDataSyncItem::s3Destination)).setter(setter(Builder::s3Destination))
            .constructor(ResourceDataSyncS3Destination::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Destination").build()).build();

    private static final SdkField<Instant> LAST_SYNC_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastSyncTime").getter(getter(ResourceDataSyncItem::lastSyncTime)).setter(setter(Builder::lastSyncTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSyncTime").build()).build();

    private static final SdkField<Instant> LAST_SUCCESSFUL_SYNC_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastSuccessfulSyncTime").getter(getter(ResourceDataSyncItem::lastSuccessfulSyncTime))
            .setter(setter(Builder::lastSuccessfulSyncTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSuccessfulSyncTime").build())
            .build();

    private static final SdkField<Instant> SYNC_LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SyncLastModifiedTime").getter(getter(ResourceDataSyncItem::syncLastModifiedTime))
            .setter(setter(Builder::syncLastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncLastModifiedTime").build())
            .build();

    private static final SdkField<String> LAST_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastStatus").getter(getter(ResourceDataSyncItem::lastStatusAsString))
            .setter(setter(Builder::lastStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastStatus").build()).build();

    private static final SdkField<Instant> SYNC_CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SyncCreatedTime").getter(getter(ResourceDataSyncItem::syncCreatedTime))
            .setter(setter(Builder::syncCreatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncCreatedTime").build()).build();

    private static final SdkField<String> LAST_SYNC_STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastSyncStatusMessage").getter(getter(ResourceDataSyncItem::lastSyncStatusMessage))
            .setter(setter(Builder::lastSyncStatusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastSyncStatusMessage").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SYNC_NAME_FIELD,
            SYNC_TYPE_FIELD, SYNC_SOURCE_FIELD, S3_DESTINATION_FIELD, LAST_SYNC_TIME_FIELD, LAST_SUCCESSFUL_SYNC_TIME_FIELD,
            SYNC_LAST_MODIFIED_TIME_FIELD, LAST_STATUS_FIELD, SYNC_CREATED_TIME_FIELD, LAST_SYNC_STATUS_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String syncName;

    private final String syncType;

    private final ResourceDataSyncSourceWithState syncSource;

    private final ResourceDataSyncS3Destination s3Destination;

    private final Instant lastSyncTime;

    private final Instant lastSuccessfulSyncTime;

    private final Instant syncLastModifiedTime;

    private final String lastStatus;

    private final Instant syncCreatedTime;

    private final String lastSyncStatusMessage;

    private ResourceDataSyncItem(BuilderImpl builder) {
        this.syncName = builder.syncName;
        this.syncType = builder.syncType;
        this.syncSource = builder.syncSource;
        this.s3Destination = builder.s3Destination;
        this.lastSyncTime = builder.lastSyncTime;
        this.lastSuccessfulSyncTime = builder.lastSuccessfulSyncTime;
        this.syncLastModifiedTime = builder.syncLastModifiedTime;
        this.lastStatus = builder.lastStatus;
        this.syncCreatedTime = builder.syncCreatedTime;
        this.lastSyncStatusMessage = builder.lastSyncStatusMessage;
    }

    /**
     * <p>
     * The name of the resource data sync.
     * </p>
     * 
     * @return The name of the resource data sync.
     */
    public final String syncName() {
        return syncName;
    }

    /**
     * <p>
     * The type of resource data sync. If <code>SyncType</code> is <code>SyncToDestination</code>, then the resource
     * data sync synchronizes data to an S3 bucket. If the <code>SyncType</code> is <code>SyncFromSource</code> then the
     * resource data sync synchronizes data from Organizations or from multiple Amazon Web Services Regions.
     * </p>
     * 
     * @return The type of resource data sync. If <code>SyncType</code> is <code>SyncToDestination</code>, then the
     *         resource data sync synchronizes data to an S3 bucket. If the <code>SyncType</code> is
     *         <code>SyncFromSource</code> then the resource data sync synchronizes data from Organizations or from
     *         multiple Amazon Web Services Regions.
     */
    public final String syncType() {
        return syncType;
    }

    /**
     * <p>
     * Information about the source where the data was synchronized.
     * </p>
     * 
     * @return Information about the source where the data was synchronized.
     */
    public final ResourceDataSyncSourceWithState syncSource() {
        return syncSource;
    }

    /**
     * <p>
     * Configuration information for the target S3 bucket.
     * </p>
     * 
     * @return Configuration information for the target S3 bucket.
     */
    public final ResourceDataSyncS3Destination s3Destination() {
        return s3Destination;
    }

    /**
     * <p>
     * The last time the configuration attempted to sync (UTC).
     * </p>
     * 
     * @return The last time the configuration attempted to sync (UTC).
     */
    public final Instant lastSyncTime() {
        return lastSyncTime;
    }

    /**
     * <p>
     * The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
     * </p>
     * 
     * @return The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
     */
    public final Instant lastSuccessfulSyncTime() {
        return lastSuccessfulSyncTime;
    }

    /**
     * <p>
     * The date and time the resource data sync was changed.
     * </p>
     * 
     * @return The date and time the resource data sync was changed.
     */
    public final Instant syncLastModifiedTime() {
        return syncLastModifiedTime;
    }

    /**
     * <p>
     * The status reported by the last sync.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lastStatus} will
     * return {@link LastResourceDataSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lastStatusAsString}.
     * </p>
     * 
     * @return The status reported by the last sync.
     * @see LastResourceDataSyncStatus
     */
    public final LastResourceDataSyncStatus lastStatus() {
        return LastResourceDataSyncStatus.fromValue(lastStatus);
    }

    /**
     * <p>
     * The status reported by the last sync.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lastStatus} will
     * return {@link LastResourceDataSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lastStatusAsString}.
     * </p>
     * 
     * @return The status reported by the last sync.
     * @see LastResourceDataSyncStatus
     */
    public final String lastStatusAsString() {
        return lastStatus;
    }

    /**
     * <p>
     * The date and time the configuration was created (UTC).
     * </p>
     * 
     * @return The date and time the configuration was created (UTC).
     */
    public final Instant syncCreatedTime() {
        return syncCreatedTime;
    }

    /**
     * <p>
     * The status message details reported by the last sync.
     * </p>
     * 
     * @return The status message details reported by the last sync.
     */
    public final String lastSyncStatusMessage() {
        return lastSyncStatusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(syncName());
        hashCode = 31 * hashCode + Objects.hashCode(syncType());
        hashCode = 31 * hashCode + Objects.hashCode(syncSource());
        hashCode = 31 * hashCode + Objects.hashCode(s3Destination());
        hashCode = 31 * hashCode + Objects.hashCode(lastSyncTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfulSyncTime());
        hashCode = 31 * hashCode + Objects.hashCode(syncLastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(syncCreatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastSyncStatusMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDataSyncItem)) {
            return false;
        }
        ResourceDataSyncItem other = (ResourceDataSyncItem) obj;
        return Objects.equals(syncName(), other.syncName()) && Objects.equals(syncType(), other.syncType())
                && Objects.equals(syncSource(), other.syncSource()) && Objects.equals(s3Destination(), other.s3Destination())
                && Objects.equals(lastSyncTime(), other.lastSyncTime())
                && Objects.equals(lastSuccessfulSyncTime(), other.lastSuccessfulSyncTime())
                && Objects.equals(syncLastModifiedTime(), other.syncLastModifiedTime())
                && Objects.equals(lastStatusAsString(), other.lastStatusAsString())
                && Objects.equals(syncCreatedTime(), other.syncCreatedTime())
                && Objects.equals(lastSyncStatusMessage(), other.lastSyncStatusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceDataSyncItem").add("SyncName", syncName()).add("SyncType", syncType())
                .add("SyncSource", syncSource()).add("S3Destination", s3Destination()).add("LastSyncTime", lastSyncTime())
                .add("LastSuccessfulSyncTime", lastSuccessfulSyncTime()).add("SyncLastModifiedTime", syncLastModifiedTime())
                .add("LastStatus", lastStatusAsString()).add("SyncCreatedTime", syncCreatedTime())
                .add("LastSyncStatusMessage", lastSyncStatusMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SyncName":
            return Optional.ofNullable(clazz.cast(syncName()));
        case "SyncType":
            return Optional.ofNullable(clazz.cast(syncType()));
        case "SyncSource":
            return Optional.ofNullable(clazz.cast(syncSource()));
        case "S3Destination":
            return Optional.ofNullable(clazz.cast(s3Destination()));
        case "LastSyncTime":
            return Optional.ofNullable(clazz.cast(lastSyncTime()));
        case "LastSuccessfulSyncTime":
            return Optional.ofNullable(clazz.cast(lastSuccessfulSyncTime()));
        case "SyncLastModifiedTime":
            return Optional.ofNullable(clazz.cast(syncLastModifiedTime()));
        case "LastStatus":
            return Optional.ofNullable(clazz.cast(lastStatusAsString()));
        case "SyncCreatedTime":
            return Optional.ofNullable(clazz.cast(syncCreatedTime()));
        case "LastSyncStatusMessage":
            return Optional.ofNullable(clazz.cast(lastSyncStatusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SyncName", SYNC_NAME_FIELD);
        map.put("SyncType", SYNC_TYPE_FIELD);
        map.put("SyncSource", SYNC_SOURCE_FIELD);
        map.put("S3Destination", S3_DESTINATION_FIELD);
        map.put("LastSyncTime", LAST_SYNC_TIME_FIELD);
        map.put("LastSuccessfulSyncTime", LAST_SUCCESSFUL_SYNC_TIME_FIELD);
        map.put("SyncLastModifiedTime", SYNC_LAST_MODIFIED_TIME_FIELD);
        map.put("LastStatus", LAST_STATUS_FIELD);
        map.put("SyncCreatedTime", SYNC_CREATED_TIME_FIELD);
        map.put("LastSyncStatusMessage", LAST_SYNC_STATUS_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceDataSyncItem, T> g) {
        return obj -> g.apply((ResourceDataSyncItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceDataSyncItem> {
        /**
         * <p>
         * The name of the resource data sync.
         * </p>
         * 
         * @param syncName
         *        The name of the resource data sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncName(String syncName);

        /**
         * <p>
         * The type of resource data sync. If <code>SyncType</code> is <code>SyncToDestination</code>, then the resource
         * data sync synchronizes data to an S3 bucket. If the <code>SyncType</code> is <code>SyncFromSource</code> then
         * the resource data sync synchronizes data from Organizations or from multiple Amazon Web Services Regions.
         * </p>
         * 
         * @param syncType
         *        The type of resource data sync. If <code>SyncType</code> is <code>SyncToDestination</code>, then the
         *        resource data sync synchronizes data to an S3 bucket. If the <code>SyncType</code> is
         *        <code>SyncFromSource</code> then the resource data sync synchronizes data from Organizations or from
         *        multiple Amazon Web Services Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncType(String syncType);

        /**
         * <p>
         * Information about the source where the data was synchronized.
         * </p>
         * 
         * @param syncSource
         *        Information about the source where the data was synchronized.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncSource(ResourceDataSyncSourceWithState syncSource);

        /**
         * <p>
         * Information about the source where the data was synchronized.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceDataSyncSourceWithState.Builder}
         * avoiding the need to create one manually via {@link ResourceDataSyncSourceWithState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceDataSyncSourceWithState.Builder#build()} is called
         * immediately and its result is passed to {@link #syncSource(ResourceDataSyncSourceWithState)}.
         * 
         * @param syncSource
         *        a consumer that will call methods on {@link ResourceDataSyncSourceWithState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #syncSource(ResourceDataSyncSourceWithState)
         */
        default Builder syncSource(Consumer<ResourceDataSyncSourceWithState.Builder> syncSource) {
            return syncSource(ResourceDataSyncSourceWithState.builder().applyMutation(syncSource).build());
        }

        /**
         * <p>
         * Configuration information for the target S3 bucket.
         * </p>
         * 
         * @param s3Destination
         *        Configuration information for the target S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Destination(ResourceDataSyncS3Destination s3Destination);

        /**
         * <p>
         * Configuration information for the target S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceDataSyncS3Destination.Builder}
         * avoiding the need to create one manually via {@link ResourceDataSyncS3Destination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceDataSyncS3Destination.Builder#build()} is called
         * immediately and its result is passed to {@link #s3Destination(ResourceDataSyncS3Destination)}.
         * 
         * @param s3Destination
         *        a consumer that will call methods on {@link ResourceDataSyncS3Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Destination(ResourceDataSyncS3Destination)
         */
        default Builder s3Destination(Consumer<ResourceDataSyncS3Destination.Builder> s3Destination) {
            return s3Destination(ResourceDataSyncS3Destination.builder().applyMutation(s3Destination).build());
        }

        /**
         * <p>
         * The last time the configuration attempted to sync (UTC).
         * </p>
         * 
         * @param lastSyncTime
         *        The last time the configuration attempted to sync (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSyncTime(Instant lastSyncTime);

        /**
         * <p>
         * The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
         * </p>
         * 
         * @param lastSuccessfulSyncTime
         *        The last time the sync operations returned a status of <code>SUCCESSFUL</code> (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfulSyncTime(Instant lastSuccessfulSyncTime);

        /**
         * <p>
         * The date and time the resource data sync was changed.
         * </p>
         * 
         * @param syncLastModifiedTime
         *        The date and time the resource data sync was changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncLastModifiedTime(Instant syncLastModifiedTime);

        /**
         * <p>
         * The status reported by the last sync.
         * </p>
         * 
         * @param lastStatus
         *        The status reported by the last sync.
         * @see LastResourceDataSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LastResourceDataSyncStatus
         */
        Builder lastStatus(String lastStatus);

        /**
         * <p>
         * The status reported by the last sync.
         * </p>
         * 
         * @param lastStatus
         *        The status reported by the last sync.
         * @see LastResourceDataSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LastResourceDataSyncStatus
         */
        Builder lastStatus(LastResourceDataSyncStatus lastStatus);

        /**
         * <p>
         * The date and time the configuration was created (UTC).
         * </p>
         * 
         * @param syncCreatedTime
         *        The date and time the configuration was created (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncCreatedTime(Instant syncCreatedTime);

        /**
         * <p>
         * The status message details reported by the last sync.
         * </p>
         * 
         * @param lastSyncStatusMessage
         *        The status message details reported by the last sync.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSyncStatusMessage(String lastSyncStatusMessage);
    }

    static final class BuilderImpl implements Builder {
        private String syncName;

        private String syncType;

        private ResourceDataSyncSourceWithState syncSource;

        private ResourceDataSyncS3Destination s3Destination;

        private Instant lastSyncTime;

        private Instant lastSuccessfulSyncTime;

        private Instant syncLastModifiedTime;

        private String lastStatus;

        private Instant syncCreatedTime;

        private String lastSyncStatusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDataSyncItem model) {
            syncName(model.syncName);
            syncType(model.syncType);
            syncSource(model.syncSource);
            s3Destination(model.s3Destination);
            lastSyncTime(model.lastSyncTime);
            lastSuccessfulSyncTime(model.lastSuccessfulSyncTime);
            syncLastModifiedTime(model.syncLastModifiedTime);
            lastStatus(model.lastStatus);
            syncCreatedTime(model.syncCreatedTime);
            lastSyncStatusMessage(model.lastSyncStatusMessage);
        }

        public final String getSyncName() {
            return syncName;
        }

        public final void setSyncName(String syncName) {
            this.syncName = syncName;
        }

        @Override
        public final Builder syncName(String syncName) {
            this.syncName = syncName;
            return this;
        }

        public final String getSyncType() {
            return syncType;
        }

        public final void setSyncType(String syncType) {
            this.syncType = syncType;
        }

        @Override
        public final Builder syncType(String syncType) {
            this.syncType = syncType;
            return this;
        }

        public final ResourceDataSyncSourceWithState.Builder getSyncSource() {
            return syncSource != null ? syncSource.toBuilder() : null;
        }

        public final void setSyncSource(ResourceDataSyncSourceWithState.BuilderImpl syncSource) {
            this.syncSource = syncSource != null ? syncSource.build() : null;
        }

        @Override
        public final Builder syncSource(ResourceDataSyncSourceWithState syncSource) {
            this.syncSource = syncSource;
            return this;
        }

        public final ResourceDataSyncS3Destination.Builder getS3Destination() {
            return s3Destination != null ? s3Destination.toBuilder() : null;
        }

        public final void setS3Destination(ResourceDataSyncS3Destination.BuilderImpl s3Destination) {
            this.s3Destination = s3Destination != null ? s3Destination.build() : null;
        }

        @Override
        public final Builder s3Destination(ResourceDataSyncS3Destination s3Destination) {
            this.s3Destination = s3Destination;
            return this;
        }

        public final Instant getLastSyncTime() {
            return lastSyncTime;
        }

        public final void setLastSyncTime(Instant lastSyncTime) {
            this.lastSyncTime = lastSyncTime;
        }

        @Override
        public final Builder lastSyncTime(Instant lastSyncTime) {
            this.lastSyncTime = lastSyncTime;
            return this;
        }

        public final Instant getLastSuccessfulSyncTime() {
            return lastSuccessfulSyncTime;
        }

        public final void setLastSuccessfulSyncTime(Instant lastSuccessfulSyncTime) {
            this.lastSuccessfulSyncTime = lastSuccessfulSyncTime;
        }

        @Override
        public final Builder lastSuccessfulSyncTime(Instant lastSuccessfulSyncTime) {
            this.lastSuccessfulSyncTime = lastSuccessfulSyncTime;
            return this;
        }

        public final Instant getSyncLastModifiedTime() {
            return syncLastModifiedTime;
        }

        public final void setSyncLastModifiedTime(Instant syncLastModifiedTime) {
            this.syncLastModifiedTime = syncLastModifiedTime;
        }

        @Override
        public final Builder syncLastModifiedTime(Instant syncLastModifiedTime) {
            this.syncLastModifiedTime = syncLastModifiedTime;
            return this;
        }

        public final String getLastStatus() {
            return lastStatus;
        }

        public final void setLastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
        }

        @Override
        public final Builder lastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
            return this;
        }

        @Override
        public final Builder lastStatus(LastResourceDataSyncStatus lastStatus) {
            this.lastStatus(lastStatus == null ? null : lastStatus.toString());
            return this;
        }

        public final Instant getSyncCreatedTime() {
            return syncCreatedTime;
        }

        public final void setSyncCreatedTime(Instant syncCreatedTime) {
            this.syncCreatedTime = syncCreatedTime;
        }

        @Override
        public final Builder syncCreatedTime(Instant syncCreatedTime) {
            this.syncCreatedTime = syncCreatedTime;
            return this;
        }

        public final String getLastSyncStatusMessage() {
            return lastSyncStatusMessage;
        }

        public final void setLastSyncStatusMessage(String lastSyncStatusMessage) {
            this.lastSyncStatusMessage = lastSyncStatusMessage;
        }

        @Override
        public final Builder lastSyncStatusMessage(String lastSyncStatusMessage) {
            this.lastSyncStatusMessage = lastSyncStatusMessage;
            return this;
        }

        @Override
        public ResourceDataSyncItem build() {
            return new ResourceDataSyncItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
