/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the high-level patch compliance state for a managed node, providing information about the number of
 * installed, missing, not applicable, and failed patches along with metadata about the operation when this information
 * was gathered for the managed node.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstancePatchState implements SdkPojo, Serializable,
        ToCopyableBuilder<InstancePatchState.Builder, InstancePatchState> {
    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(InstancePatchState::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> PATCH_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PatchGroup").getter(getter(InstancePatchState::patchGroup)).setter(setter(Builder::patchGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PatchGroup").build()).build();

    private static final SdkField<String> BASELINE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaselineId").getter(getter(InstancePatchState::baselineId)).setter(setter(Builder::baselineId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaselineId").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotId").getter(getter(InstancePatchState::snapshotId)).setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId").build()).build();

    private static final SdkField<String> INSTALL_OVERRIDE_LIST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstallOverrideList").getter(getter(InstancePatchState::installOverrideList))
            .setter(setter(Builder::installOverrideList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstallOverrideList").build())
            .build();

    private static final SdkField<String> OWNER_INFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerInformation").getter(getter(InstancePatchState::ownerInformation))
            .setter(setter(Builder::ownerInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerInformation").build()).build();

    private static final SdkField<Integer> INSTALLED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstalledCount").getter(getter(InstancePatchState::installedCount))
            .setter(setter(Builder::installedCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledCount").build()).build();

    private static final SdkField<Integer> INSTALLED_OTHER_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstalledOtherCount").getter(getter(InstancePatchState::installedOtherCount))
            .setter(setter(Builder::installedOtherCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledOtherCount").build())
            .build();

    private static final SdkField<Integer> INSTALLED_PENDING_REBOOT_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstalledPendingRebootCount")
            .getter(getter(InstancePatchState::installedPendingRebootCount))
            .setter(setter(Builder::installedPendingRebootCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledPendingRebootCount")
                    .build()).build();

    private static final SdkField<Integer> INSTALLED_REJECTED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstalledRejectedCount").getter(getter(InstancePatchState::installedRejectedCount))
            .setter(setter(Builder::installedRejectedCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstalledRejectedCount").build())
            .build();

    private static final SdkField<Integer> MISSING_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MissingCount").getter(getter(InstancePatchState::missingCount)).setter(setter(Builder::missingCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MissingCount").build()).build();

    private static final SdkField<Integer> FAILED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("FailedCount").getter(getter(InstancePatchState::failedCount)).setter(setter(Builder::failedCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailedCount").build()).build();

    private static final SdkField<Integer> UNREPORTED_NOT_APPLICABLE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("UnreportedNotApplicableCount")
            .getter(getter(InstancePatchState::unreportedNotApplicableCount))
            .setter(setter(Builder::unreportedNotApplicableCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnreportedNotApplicableCount")
                    .build()).build();

    private static final SdkField<Integer> NOT_APPLICABLE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NotApplicableCount").getter(getter(InstancePatchState::notApplicableCount))
            .setter(setter(Builder::notApplicableCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotApplicableCount").build())
            .build();

    private static final SdkField<Integer> AVAILABLE_SECURITY_UPDATE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AvailableSecurityUpdateCount")
            .getter(getter(InstancePatchState::availableSecurityUpdateCount))
            .setter(setter(Builder::availableSecurityUpdateCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailableSecurityUpdateCount")
                    .build()).build();

    private static final SdkField<Instant> OPERATION_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("OperationStartTime").getter(getter(InstancePatchState::operationStartTime))
            .setter(setter(Builder::operationStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationStartTime").build())
            .build();

    private static final SdkField<Instant> OPERATION_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("OperationEndTime").getter(getter(InstancePatchState::operationEndTime))
            .setter(setter(Builder::operationEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationEndTime").build()).build();

    private static final SdkField<String> OPERATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Operation").getter(getter(InstancePatchState::operationAsString)).setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation").build()).build();

    private static final SdkField<Instant> LAST_NO_REBOOT_INSTALL_OPERATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastNoRebootInstallOperationTime")
            .getter(getter(InstancePatchState::lastNoRebootInstallOperationTime))
            .setter(setter(Builder::lastNoRebootInstallOperationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastNoRebootInstallOperationTime")
                    .build()).build();

    private static final SdkField<String> REBOOT_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RebootOption").getter(getter(InstancePatchState::rebootOptionAsString))
            .setter(setter(Builder::rebootOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RebootOption").build()).build();

    private static final SdkField<Integer> CRITICAL_NON_COMPLIANT_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("CriticalNonCompliantCount")
            .getter(getter(InstancePatchState::criticalNonCompliantCount)).setter(setter(Builder::criticalNonCompliantCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CriticalNonCompliantCount").build())
            .build();

    private static final SdkField<Integer> SECURITY_NON_COMPLIANT_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("SecurityNonCompliantCount")
            .getter(getter(InstancePatchState::securityNonCompliantCount)).setter(setter(Builder::securityNonCompliantCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityNonCompliantCount").build())
            .build();

    private static final SdkField<Integer> OTHER_NON_COMPLIANT_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("OtherNonCompliantCount").getter(getter(InstancePatchState::otherNonCompliantCount))
            .setter(setter(Builder::otherNonCompliantCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OtherNonCompliantCount").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ID_FIELD,
            PATCH_GROUP_FIELD, BASELINE_ID_FIELD, SNAPSHOT_ID_FIELD, INSTALL_OVERRIDE_LIST_FIELD, OWNER_INFORMATION_FIELD,
            INSTALLED_COUNT_FIELD, INSTALLED_OTHER_COUNT_FIELD, INSTALLED_PENDING_REBOOT_COUNT_FIELD,
            INSTALLED_REJECTED_COUNT_FIELD, MISSING_COUNT_FIELD, FAILED_COUNT_FIELD, UNREPORTED_NOT_APPLICABLE_COUNT_FIELD,
            NOT_APPLICABLE_COUNT_FIELD, AVAILABLE_SECURITY_UPDATE_COUNT_FIELD, OPERATION_START_TIME_FIELD,
            OPERATION_END_TIME_FIELD, OPERATION_FIELD, LAST_NO_REBOOT_INSTALL_OPERATION_TIME_FIELD, REBOOT_OPTION_FIELD,
            CRITICAL_NON_COMPLIANT_COUNT_FIELD, SECURITY_NON_COMPLIANT_COUNT_FIELD, OTHER_NON_COMPLIANT_COUNT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceId;

    private final String patchGroup;

    private final String baselineId;

    private final String snapshotId;

    private final String installOverrideList;

    private final String ownerInformation;

    private final Integer installedCount;

    private final Integer installedOtherCount;

    private final Integer installedPendingRebootCount;

    private final Integer installedRejectedCount;

    private final Integer missingCount;

    private final Integer failedCount;

    private final Integer unreportedNotApplicableCount;

    private final Integer notApplicableCount;

    private final Integer availableSecurityUpdateCount;

    private final Instant operationStartTime;

    private final Instant operationEndTime;

    private final String operation;

    private final Instant lastNoRebootInstallOperationTime;

    private final String rebootOption;

    private final Integer criticalNonCompliantCount;

    private final Integer securityNonCompliantCount;

    private final Integer otherNonCompliantCount;

    private InstancePatchState(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.patchGroup = builder.patchGroup;
        this.baselineId = builder.baselineId;
        this.snapshotId = builder.snapshotId;
        this.installOverrideList = builder.installOverrideList;
        this.ownerInformation = builder.ownerInformation;
        this.installedCount = builder.installedCount;
        this.installedOtherCount = builder.installedOtherCount;
        this.installedPendingRebootCount = builder.installedPendingRebootCount;
        this.installedRejectedCount = builder.installedRejectedCount;
        this.missingCount = builder.missingCount;
        this.failedCount = builder.failedCount;
        this.unreportedNotApplicableCount = builder.unreportedNotApplicableCount;
        this.notApplicableCount = builder.notApplicableCount;
        this.availableSecurityUpdateCount = builder.availableSecurityUpdateCount;
        this.operationStartTime = builder.operationStartTime;
        this.operationEndTime = builder.operationEndTime;
        this.operation = builder.operation;
        this.lastNoRebootInstallOperationTime = builder.lastNoRebootInstallOperationTime;
        this.rebootOption = builder.rebootOption;
        this.criticalNonCompliantCount = builder.criticalNonCompliantCount;
        this.securityNonCompliantCount = builder.securityNonCompliantCount;
        this.otherNonCompliantCount = builder.otherNonCompliantCount;
    }

    /**
     * <p>
     * The ID of the managed node the high-level patch compliance information was collected for.
     * </p>
     * 
     * @return The ID of the managed node the high-level patch compliance information was collected for.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The name of the patch group the managed node belongs to.
     * </p>
     * 
     * @return The name of the patch group the managed node belongs to.
     */
    public final String patchGroup() {
        return patchGroup;
    }

    /**
     * <p>
     * The ID of the patch baseline used to patch the managed node.
     * </p>
     * 
     * @return The ID of the patch baseline used to patch the managed node.
     */
    public final String baselineId() {
        return baselineId;
    }

    /**
     * <p>
     * The ID of the patch baseline snapshot used during the patching operation when this compliance data was collected.
     * </p>
     * 
     * @return The ID of the patch baseline snapshot used during the patching operation when this compliance data was
     *         collected.
     */
    public final String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * An https URL or an Amazon Simple Storage Service (Amazon S3) path-style URL to a list of patches to be installed.
     * This patch installation list, which you maintain in an S3 bucket in YAML format and specify in the SSM document
     * <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch baseline.
     * </p>
     * <p>
     * For more information about the <code>InstallOverrideList</code> parameter, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html">SSM
     * Command document for patching: <code>AWS-RunPatchBaseline</code> </a> in the <i>Amazon Web Services Systems
     * Manager User Guide</i>.
     * </p>
     * 
     * @return An https URL or an Amazon Simple Storage Service (Amazon S3) path-style URL to a list of patches to be
     *         installed. This patch installation list, which you maintain in an S3 bucket in YAML format and specify in
     *         the SSM document <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch
     *         baseline.</p>
     *         <p>
     *         For more information about the <code>InstallOverrideList</code> parameter, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html"
     *         >SSM Command document for patching: <code>AWS-RunPatchBaseline</code> </a> in the <i>Amazon Web Services
     *         Systems Manager User Guide</i>.
     */
    public final String installOverrideList() {
        return installOverrideList;
    }

    /**
     * <p>
     * Placeholder information. This field will always be empty in the current release of the service.
     * </p>
     * 
     * @return Placeholder information. This field will always be empty in the current release of the service.
     */
    public final String ownerInformation() {
        return ownerInformation;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that are installed on the managed node.
     * </p>
     * 
     * @return The number of patches from the patch baseline that are installed on the managed node.
     */
    public final Integer installedCount() {
        return installedCount;
    }

    /**
     * <p>
     * The number of patches not specified in the patch baseline that are installed on the managed node.
     * </p>
     * 
     * @return The number of patches not specified in the patch baseline that are installed on the managed node.
     */
    public final Integer installedOtherCount() {
        return installedOtherCount;
    }

    /**
     * <p>
     * The number of patches installed by Patch Manager since the last time the managed node was rebooted.
     * </p>
     * 
     * @return The number of patches installed by Patch Manager since the last time the managed node was rebooted.
     */
    public final Integer installedPendingRebootCount() {
        return installedPendingRebootCount;
    }

    /**
     * <p>
     * The number of patches installed on a managed node that are specified in a <code>RejectedPatches</code> list.
     * Patches with a status of <code>InstalledRejected</code> were typically installed before they were added to a
     * <code>RejectedPatches</code> list.
     * </p>
     * <note>
     * <p>
     * If <code>ALLOW_AS_DEPENDENCY</code> is the specified option for <code>RejectedPatchesAction</code>, the value of
     * <code>InstalledRejectedCount</code> will always be <code>0</code> (zero).
     * </p>
     * </note>
     * 
     * @return The number of patches installed on a managed node that are specified in a <code>RejectedPatches</code>
     *         list. Patches with a status of <code>InstalledRejected</code> were typically installed before they were
     *         added to a <code>RejectedPatches</code> list.</p> <note>
     *         <p>
     *         If <code>ALLOW_AS_DEPENDENCY</code> is the specified option for <code>RejectedPatchesAction</code>, the
     *         value of <code>InstalledRejectedCount</code> will always be <code>0</code> (zero).
     *         </p>
     */
    public final Integer installedRejectedCount() {
        return installedRejectedCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that are applicable for the managed node but aren't currently
     * installed.
     * </p>
     * 
     * @return The number of patches from the patch baseline that are applicable for the managed node but aren't
     *         currently installed.
     */
    public final Integer missingCount() {
        return missingCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that were attempted to be installed during the last patching
     * operation, but failed to install.
     * </p>
     * 
     * @return The number of patches from the patch baseline that were attempted to be installed during the last
     *         patching operation, but failed to install.
     */
    public final Integer failedCount() {
        return failedCount;
    }

    /**
     * <p>
     * The number of patches beyond the supported limit of <code>NotApplicableCount</code> that aren't reported by name
     * to Inventory. Inventory is a tool in Amazon Web Services Systems Manager.
     * </p>
     * 
     * @return The number of patches beyond the supported limit of <code>NotApplicableCount</code> that aren't reported
     *         by name to Inventory. Inventory is a tool in Amazon Web Services Systems Manager.
     */
    public final Integer unreportedNotApplicableCount() {
        return unreportedNotApplicableCount;
    }

    /**
     * <p>
     * The number of patches from the patch baseline that aren't applicable for the managed node and therefore aren't
     * installed on the node. This number may be truncated if the list of patch names is very large. The number of
     * patches beyond this limit are reported in <code>UnreportedNotApplicableCount</code>.
     * </p>
     * 
     * @return The number of patches from the patch baseline that aren't applicable for the managed node and therefore
     *         aren't installed on the node. This number may be truncated if the list of patch names is very large. The
     *         number of patches beyond this limit are reported in <code>UnreportedNotApplicableCount</code>.
     */
    public final Integer notApplicableCount() {
        return notApplicableCount;
    }

    /**
     * <p>
     * The number of security-related patches that are available but not approved because they didn't meet the patch
     * baseline requirements. For example, an updated version of a patch might have been released before the specified
     * auto-approval period was over.
     * </p>
     * <p>
     * Applies to Windows Server managed nodes only.
     * </p>
     * 
     * @return The number of security-related patches that are available but not approved because they didn't meet the
     *         patch baseline requirements. For example, an updated version of a patch might have been released before
     *         the specified auto-approval period was over.</p>
     *         <p>
     *         Applies to Windows Server managed nodes only.
     */
    public final Integer availableSecurityUpdateCount() {
        return availableSecurityUpdateCount;
    }

    /**
     * <p>
     * The time the most recent patching operation was started on the managed node.
     * </p>
     * 
     * @return The time the most recent patching operation was started on the managed node.
     */
    public final Instant operationStartTime() {
        return operationStartTime;
    }

    /**
     * <p>
     * The time the most recent patching operation completed on the managed node.
     * </p>
     * 
     * @return The time the most recent patching operation completed on the managed node.
     */
    public final Instant operationEndTime() {
        return operationEndTime;
    }

    /**
     * <p>
     * The type of patching operation that was performed: or
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SCAN</code> assesses the patch compliance state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSTALL</code> installs missing patches.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link PatchOperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The type of patching operation that was performed: or </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SCAN</code> assesses the patch compliance state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSTALL</code> installs missing patches.
     *         </p>
     *         </li>
     * @see PatchOperationType
     */
    public final PatchOperationType operation() {
        return PatchOperationType.fromValue(operation);
    }

    /**
     * <p>
     * The type of patching operation that was performed: or
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SCAN</code> assesses the patch compliance state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSTALL</code> installs missing patches.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link PatchOperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The type of patching operation that was performed: or </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SCAN</code> assesses the patch compliance state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSTALL</code> installs missing patches.
     *         </p>
     *         </li>
     * @see PatchOperationType
     */
    public final String operationAsString() {
        return operation;
    }

    /**
     * <p>
     * The time of the last attempt to patch the managed node with <code>NoReboot</code> specified as the reboot option.
     * </p>
     * 
     * @return The time of the last attempt to patch the managed node with <code>NoReboot</code> specified as the reboot
     *         option.
     */
    public final Instant lastNoRebootInstallOperationTime() {
        return lastNoRebootInstallOperationTime;
    }

    /**
     * <p>
     * Indicates the reboot option specified in the patch baseline.
     * </p>
     * <note>
     * <p>
     * Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch Manager
     * <code>Scan</code> operations.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any patches, or if
     * any patches are detected with a status of <code>InstalledPendingReboot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the system.
     * Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These patches
     * might not be in effect until a reboot is performed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rebootOption} will
     * return {@link RebootOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rebootOptionAsString}.
     * </p>
     * 
     * @return Indicates the reboot option specified in the patch baseline.</p> <note>
     *         <p>
     *         Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch Manager
     *         <code>Scan</code> operations.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any patches,
     *         or if any patches are detected with a status of <code>InstalledPendingReboot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the
     *         system. Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>.
     *         These patches might not be in effect until a reboot is performed.
     *         </p>
     *         </li>
     * @see RebootOption
     */
    public final RebootOption rebootOption() {
        return RebootOption.fromValue(rebootOption);
    }

    /**
     * <p>
     * Indicates the reboot option specified in the patch baseline.
     * </p>
     * <note>
     * <p>
     * Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch Manager
     * <code>Scan</code> operations.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any patches, or if
     * any patches are detected with a status of <code>InstalledPendingReboot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the system.
     * Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>. These patches
     * might not be in effect until a reboot is performed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rebootOption} will
     * return {@link RebootOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #rebootOptionAsString}.
     * </p>
     * 
     * @return Indicates the reboot option specified in the patch baseline.</p> <note>
     *         <p>
     *         Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch Manager
     *         <code>Scan</code> operations.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any patches,
     *         or if any patches are detected with a status of <code>InstalledPendingReboot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the
     *         system. Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>.
     *         These patches might not be in effect until a reboot is performed.
     *         </p>
     *         </li>
     * @see RebootOption
     */
    public final String rebootOptionAsString() {
        return rebootOption;
    }

    /**
     * <p>
     * The number of patches per node that are specified as <code>Critical</code> for compliance reporting in the patch
     * baseline aren't installed. These patches might be missing, have failed installation, were rejected, or were
     * installed but awaiting a required managed node reboot. The status of these managed nodes is
     * <code>NON_COMPLIANT</code>.
     * </p>
     * 
     * @return The number of patches per node that are specified as <code>Critical</code> for compliance reporting in
     *         the patch baseline aren't installed. These patches might be missing, have failed installation, were
     *         rejected, or were installed but awaiting a required managed node reboot. The status of these managed
     *         nodes is <code>NON_COMPLIANT</code>.
     */
    public final Integer criticalNonCompliantCount() {
        return criticalNonCompliantCount;
    }

    /**
     * <p>
     * The number of patches per node that are specified as <code>Security</code> in a patch advisory aren't installed.
     * These patches might be missing, have failed installation, were rejected, or were installed but awaiting a
     * required managed node reboot. The status of these managed nodes is <code>NON_COMPLIANT</code>.
     * </p>
     * 
     * @return The number of patches per node that are specified as <code>Security</code> in a patch advisory aren't
     *         installed. These patches might be missing, have failed installation, were rejected, or were installed but
     *         awaiting a required managed node reboot. The status of these managed nodes is <code>NON_COMPLIANT</code>.
     */
    public final Integer securityNonCompliantCount() {
        return securityNonCompliantCount;
    }

    /**
     * <p>
     * The number of patches per node that are specified as other than <code>Critical</code> or <code>Security</code>
     * but aren't compliant with the patch baseline. The status of these managed nodes is <code>NON_COMPLIANT</code>.
     * </p>
     * 
     * @return The number of patches per node that are specified as other than <code>Critical</code> or
     *         <code>Security</code> but aren't compliant with the patch baseline. The status of these managed nodes is
     *         <code>NON_COMPLIANT</code>.
     */
    public final Integer otherNonCompliantCount() {
        return otherNonCompliantCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(patchGroup());
        hashCode = 31 * hashCode + Objects.hashCode(baselineId());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(installOverrideList());
        hashCode = 31 * hashCode + Objects.hashCode(ownerInformation());
        hashCode = 31 * hashCode + Objects.hashCode(installedCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedOtherCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedPendingRebootCount());
        hashCode = 31 * hashCode + Objects.hashCode(installedRejectedCount());
        hashCode = 31 * hashCode + Objects.hashCode(missingCount());
        hashCode = 31 * hashCode + Objects.hashCode(failedCount());
        hashCode = 31 * hashCode + Objects.hashCode(unreportedNotApplicableCount());
        hashCode = 31 * hashCode + Objects.hashCode(notApplicableCount());
        hashCode = 31 * hashCode + Objects.hashCode(availableSecurityUpdateCount());
        hashCode = 31 * hashCode + Objects.hashCode(operationStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastNoRebootInstallOperationTime());
        hashCode = 31 * hashCode + Objects.hashCode(rebootOptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(criticalNonCompliantCount());
        hashCode = 31 * hashCode + Objects.hashCode(securityNonCompliantCount());
        hashCode = 31 * hashCode + Objects.hashCode(otherNonCompliantCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstancePatchState)) {
            return false;
        }
        InstancePatchState other = (InstancePatchState) obj;
        return Objects.equals(instanceId(), other.instanceId()) && Objects.equals(patchGroup(), other.patchGroup())
                && Objects.equals(baselineId(), other.baselineId()) && Objects.equals(snapshotId(), other.snapshotId())
                && Objects.equals(installOverrideList(), other.installOverrideList())
                && Objects.equals(ownerInformation(), other.ownerInformation())
                && Objects.equals(installedCount(), other.installedCount())
                && Objects.equals(installedOtherCount(), other.installedOtherCount())
                && Objects.equals(installedPendingRebootCount(), other.installedPendingRebootCount())
                && Objects.equals(installedRejectedCount(), other.installedRejectedCount())
                && Objects.equals(missingCount(), other.missingCount()) && Objects.equals(failedCount(), other.failedCount())
                && Objects.equals(unreportedNotApplicableCount(), other.unreportedNotApplicableCount())
                && Objects.equals(notApplicableCount(), other.notApplicableCount())
                && Objects.equals(availableSecurityUpdateCount(), other.availableSecurityUpdateCount())
                && Objects.equals(operationStartTime(), other.operationStartTime())
                && Objects.equals(operationEndTime(), other.operationEndTime())
                && Objects.equals(operationAsString(), other.operationAsString())
                && Objects.equals(lastNoRebootInstallOperationTime(), other.lastNoRebootInstallOperationTime())
                && Objects.equals(rebootOptionAsString(), other.rebootOptionAsString())
                && Objects.equals(criticalNonCompliantCount(), other.criticalNonCompliantCount())
                && Objects.equals(securityNonCompliantCount(), other.securityNonCompliantCount())
                && Objects.equals(otherNonCompliantCount(), other.otherNonCompliantCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstancePatchState").add("InstanceId", instanceId()).add("PatchGroup", patchGroup())
                .add("BaselineId", baselineId()).add("SnapshotId", snapshotId())
                .add("InstallOverrideList", installOverrideList())
                .add("OwnerInformation", ownerInformation() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InstalledCount", installedCount()).add("InstalledOtherCount", installedOtherCount())
                .add("InstalledPendingRebootCount", installedPendingRebootCount())
                .add("InstalledRejectedCount", installedRejectedCount()).add("MissingCount", missingCount())
                .add("FailedCount", failedCount()).add("UnreportedNotApplicableCount", unreportedNotApplicableCount())
                .add("NotApplicableCount", notApplicableCount())
                .add("AvailableSecurityUpdateCount", availableSecurityUpdateCount())
                .add("OperationStartTime", operationStartTime()).add("OperationEndTime", operationEndTime())
                .add("Operation", operationAsString())
                .add("LastNoRebootInstallOperationTime", lastNoRebootInstallOperationTime())
                .add("RebootOption", rebootOptionAsString()).add("CriticalNonCompliantCount", criticalNonCompliantCount())
                .add("SecurityNonCompliantCount", securityNonCompliantCount())
                .add("OtherNonCompliantCount", otherNonCompliantCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "PatchGroup":
            return Optional.ofNullable(clazz.cast(patchGroup()));
        case "BaselineId":
            return Optional.ofNullable(clazz.cast(baselineId()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "InstallOverrideList":
            return Optional.ofNullable(clazz.cast(installOverrideList()));
        case "OwnerInformation":
            return Optional.ofNullable(clazz.cast(ownerInformation()));
        case "InstalledCount":
            return Optional.ofNullable(clazz.cast(installedCount()));
        case "InstalledOtherCount":
            return Optional.ofNullable(clazz.cast(installedOtherCount()));
        case "InstalledPendingRebootCount":
            return Optional.ofNullable(clazz.cast(installedPendingRebootCount()));
        case "InstalledRejectedCount":
            return Optional.ofNullable(clazz.cast(installedRejectedCount()));
        case "MissingCount":
            return Optional.ofNullable(clazz.cast(missingCount()));
        case "FailedCount":
            return Optional.ofNullable(clazz.cast(failedCount()));
        case "UnreportedNotApplicableCount":
            return Optional.ofNullable(clazz.cast(unreportedNotApplicableCount()));
        case "NotApplicableCount":
            return Optional.ofNullable(clazz.cast(notApplicableCount()));
        case "AvailableSecurityUpdateCount":
            return Optional.ofNullable(clazz.cast(availableSecurityUpdateCount()));
        case "OperationStartTime":
            return Optional.ofNullable(clazz.cast(operationStartTime()));
        case "OperationEndTime":
            return Optional.ofNullable(clazz.cast(operationEndTime()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operationAsString()));
        case "LastNoRebootInstallOperationTime":
            return Optional.ofNullable(clazz.cast(lastNoRebootInstallOperationTime()));
        case "RebootOption":
            return Optional.ofNullable(clazz.cast(rebootOptionAsString()));
        case "CriticalNonCompliantCount":
            return Optional.ofNullable(clazz.cast(criticalNonCompliantCount()));
        case "SecurityNonCompliantCount":
            return Optional.ofNullable(clazz.cast(securityNonCompliantCount()));
        case "OtherNonCompliantCount":
            return Optional.ofNullable(clazz.cast(otherNonCompliantCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceId", INSTANCE_ID_FIELD);
        map.put("PatchGroup", PATCH_GROUP_FIELD);
        map.put("BaselineId", BASELINE_ID_FIELD);
        map.put("SnapshotId", SNAPSHOT_ID_FIELD);
        map.put("InstallOverrideList", INSTALL_OVERRIDE_LIST_FIELD);
        map.put("OwnerInformation", OWNER_INFORMATION_FIELD);
        map.put("InstalledCount", INSTALLED_COUNT_FIELD);
        map.put("InstalledOtherCount", INSTALLED_OTHER_COUNT_FIELD);
        map.put("InstalledPendingRebootCount", INSTALLED_PENDING_REBOOT_COUNT_FIELD);
        map.put("InstalledRejectedCount", INSTALLED_REJECTED_COUNT_FIELD);
        map.put("MissingCount", MISSING_COUNT_FIELD);
        map.put("FailedCount", FAILED_COUNT_FIELD);
        map.put("UnreportedNotApplicableCount", UNREPORTED_NOT_APPLICABLE_COUNT_FIELD);
        map.put("NotApplicableCount", NOT_APPLICABLE_COUNT_FIELD);
        map.put("AvailableSecurityUpdateCount", AVAILABLE_SECURITY_UPDATE_COUNT_FIELD);
        map.put("OperationStartTime", OPERATION_START_TIME_FIELD);
        map.put("OperationEndTime", OPERATION_END_TIME_FIELD);
        map.put("Operation", OPERATION_FIELD);
        map.put("LastNoRebootInstallOperationTime", LAST_NO_REBOOT_INSTALL_OPERATION_TIME_FIELD);
        map.put("RebootOption", REBOOT_OPTION_FIELD);
        map.put("CriticalNonCompliantCount", CRITICAL_NON_COMPLIANT_COUNT_FIELD);
        map.put("SecurityNonCompliantCount", SECURITY_NON_COMPLIANT_COUNT_FIELD);
        map.put("OtherNonCompliantCount", OTHER_NON_COMPLIANT_COUNT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstancePatchState, T> g) {
        return obj -> g.apply((InstancePatchState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstancePatchState> {
        /**
         * <p>
         * The ID of the managed node the high-level patch compliance information was collected for.
         * </p>
         * 
         * @param instanceId
         *        The ID of the managed node the high-level patch compliance information was collected for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The name of the patch group the managed node belongs to.
         * </p>
         * 
         * @param patchGroup
         *        The name of the patch group the managed node belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patchGroup(String patchGroup);

        /**
         * <p>
         * The ID of the patch baseline used to patch the managed node.
         * </p>
         * 
         * @param baselineId
         *        The ID of the patch baseline used to patch the managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineId(String baselineId);

        /**
         * <p>
         * The ID of the patch baseline snapshot used during the patching operation when this compliance data was
         * collected.
         * </p>
         * 
         * @param snapshotId
         *        The ID of the patch baseline snapshot used during the patching operation when this compliance data was
         *        collected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * An https URL or an Amazon Simple Storage Service (Amazon S3) path-style URL to a list of patches to be
         * installed. This patch installation list, which you maintain in an S3 bucket in YAML format and specify in the
         * SSM document <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default patch
         * baseline.
         * </p>
         * <p>
         * For more information about the <code>InstallOverrideList</code> parameter, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html"
         * >SSM Command document for patching: <code>AWS-RunPatchBaseline</code> </a> in the <i>Amazon Web Services
         * Systems Manager User Guide</i>.
         * </p>
         * 
         * @param installOverrideList
         *        An https URL or an Amazon Simple Storage Service (Amazon S3) path-style URL to a list of patches to be
         *        installed. This patch installation list, which you maintain in an S3 bucket in YAML format and specify
         *        in the SSM document <code>AWS-RunPatchBaseline</code>, overrides the patches specified by the default
         *        patch baseline.</p>
         *        <p>
         *        For more information about the <code>InstallOverrideList</code> parameter, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/patch-manager-about-aws-runpatchbaseline.html"
         *        >SSM Command document for patching: <code>AWS-RunPatchBaseline</code> </a> in the <i>Amazon Web
         *        Services Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installOverrideList(String installOverrideList);

        /**
         * <p>
         * Placeholder information. This field will always be empty in the current release of the service.
         * </p>
         * 
         * @param ownerInformation
         *        Placeholder information. This field will always be empty in the current release of the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerInformation(String ownerInformation);

        /**
         * <p>
         * The number of patches from the patch baseline that are installed on the managed node.
         * </p>
         * 
         * @param installedCount
         *        The number of patches from the patch baseline that are installed on the managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedCount(Integer installedCount);

        /**
         * <p>
         * The number of patches not specified in the patch baseline that are installed on the managed node.
         * </p>
         * 
         * @param installedOtherCount
         *        The number of patches not specified in the patch baseline that are installed on the managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedOtherCount(Integer installedOtherCount);

        /**
         * <p>
         * The number of patches installed by Patch Manager since the last time the managed node was rebooted.
         * </p>
         * 
         * @param installedPendingRebootCount
         *        The number of patches installed by Patch Manager since the last time the managed node was rebooted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedPendingRebootCount(Integer installedPendingRebootCount);

        /**
         * <p>
         * The number of patches installed on a managed node that are specified in a <code>RejectedPatches</code> list.
         * Patches with a status of <code>InstalledRejected</code> were typically installed before they were added to a
         * <code>RejectedPatches</code> list.
         * </p>
         * <note>
         * <p>
         * If <code>ALLOW_AS_DEPENDENCY</code> is the specified option for <code>RejectedPatchesAction</code>, the value
         * of <code>InstalledRejectedCount</code> will always be <code>0</code> (zero).
         * </p>
         * </note>
         * 
         * @param installedRejectedCount
         *        The number of patches installed on a managed node that are specified in a <code>RejectedPatches</code>
         *        list. Patches with a status of <code>InstalledRejected</code> were typically installed before they
         *        were added to a <code>RejectedPatches</code> list.</p> <note>
         *        <p>
         *        If <code>ALLOW_AS_DEPENDENCY</code> is the specified option for <code>RejectedPatchesAction</code>,
         *        the value of <code>InstalledRejectedCount</code> will always be <code>0</code> (zero).
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder installedRejectedCount(Integer installedRejectedCount);

        /**
         * <p>
         * The number of patches from the patch baseline that are applicable for the managed node but aren't currently
         * installed.
         * </p>
         * 
         * @param missingCount
         *        The number of patches from the patch baseline that are applicable for the managed node but aren't
         *        currently installed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingCount(Integer missingCount);

        /**
         * <p>
         * The number of patches from the patch baseline that were attempted to be installed during the last patching
         * operation, but failed to install.
         * </p>
         * 
         * @param failedCount
         *        The number of patches from the patch baseline that were attempted to be installed during the last
         *        patching operation, but failed to install.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedCount(Integer failedCount);

        /**
         * <p>
         * The number of patches beyond the supported limit of <code>NotApplicableCount</code> that aren't reported by
         * name to Inventory. Inventory is a tool in Amazon Web Services Systems Manager.
         * </p>
         * 
         * @param unreportedNotApplicableCount
         *        The number of patches beyond the supported limit of <code>NotApplicableCount</code> that aren't
         *        reported by name to Inventory. Inventory is a tool in Amazon Web Services Systems Manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unreportedNotApplicableCount(Integer unreportedNotApplicableCount);

        /**
         * <p>
         * The number of patches from the patch baseline that aren't applicable for the managed node and therefore
         * aren't installed on the node. This number may be truncated if the list of patch names is very large. The
         * number of patches beyond this limit are reported in <code>UnreportedNotApplicableCount</code>.
         * </p>
         * 
         * @param notApplicableCount
         *        The number of patches from the patch baseline that aren't applicable for the managed node and
         *        therefore aren't installed on the node. This number may be truncated if the list of patch names is
         *        very large. The number of patches beyond this limit are reported in
         *        <code>UnreportedNotApplicableCount</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notApplicableCount(Integer notApplicableCount);

        /**
         * <p>
         * The number of security-related patches that are available but not approved because they didn't meet the patch
         * baseline requirements. For example, an updated version of a patch might have been released before the
         * specified auto-approval period was over.
         * </p>
         * <p>
         * Applies to Windows Server managed nodes only.
         * </p>
         * 
         * @param availableSecurityUpdateCount
         *        The number of security-related patches that are available but not approved because they didn't meet
         *        the patch baseline requirements. For example, an updated version of a patch might have been released
         *        before the specified auto-approval period was over.</p>
         *        <p>
         *        Applies to Windows Server managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableSecurityUpdateCount(Integer availableSecurityUpdateCount);

        /**
         * <p>
         * The time the most recent patching operation was started on the managed node.
         * </p>
         * 
         * @param operationStartTime
         *        The time the most recent patching operation was started on the managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationStartTime(Instant operationStartTime);

        /**
         * <p>
         * The time the most recent patching operation completed on the managed node.
         * </p>
         * 
         * @param operationEndTime
         *        The time the most recent patching operation completed on the managed node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationEndTime(Instant operationEndTime);

        /**
         * <p>
         * The type of patching operation that was performed: or
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SCAN</code> assesses the patch compliance state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSTALL</code> installs missing patches.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operation
         *        The type of patching operation that was performed: or </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SCAN</code> assesses the patch compliance state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSTALL</code> installs missing patches.
         *        </p>
         *        </li>
         * @see PatchOperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchOperationType
         */
        Builder operation(String operation);

        /**
         * <p>
         * The type of patching operation that was performed: or
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SCAN</code> assesses the patch compliance state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSTALL</code> installs missing patches.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operation
         *        The type of patching operation that was performed: or </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SCAN</code> assesses the patch compliance state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSTALL</code> installs missing patches.
         *        </p>
         *        </li>
         * @see PatchOperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchOperationType
         */
        Builder operation(PatchOperationType operation);

        /**
         * <p>
         * The time of the last attempt to patch the managed node with <code>NoReboot</code> specified as the reboot
         * option.
         * </p>
         * 
         * @param lastNoRebootInstallOperationTime
         *        The time of the last attempt to patch the managed node with <code>NoReboot</code> specified as the
         *        reboot option.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastNoRebootInstallOperationTime(Instant lastNoRebootInstallOperationTime);

        /**
         * <p>
         * Indicates the reboot option specified in the patch baseline.
         * </p>
         * <note>
         * <p>
         * Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch Manager
         * <code>Scan</code> operations.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any patches, or
         * if any patches are detected with a status of <code>InstalledPendingReboot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the
         * system. Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>.
         * These patches might not be in effect until a reboot is performed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rebootOption
         *        Indicates the reboot option specified in the patch baseline.</p> <note>
         *        <p>
         *        Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch
         *        Manager <code>Scan</code> operations.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any
         *        patches, or if any patches are detected with a status of <code>InstalledPendingReboot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the
         *        system. Patches installed with this option are assigned a status of
         *        <code>InstalledPendingReboot</code>. These patches might not be in effect until a reboot is performed.
         *        </p>
         *        </li>
         * @see RebootOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RebootOption
         */
        Builder rebootOption(String rebootOption);

        /**
         * <p>
         * Indicates the reboot option specified in the patch baseline.
         * </p>
         * <note>
         * <p>
         * Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch Manager
         * <code>Scan</code> operations.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any patches, or
         * if any patches are detected with a status of <code>InstalledPendingReboot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the
         * system. Patches installed with this option are assigned a status of <code>InstalledPendingReboot</code>.
         * These patches might not be in effect until a reboot is performed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rebootOption
         *        Indicates the reboot option specified in the patch baseline.</p> <note>
         *        <p>
         *        Reboot options apply to <code>Install</code> operations only. Reboots aren't attempted for Patch
         *        Manager <code>Scan</code> operations.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>RebootIfNeeded</code>: Patch Manager tries to reboot the managed node if it installed any
         *        patches, or if any patches are detected with a status of <code>InstalledPendingReboot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NoReboot</code>: Patch Manager attempts to install missing packages without trying to reboot the
         *        system. Patches installed with this option are assigned a status of
         *        <code>InstalledPendingReboot</code>. These patches might not be in effect until a reboot is performed.
         *        </p>
         *        </li>
         * @see RebootOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RebootOption
         */
        Builder rebootOption(RebootOption rebootOption);

        /**
         * <p>
         * The number of patches per node that are specified as <code>Critical</code> for compliance reporting in the
         * patch baseline aren't installed. These patches might be missing, have failed installation, were rejected, or
         * were installed but awaiting a required managed node reboot. The status of these managed nodes is
         * <code>NON_COMPLIANT</code>.
         * </p>
         * 
         * @param criticalNonCompliantCount
         *        The number of patches per node that are specified as <code>Critical</code> for compliance reporting in
         *        the patch baseline aren't installed. These patches might be missing, have failed installation, were
         *        rejected, or were installed but awaiting a required managed node reboot. The status of these managed
         *        nodes is <code>NON_COMPLIANT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder criticalNonCompliantCount(Integer criticalNonCompliantCount);

        /**
         * <p>
         * The number of patches per node that are specified as <code>Security</code> in a patch advisory aren't
         * installed. These patches might be missing, have failed installation, were rejected, or were installed but
         * awaiting a required managed node reboot. The status of these managed nodes is <code>NON_COMPLIANT</code>.
         * </p>
         * 
         * @param securityNonCompliantCount
         *        The number of patches per node that are specified as <code>Security</code> in a patch advisory aren't
         *        installed. These patches might be missing, have failed installation, were rejected, or were installed
         *        but awaiting a required managed node reboot. The status of these managed nodes is
         *        <code>NON_COMPLIANT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityNonCompliantCount(Integer securityNonCompliantCount);

        /**
         * <p>
         * The number of patches per node that are specified as other than <code>Critical</code> or
         * <code>Security</code> but aren't compliant with the patch baseline. The status of these managed nodes is
         * <code>NON_COMPLIANT</code>.
         * </p>
         * 
         * @param otherNonCompliantCount
         *        The number of patches per node that are specified as other than <code>Critical</code> or
         *        <code>Security</code> but aren't compliant with the patch baseline. The status of these managed nodes
         *        is <code>NON_COMPLIANT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder otherNonCompliantCount(Integer otherNonCompliantCount);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private String patchGroup;

        private String baselineId;

        private String snapshotId;

        private String installOverrideList;

        private String ownerInformation;

        private Integer installedCount;

        private Integer installedOtherCount;

        private Integer installedPendingRebootCount;

        private Integer installedRejectedCount;

        private Integer missingCount;

        private Integer failedCount;

        private Integer unreportedNotApplicableCount;

        private Integer notApplicableCount;

        private Integer availableSecurityUpdateCount;

        private Instant operationStartTime;

        private Instant operationEndTime;

        private String operation;

        private Instant lastNoRebootInstallOperationTime;

        private String rebootOption;

        private Integer criticalNonCompliantCount;

        private Integer securityNonCompliantCount;

        private Integer otherNonCompliantCount;

        private BuilderImpl() {
        }

        private BuilderImpl(InstancePatchState model) {
            instanceId(model.instanceId);
            patchGroup(model.patchGroup);
            baselineId(model.baselineId);
            snapshotId(model.snapshotId);
            installOverrideList(model.installOverrideList);
            ownerInformation(model.ownerInformation);
            installedCount(model.installedCount);
            installedOtherCount(model.installedOtherCount);
            installedPendingRebootCount(model.installedPendingRebootCount);
            installedRejectedCount(model.installedRejectedCount);
            missingCount(model.missingCount);
            failedCount(model.failedCount);
            unreportedNotApplicableCount(model.unreportedNotApplicableCount);
            notApplicableCount(model.notApplicableCount);
            availableSecurityUpdateCount(model.availableSecurityUpdateCount);
            operationStartTime(model.operationStartTime);
            operationEndTime(model.operationEndTime);
            operation(model.operation);
            lastNoRebootInstallOperationTime(model.lastNoRebootInstallOperationTime);
            rebootOption(model.rebootOption);
            criticalNonCompliantCount(model.criticalNonCompliantCount);
            securityNonCompliantCount(model.securityNonCompliantCount);
            otherNonCompliantCount(model.otherNonCompliantCount);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getPatchGroup() {
            return patchGroup;
        }

        public final void setPatchGroup(String patchGroup) {
            this.patchGroup = patchGroup;
        }

        @Override
        public final Builder patchGroup(String patchGroup) {
            this.patchGroup = patchGroup;
            return this;
        }

        public final String getBaselineId() {
            return baselineId;
        }

        public final void setBaselineId(String baselineId) {
            this.baselineId = baselineId;
        }

        @Override
        public final Builder baselineId(String baselineId) {
            this.baselineId = baselineId;
            return this;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final String getInstallOverrideList() {
            return installOverrideList;
        }

        public final void setInstallOverrideList(String installOverrideList) {
            this.installOverrideList = installOverrideList;
        }

        @Override
        public final Builder installOverrideList(String installOverrideList) {
            this.installOverrideList = installOverrideList;
            return this;
        }

        public final String getOwnerInformation() {
            return ownerInformation;
        }

        public final void setOwnerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
        }

        @Override
        public final Builder ownerInformation(String ownerInformation) {
            this.ownerInformation = ownerInformation;
            return this;
        }

        public final Integer getInstalledCount() {
            return installedCount;
        }

        public final void setInstalledCount(Integer installedCount) {
            this.installedCount = installedCount;
        }

        @Override
        public final Builder installedCount(Integer installedCount) {
            this.installedCount = installedCount;
            return this;
        }

        public final Integer getInstalledOtherCount() {
            return installedOtherCount;
        }

        public final void setInstalledOtherCount(Integer installedOtherCount) {
            this.installedOtherCount = installedOtherCount;
        }

        @Override
        public final Builder installedOtherCount(Integer installedOtherCount) {
            this.installedOtherCount = installedOtherCount;
            return this;
        }

        public final Integer getInstalledPendingRebootCount() {
            return installedPendingRebootCount;
        }

        public final void setInstalledPendingRebootCount(Integer installedPendingRebootCount) {
            this.installedPendingRebootCount = installedPendingRebootCount;
        }

        @Override
        public final Builder installedPendingRebootCount(Integer installedPendingRebootCount) {
            this.installedPendingRebootCount = installedPendingRebootCount;
            return this;
        }

        public final Integer getInstalledRejectedCount() {
            return installedRejectedCount;
        }

        public final void setInstalledRejectedCount(Integer installedRejectedCount) {
            this.installedRejectedCount = installedRejectedCount;
        }

        @Override
        public final Builder installedRejectedCount(Integer installedRejectedCount) {
            this.installedRejectedCount = installedRejectedCount;
            return this;
        }

        public final Integer getMissingCount() {
            return missingCount;
        }

        public final void setMissingCount(Integer missingCount) {
            this.missingCount = missingCount;
        }

        @Override
        public final Builder missingCount(Integer missingCount) {
            this.missingCount = missingCount;
            return this;
        }

        public final Integer getFailedCount() {
            return failedCount;
        }

        public final void setFailedCount(Integer failedCount) {
            this.failedCount = failedCount;
        }

        @Override
        public final Builder failedCount(Integer failedCount) {
            this.failedCount = failedCount;
            return this;
        }

        public final Integer getUnreportedNotApplicableCount() {
            return unreportedNotApplicableCount;
        }

        public final void setUnreportedNotApplicableCount(Integer unreportedNotApplicableCount) {
            this.unreportedNotApplicableCount = unreportedNotApplicableCount;
        }

        @Override
        public final Builder unreportedNotApplicableCount(Integer unreportedNotApplicableCount) {
            this.unreportedNotApplicableCount = unreportedNotApplicableCount;
            return this;
        }

        public final Integer getNotApplicableCount() {
            return notApplicableCount;
        }

        public final void setNotApplicableCount(Integer notApplicableCount) {
            this.notApplicableCount = notApplicableCount;
        }

        @Override
        public final Builder notApplicableCount(Integer notApplicableCount) {
            this.notApplicableCount = notApplicableCount;
            return this;
        }

        public final Integer getAvailableSecurityUpdateCount() {
            return availableSecurityUpdateCount;
        }

        public final void setAvailableSecurityUpdateCount(Integer availableSecurityUpdateCount) {
            this.availableSecurityUpdateCount = availableSecurityUpdateCount;
        }

        @Override
        public final Builder availableSecurityUpdateCount(Integer availableSecurityUpdateCount) {
            this.availableSecurityUpdateCount = availableSecurityUpdateCount;
            return this;
        }

        public final Instant getOperationStartTime() {
            return operationStartTime;
        }

        public final void setOperationStartTime(Instant operationStartTime) {
            this.operationStartTime = operationStartTime;
        }

        @Override
        public final Builder operationStartTime(Instant operationStartTime) {
            this.operationStartTime = operationStartTime;
            return this;
        }

        public final Instant getOperationEndTime() {
            return operationEndTime;
        }

        public final void setOperationEndTime(Instant operationEndTime) {
            this.operationEndTime = operationEndTime;
        }

        @Override
        public final Builder operationEndTime(Instant operationEndTime) {
            this.operationEndTime = operationEndTime;
            return this;
        }

        public final String getOperation() {
            return operation;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        public final Builder operation(PatchOperationType operation) {
            this.operation(operation == null ? null : operation.toString());
            return this;
        }

        public final Instant getLastNoRebootInstallOperationTime() {
            return lastNoRebootInstallOperationTime;
        }

        public final void setLastNoRebootInstallOperationTime(Instant lastNoRebootInstallOperationTime) {
            this.lastNoRebootInstallOperationTime = lastNoRebootInstallOperationTime;
        }

        @Override
        public final Builder lastNoRebootInstallOperationTime(Instant lastNoRebootInstallOperationTime) {
            this.lastNoRebootInstallOperationTime = lastNoRebootInstallOperationTime;
            return this;
        }

        public final String getRebootOption() {
            return rebootOption;
        }

        public final void setRebootOption(String rebootOption) {
            this.rebootOption = rebootOption;
        }

        @Override
        public final Builder rebootOption(String rebootOption) {
            this.rebootOption = rebootOption;
            return this;
        }

        @Override
        public final Builder rebootOption(RebootOption rebootOption) {
            this.rebootOption(rebootOption == null ? null : rebootOption.toString());
            return this;
        }

        public final Integer getCriticalNonCompliantCount() {
            return criticalNonCompliantCount;
        }

        public final void setCriticalNonCompliantCount(Integer criticalNonCompliantCount) {
            this.criticalNonCompliantCount = criticalNonCompliantCount;
        }

        @Override
        public final Builder criticalNonCompliantCount(Integer criticalNonCompliantCount) {
            this.criticalNonCompliantCount = criticalNonCompliantCount;
            return this;
        }

        public final Integer getSecurityNonCompliantCount() {
            return securityNonCompliantCount;
        }

        public final void setSecurityNonCompliantCount(Integer securityNonCompliantCount) {
            this.securityNonCompliantCount = securityNonCompliantCount;
        }

        @Override
        public final Builder securityNonCompliantCount(Integer securityNonCompliantCount) {
            this.securityNonCompliantCount = securityNonCompliantCount;
            return this;
        }

        public final Integer getOtherNonCompliantCount() {
            return otherNonCompliantCount;
        }

        public final void setOtherNonCompliantCount(Integer otherNonCompliantCount) {
            this.otherNonCompliantCount = otherNonCompliantCount;
        }

        @Override
        public final Builder otherNonCompliantCount(Integer otherNonCompliantCount) {
            this.otherNonCompliantCount = otherNonCompliantCount;
            return this;
        }

        @Override
        public InstancePatchState build() {
            return new InstancePatchState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
