/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents metadata about a patch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Patch implements SdkPojo, Serializable, ToCopyableBuilder<Patch.Builder, Patch> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Patch::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Instant> RELEASE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReleaseDate").getter(getter(Patch::releaseDate)).setter(setter(Builder::releaseDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseDate").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Patch::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Patch::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CONTENT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentUrl").getter(getter(Patch::contentUrl)).setter(setter(Builder::contentUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentUrl").build()).build();

    private static final SdkField<String> VENDOR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Vendor")
            .getter(getter(Patch::vendor)).setter(setter(Builder::vendor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Vendor").build()).build();

    private static final SdkField<String> PRODUCT_FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProductFamily").getter(getter(Patch::productFamily)).setter(setter(Builder::productFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductFamily").build()).build();

    private static final SdkField<String> PRODUCT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Product")
            .getter(getter(Patch::product)).setter(setter(Builder::product))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Product").build()).build();

    private static final SdkField<String> CLASSIFICATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Classification").getter(getter(Patch::classification)).setter(setter(Builder::classification))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Classification").build()).build();

    private static final SdkField<String> MSRC_SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MsrcSeverity").getter(getter(Patch::msrcSeverity)).setter(setter(Builder::msrcSeverity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MsrcSeverity").build()).build();

    private static final SdkField<String> KB_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KbNumber").getter(getter(Patch::kbNumber)).setter(setter(Builder::kbNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KbNumber").build()).build();

    private static final SdkField<String> MSRC_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MsrcNumber").getter(getter(Patch::msrcNumber)).setter(setter(Builder::msrcNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MsrcNumber").build()).build();

    private static final SdkField<String> LANGUAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Language").getter(getter(Patch::language)).setter(setter(Builder::language))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Language").build()).build();

    private static final SdkField<List<String>> ADVISORY_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdvisoryIds")
            .getter(getter(Patch::advisoryIds))
            .setter(setter(Builder::advisoryIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvisoryIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> BUGZILLA_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("BugzillaIds")
            .getter(getter(Patch::bugzillaIds))
            .setter(setter(Builder::bugzillaIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BugzillaIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CVE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CVEIds")
            .getter(getter(Patch::cveIds))
            .setter(setter(Builder::cveIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CVEIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Patch::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Integer> EPOCH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Epoch")
            .getter(getter(Patch::epoch)).setter(setter(Builder::epoch))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Epoch").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(Patch::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> RELEASE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Release")
            .getter(getter(Patch::release)).setter(setter(Builder::release))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Release").build()).build();

    private static final SdkField<String> ARCH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arch")
            .getter(getter(Patch::arch)).setter(setter(Builder::arch))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arch").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(Patch::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<String> REPOSITORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Repository").getter(getter(Patch::repository)).setter(setter(Builder::repository))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Repository").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, RELEASE_DATE_FIELD,
            TITLE_FIELD, DESCRIPTION_FIELD, CONTENT_URL_FIELD, VENDOR_FIELD, PRODUCT_FAMILY_FIELD, PRODUCT_FIELD,
            CLASSIFICATION_FIELD, MSRC_SEVERITY_FIELD, KB_NUMBER_FIELD, MSRC_NUMBER_FIELD, LANGUAGE_FIELD, ADVISORY_IDS_FIELD,
            BUGZILLA_IDS_FIELD, CVE_IDS_FIELD, NAME_FIELD, EPOCH_FIELD, VERSION_FIELD, RELEASE_FIELD, ARCH_FIELD, SEVERITY_FIELD,
            REPOSITORY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Instant releaseDate;

    private final String title;

    private final String description;

    private final String contentUrl;

    private final String vendor;

    private final String productFamily;

    private final String product;

    private final String classification;

    private final String msrcSeverity;

    private final String kbNumber;

    private final String msrcNumber;

    private final String language;

    private final List<String> advisoryIds;

    private final List<String> bugzillaIds;

    private final List<String> cveIds;

    private final String name;

    private final Integer epoch;

    private final String version;

    private final String release;

    private final String arch;

    private final String severity;

    private final String repository;

    private Patch(BuilderImpl builder) {
        this.id = builder.id;
        this.releaseDate = builder.releaseDate;
        this.title = builder.title;
        this.description = builder.description;
        this.contentUrl = builder.contentUrl;
        this.vendor = builder.vendor;
        this.productFamily = builder.productFamily;
        this.product = builder.product;
        this.classification = builder.classification;
        this.msrcSeverity = builder.msrcSeverity;
        this.kbNumber = builder.kbNumber;
        this.msrcNumber = builder.msrcNumber;
        this.language = builder.language;
        this.advisoryIds = builder.advisoryIds;
        this.bugzillaIds = builder.bugzillaIds;
        this.cveIds = builder.cveIds;
        this.name = builder.name;
        this.epoch = builder.epoch;
        this.version = builder.version;
        this.release = builder.release;
        this.arch = builder.arch;
        this.severity = builder.severity;
        this.repository = builder.repository;
    }

    /**
     * <p>
     * The ID of the patch. Applies to Windows patches only.
     * </p>
     * <note>
     * <p>
     * This ID isn't the same as the Microsoft Knowledge Base ID.
     * </p>
     * </note>
     * 
     * @return The ID of the patch. Applies to Windows patches only.</p> <note>
     *         <p>
     *         This ID isn't the same as the Microsoft Knowledge Base ID.
     *         </p>
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The date the patch was released.
     * </p>
     * 
     * @return The date the patch was released.
     */
    public final Instant releaseDate() {
        return releaseDate;
    }

    /**
     * <p>
     * The title of the patch.
     * </p>
     * 
     * @return The title of the patch.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The description of the patch.
     * </p>
     * 
     * @return The description of the patch.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The URL where more information can be obtained about the patch.
     * </p>
     * 
     * @return The URL where more information can be obtained about the patch.
     */
    public final String contentUrl() {
        return contentUrl;
    }

    /**
     * <p>
     * The name of the vendor providing the patch.
     * </p>
     * 
     * @return The name of the vendor providing the patch.
     */
    public final String vendor() {
        return vendor;
    }

    /**
     * <p>
     * The product family the patch is applicable for. For example, <code>Windows</code> or <code>Amazon Linux 2</code>.
     * </p>
     * 
     * @return The product family the patch is applicable for. For example, <code>Windows</code> or
     *         <code>Amazon Linux 2</code>.
     */
    public final String productFamily() {
        return productFamily;
    }

    /**
     * <p>
     * The specific product the patch is applicable for. For example, <code>WindowsServer2016</code> or
     * <code>AmazonLinux2018.03</code>.
     * </p>
     * 
     * @return The specific product the patch is applicable for. For example, <code>WindowsServer2016</code> or
     *         <code>AmazonLinux2018.03</code>.
     */
    public final String product() {
        return product;
    }

    /**
     * <p>
     * The classification of the patch. For example, <code>SecurityUpdates</code>, <code>Updates</code>, or
     * <code>CriticalUpdates</code>.
     * </p>
     * 
     * @return The classification of the patch. For example, <code>SecurityUpdates</code>, <code>Updates</code>, or
     *         <code>CriticalUpdates</code>.
     */
    public final String classification() {
        return classification;
    }

    /**
     * <p>
     * The severity of the patch, such as <code>Critical</code>, <code>Important</code>, or <code>Moderate</code>.
     * Applies to Windows patches only.
     * </p>
     * 
     * @return The severity of the patch, such as <code>Critical</code>, <code>Important</code>, or
     *         <code>Moderate</code>. Applies to Windows patches only.
     */
    public final String msrcSeverity() {
        return msrcSeverity;
    }

    /**
     * <p>
     * The Microsoft Knowledge Base ID of the patch. Applies to Windows patches only.
     * </p>
     * 
     * @return The Microsoft Knowledge Base ID of the patch. Applies to Windows patches only.
     */
    public final String kbNumber() {
        return kbNumber;
    }

    /**
     * <p>
     * The ID of the Microsoft Security Response Center (MSRC) bulletin the patch is related to. For example,
     * <code>MS14-045</code>. Applies to Windows patches only.
     * </p>
     * 
     * @return The ID of the Microsoft Security Response Center (MSRC) bulletin the patch is related to. For example,
     *         <code>MS14-045</code>. Applies to Windows patches only.
     */
    public final String msrcNumber() {
        return msrcNumber;
    }

    /**
     * <p>
     * The language of the patch if it's language-specific.
     * </p>
     * 
     * @return The language of the patch if it's language-specific.
     */
    public final String language() {
        return language;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdvisoryIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdvisoryIds() {
        return advisoryIds != null && !(advisoryIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Advisory ID of the patch. For example, <code>RHSA-2020:3779</code>. Applies to Linux-based managed nodes
     * only.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdvisoryIds} method.
     * </p>
     * 
     * @return The Advisory ID of the patch. For example, <code>RHSA-2020:3779</code>. Applies to Linux-based managed
     *         nodes only.
     */
    public final List<String> advisoryIds() {
        return advisoryIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the BugzillaIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBugzillaIds() {
        return bugzillaIds != null && !(bugzillaIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Bugzilla ID of the patch. For example, <code>1600646</code>. Applies to Linux-based managed nodes only.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBugzillaIds} method.
     * </p>
     * 
     * @return The Bugzilla ID of the patch. For example, <code>1600646</code>. Applies to Linux-based managed nodes
     *         only.
     */
    public final List<String> bugzillaIds() {
        return bugzillaIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the CVEIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCveIds() {
        return cveIds != null && !(cveIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Common Vulnerabilities and Exposures (CVE) ID of the patch. For example, <code>CVE-2011-3192</code>. Applies
     * to Linux-based managed nodes only.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCveIds} method.
     * </p>
     * 
     * @return The Common Vulnerabilities and Exposures (CVE) ID of the patch. For example, <code>CVE-2011-3192</code>.
     *         Applies to Linux-based managed nodes only.
     */
    public final List<String> cveIds() {
        return cveIds;
    }

    /**
     * <p>
     * The name of the patch. Applies to Linux-based managed nodes only.
     * </p>
     * 
     * @return The name of the patch. Applies to Linux-based managed nodes only.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The epoch of the patch. For example in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the epoch value is
     * <code>20180914-2</code>. Applies to Linux-based managed nodes only.
     * </p>
     * 
     * @return The epoch of the patch. For example in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the epoch
     *         value is <code>20180914-2</code>. Applies to Linux-based managed nodes only.
     */
    public final Integer epoch() {
        return epoch;
    }

    /**
     * <p>
     * The version number of the patch. For example, in <code>example-pkg-1.710.10-2.7.abcd.x86_64</code>, the version
     * number is indicated by <code>-1</code>. Applies to Linux-based managed nodes only.
     * </p>
     * 
     * @return The version number of the patch. For example, in <code>example-pkg-1.710.10-2.7.abcd.x86_64</code>, the
     *         version number is indicated by <code>-1</code>. Applies to Linux-based managed nodes only.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The particular release of a patch. For example, in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the
     * release is <code>2.amaz1</code>. Applies to Linux-based managed nodes only.
     * </p>
     * 
     * @return The particular release of a patch. For example, in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>,
     *         the release is <code>2.amaz1</code>. Applies to Linux-based managed nodes only.
     */
    public final String release() {
        return release;
    }

    /**
     * <p>
     * The architecture of the patch. For example, in <code>example-pkg-0.710.10-2.7.abcd.x86_64</code>, the
     * architecture is indicated by <code>x86_64</code>. Applies to Linux-based managed nodes only.
     * </p>
     * 
     * @return The architecture of the patch. For example, in <code>example-pkg-0.710.10-2.7.abcd.x86_64</code>, the
     *         architecture is indicated by <code>x86_64</code>. Applies to Linux-based managed nodes only.
     */
    public final String arch() {
        return arch;
    }

    /**
     * <p>
     * The severity level of the patch. For example, <code>CRITICAL</code> or <code>MODERATE</code>.
     * </p>
     * 
     * @return The severity level of the patch. For example, <code>CRITICAL</code> or <code>MODERATE</code>.
     */
    public final String severity() {
        return severity;
    }

    /**
     * <p>
     * The source patch repository for the operating system and version, such as <code>trusty-security</code> for Ubuntu
     * Server 14.04 LTE and <code>focal-security</code> for Ubuntu Server 20.04 LTE. Applies to Linux-based managed
     * nodes only.
     * </p>
     * 
     * @return The source patch repository for the operating system and version, such as <code>trusty-security</code>
     *         for Ubuntu Server 14.04 LTE and <code>focal-security</code> for Ubuntu Server 20.04 LTE. Applies to
     *         Linux-based managed nodes only.
     */
    public final String repository() {
        return repository;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(releaseDate());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(contentUrl());
        hashCode = 31 * hashCode + Objects.hashCode(vendor());
        hashCode = 31 * hashCode + Objects.hashCode(productFamily());
        hashCode = 31 * hashCode + Objects.hashCode(product());
        hashCode = 31 * hashCode + Objects.hashCode(classification());
        hashCode = 31 * hashCode + Objects.hashCode(msrcSeverity());
        hashCode = 31 * hashCode + Objects.hashCode(kbNumber());
        hashCode = 31 * hashCode + Objects.hashCode(msrcNumber());
        hashCode = 31 * hashCode + Objects.hashCode(language());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdvisoryIds() ? advisoryIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBugzillaIds() ? bugzillaIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCveIds() ? cveIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(epoch());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(release());
        hashCode = 31 * hashCode + Objects.hashCode(arch());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(repository());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Patch)) {
            return false;
        }
        Patch other = (Patch) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(releaseDate(), other.releaseDate())
                && Objects.equals(title(), other.title()) && Objects.equals(description(), other.description())
                && Objects.equals(contentUrl(), other.contentUrl()) && Objects.equals(vendor(), other.vendor())
                && Objects.equals(productFamily(), other.productFamily()) && Objects.equals(product(), other.product())
                && Objects.equals(classification(), other.classification())
                && Objects.equals(msrcSeverity(), other.msrcSeverity()) && Objects.equals(kbNumber(), other.kbNumber())
                && Objects.equals(msrcNumber(), other.msrcNumber()) && Objects.equals(language(), other.language())
                && hasAdvisoryIds() == other.hasAdvisoryIds() && Objects.equals(advisoryIds(), other.advisoryIds())
                && hasBugzillaIds() == other.hasBugzillaIds() && Objects.equals(bugzillaIds(), other.bugzillaIds())
                && hasCveIds() == other.hasCveIds() && Objects.equals(cveIds(), other.cveIds())
                && Objects.equals(name(), other.name()) && Objects.equals(epoch(), other.epoch())
                && Objects.equals(version(), other.version()) && Objects.equals(release(), other.release())
                && Objects.equals(arch(), other.arch()) && Objects.equals(severity(), other.severity())
                && Objects.equals(repository(), other.repository());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Patch").add("Id", id()).add("ReleaseDate", releaseDate()).add("Title", title())
                .add("Description", description()).add("ContentUrl", contentUrl()).add("Vendor", vendor())
                .add("ProductFamily", productFamily()).add("Product", product()).add("Classification", classification())
                .add("MsrcSeverity", msrcSeverity()).add("KbNumber", kbNumber()).add("MsrcNumber", msrcNumber())
                .add("Language", language()).add("AdvisoryIds", hasAdvisoryIds() ? advisoryIds() : null)
                .add("BugzillaIds", hasBugzillaIds() ? bugzillaIds() : null).add("CVEIds", hasCveIds() ? cveIds() : null)
                .add("Name", name()).add("Epoch", epoch()).add("Version", version()).add("Release", release())
                .add("Arch", arch()).add("Severity", severity()).add("Repository", repository()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "ReleaseDate":
            return Optional.ofNullable(clazz.cast(releaseDate()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ContentUrl":
            return Optional.ofNullable(clazz.cast(contentUrl()));
        case "Vendor":
            return Optional.ofNullable(clazz.cast(vendor()));
        case "ProductFamily":
            return Optional.ofNullable(clazz.cast(productFamily()));
        case "Product":
            return Optional.ofNullable(clazz.cast(product()));
        case "Classification":
            return Optional.ofNullable(clazz.cast(classification()));
        case "MsrcSeverity":
            return Optional.ofNullable(clazz.cast(msrcSeverity()));
        case "KbNumber":
            return Optional.ofNullable(clazz.cast(kbNumber()));
        case "MsrcNumber":
            return Optional.ofNullable(clazz.cast(msrcNumber()));
        case "Language":
            return Optional.ofNullable(clazz.cast(language()));
        case "AdvisoryIds":
            return Optional.ofNullable(clazz.cast(advisoryIds()));
        case "BugzillaIds":
            return Optional.ofNullable(clazz.cast(bugzillaIds()));
        case "CVEIds":
            return Optional.ofNullable(clazz.cast(cveIds()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Epoch":
            return Optional.ofNullable(clazz.cast(epoch()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Release":
            return Optional.ofNullable(clazz.cast(release()));
        case "Arch":
            return Optional.ofNullable(clazz.cast(arch()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "Repository":
            return Optional.ofNullable(clazz.cast(repository()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("ReleaseDate", RELEASE_DATE_FIELD);
        map.put("Title", TITLE_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ContentUrl", CONTENT_URL_FIELD);
        map.put("Vendor", VENDOR_FIELD);
        map.put("ProductFamily", PRODUCT_FAMILY_FIELD);
        map.put("Product", PRODUCT_FIELD);
        map.put("Classification", CLASSIFICATION_FIELD);
        map.put("MsrcSeverity", MSRC_SEVERITY_FIELD);
        map.put("KbNumber", KB_NUMBER_FIELD);
        map.put("MsrcNumber", MSRC_NUMBER_FIELD);
        map.put("Language", LANGUAGE_FIELD);
        map.put("AdvisoryIds", ADVISORY_IDS_FIELD);
        map.put("BugzillaIds", BUGZILLA_IDS_FIELD);
        map.put("CVEIds", CVE_IDS_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Epoch", EPOCH_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("Release", RELEASE_FIELD);
        map.put("Arch", ARCH_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("Repository", REPOSITORY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Patch, T> g) {
        return obj -> g.apply((Patch) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Patch> {
        /**
         * <p>
         * The ID of the patch. Applies to Windows patches only.
         * </p>
         * <note>
         * <p>
         * This ID isn't the same as the Microsoft Knowledge Base ID.
         * </p>
         * </note>
         * 
         * @param id
         *        The ID of the patch. Applies to Windows patches only.</p> <note>
         *        <p>
         *        This ID isn't the same as the Microsoft Knowledge Base ID.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The date the patch was released.
         * </p>
         * 
         * @param releaseDate
         *        The date the patch was released.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseDate(Instant releaseDate);

        /**
         * <p>
         * The title of the patch.
         * </p>
         * 
         * @param title
         *        The title of the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The description of the patch.
         * </p>
         * 
         * @param description
         *        The description of the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The URL where more information can be obtained about the patch.
         * </p>
         * 
         * @param contentUrl
         *        The URL where more information can be obtained about the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentUrl(String contentUrl);

        /**
         * <p>
         * The name of the vendor providing the patch.
         * </p>
         * 
         * @param vendor
         *        The name of the vendor providing the patch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vendor(String vendor);

        /**
         * <p>
         * The product family the patch is applicable for. For example, <code>Windows</code> or
         * <code>Amazon Linux 2</code>.
         * </p>
         * 
         * @param productFamily
         *        The product family the patch is applicable for. For example, <code>Windows</code> or
         *        <code>Amazon Linux 2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productFamily(String productFamily);

        /**
         * <p>
         * The specific product the patch is applicable for. For example, <code>WindowsServer2016</code> or
         * <code>AmazonLinux2018.03</code>.
         * </p>
         * 
         * @param product
         *        The specific product the patch is applicable for. For example, <code>WindowsServer2016</code> or
         *        <code>AmazonLinux2018.03</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder product(String product);

        /**
         * <p>
         * The classification of the patch. For example, <code>SecurityUpdates</code>, <code>Updates</code>, or
         * <code>CriticalUpdates</code>.
         * </p>
         * 
         * @param classification
         *        The classification of the patch. For example, <code>SecurityUpdates</code>, <code>Updates</code>, or
         *        <code>CriticalUpdates</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classification(String classification);

        /**
         * <p>
         * The severity of the patch, such as <code>Critical</code>, <code>Important</code>, or <code>Moderate</code>.
         * Applies to Windows patches only.
         * </p>
         * 
         * @param msrcSeverity
         *        The severity of the patch, such as <code>Critical</code>, <code>Important</code>, or
         *        <code>Moderate</code>. Applies to Windows patches only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder msrcSeverity(String msrcSeverity);

        /**
         * <p>
         * The Microsoft Knowledge Base ID of the patch. Applies to Windows patches only.
         * </p>
         * 
         * @param kbNumber
         *        The Microsoft Knowledge Base ID of the patch. Applies to Windows patches only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kbNumber(String kbNumber);

        /**
         * <p>
         * The ID of the Microsoft Security Response Center (MSRC) bulletin the patch is related to. For example,
         * <code>MS14-045</code>. Applies to Windows patches only.
         * </p>
         * 
         * @param msrcNumber
         *        The ID of the Microsoft Security Response Center (MSRC) bulletin the patch is related to. For example,
         *        <code>MS14-045</code>. Applies to Windows patches only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder msrcNumber(String msrcNumber);

        /**
         * <p>
         * The language of the patch if it's language-specific.
         * </p>
         * 
         * @param language
         *        The language of the patch if it's language-specific.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder language(String language);

        /**
         * <p>
         * The Advisory ID of the patch. For example, <code>RHSA-2020:3779</code>. Applies to Linux-based managed nodes
         * only.
         * </p>
         * 
         * @param advisoryIds
         *        The Advisory ID of the patch. For example, <code>RHSA-2020:3779</code>. Applies to Linux-based managed
         *        nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advisoryIds(Collection<String> advisoryIds);

        /**
         * <p>
         * The Advisory ID of the patch. For example, <code>RHSA-2020:3779</code>. Applies to Linux-based managed nodes
         * only.
         * </p>
         * 
         * @param advisoryIds
         *        The Advisory ID of the patch. For example, <code>RHSA-2020:3779</code>. Applies to Linux-based managed
         *        nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advisoryIds(String... advisoryIds);

        /**
         * <p>
         * The Bugzilla ID of the patch. For example, <code>1600646</code>. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param bugzillaIds
         *        The Bugzilla ID of the patch. For example, <code>1600646</code>. Applies to Linux-based managed nodes
         *        only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bugzillaIds(Collection<String> bugzillaIds);

        /**
         * <p>
         * The Bugzilla ID of the patch. For example, <code>1600646</code>. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param bugzillaIds
         *        The Bugzilla ID of the patch. For example, <code>1600646</code>. Applies to Linux-based managed nodes
         *        only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bugzillaIds(String... bugzillaIds);

        /**
         * <p>
         * The Common Vulnerabilities and Exposures (CVE) ID of the patch. For example, <code>CVE-2011-3192</code>.
         * Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param cveIds
         *        The Common Vulnerabilities and Exposures (CVE) ID of the patch. For example,
         *        <code>CVE-2011-3192</code>. Applies to Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cveIds(Collection<String> cveIds);

        /**
         * <p>
         * The Common Vulnerabilities and Exposures (CVE) ID of the patch. For example, <code>CVE-2011-3192</code>.
         * Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param cveIds
         *        The Common Vulnerabilities and Exposures (CVE) ID of the patch. For example,
         *        <code>CVE-2011-3192</code>. Applies to Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cveIds(String... cveIds);

        /**
         * <p>
         * The name of the patch. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param name
         *        The name of the patch. Applies to Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The epoch of the patch. For example in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the epoch value
         * is <code>20180914-2</code>. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param epoch
         *        The epoch of the patch. For example in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the
         *        epoch value is <code>20180914-2</code>. Applies to Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder epoch(Integer epoch);

        /**
         * <p>
         * The version number of the patch. For example, in <code>example-pkg-1.710.10-2.7.abcd.x86_64</code>, the
         * version number is indicated by <code>-1</code>. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param version
         *        The version number of the patch. For example, in <code>example-pkg-1.710.10-2.7.abcd.x86_64</code>,
         *        the version number is indicated by <code>-1</code>. Applies to Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The particular release of a patch. For example, in <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the
         * release is <code>2.amaz1</code>. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param release
         *        The particular release of a patch. For example, in
         *        <code>pkg-example-EE-20180914-2.2.amzn1.noarch</code>, the release is <code>2.amaz1</code>. Applies to
         *        Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder release(String release);

        /**
         * <p>
         * The architecture of the patch. For example, in <code>example-pkg-0.710.10-2.7.abcd.x86_64</code>, the
         * architecture is indicated by <code>x86_64</code>. Applies to Linux-based managed nodes only.
         * </p>
         * 
         * @param arch
         *        The architecture of the patch. For example, in <code>example-pkg-0.710.10-2.7.abcd.x86_64</code>, the
         *        architecture is indicated by <code>x86_64</code>. Applies to Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arch(String arch);

        /**
         * <p>
         * The severity level of the patch. For example, <code>CRITICAL</code> or <code>MODERATE</code>.
         * </p>
         * 
         * @param severity
         *        The severity level of the patch. For example, <code>CRITICAL</code> or <code>MODERATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * The source patch repository for the operating system and version, such as <code>trusty-security</code> for
         * Ubuntu Server 14.04 LTE and <code>focal-security</code> for Ubuntu Server 20.04 LTE. Applies to Linux-based
         * managed nodes only.
         * </p>
         * 
         * @param repository
         *        The source patch repository for the operating system and version, such as <code>trusty-security</code>
         *        for Ubuntu Server 14.04 LTE and <code>focal-security</code> for Ubuntu Server 20.04 LTE. Applies to
         *        Linux-based managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repository(String repository);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Instant releaseDate;

        private String title;

        private String description;

        private String contentUrl;

        private String vendor;

        private String productFamily;

        private String product;

        private String classification;

        private String msrcSeverity;

        private String kbNumber;

        private String msrcNumber;

        private String language;

        private List<String> advisoryIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> bugzillaIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> cveIds = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private Integer epoch;

        private String version;

        private String release;

        private String arch;

        private String severity;

        private String repository;

        private BuilderImpl() {
        }

        private BuilderImpl(Patch model) {
            id(model.id);
            releaseDate(model.releaseDate);
            title(model.title);
            description(model.description);
            contentUrl(model.contentUrl);
            vendor(model.vendor);
            productFamily(model.productFamily);
            product(model.product);
            classification(model.classification);
            msrcSeverity(model.msrcSeverity);
            kbNumber(model.kbNumber);
            msrcNumber(model.msrcNumber);
            language(model.language);
            advisoryIds(model.advisoryIds);
            bugzillaIds(model.bugzillaIds);
            cveIds(model.cveIds);
            name(model.name);
            epoch(model.epoch);
            version(model.version);
            release(model.release);
            arch(model.arch);
            severity(model.severity);
            repository(model.repository);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Instant getReleaseDate() {
            return releaseDate;
        }

        public final void setReleaseDate(Instant releaseDate) {
            this.releaseDate = releaseDate;
        }

        @Override
        public final Builder releaseDate(Instant releaseDate) {
            this.releaseDate = releaseDate;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getContentUrl() {
            return contentUrl;
        }

        public final void setContentUrl(String contentUrl) {
            this.contentUrl = contentUrl;
        }

        @Override
        public final Builder contentUrl(String contentUrl) {
            this.contentUrl = contentUrl;
            return this;
        }

        public final String getVendor() {
            return vendor;
        }

        public final void setVendor(String vendor) {
            this.vendor = vendor;
        }

        @Override
        public final Builder vendor(String vendor) {
            this.vendor = vendor;
            return this;
        }

        public final String getProductFamily() {
            return productFamily;
        }

        public final void setProductFamily(String productFamily) {
            this.productFamily = productFamily;
        }

        @Override
        public final Builder productFamily(String productFamily) {
            this.productFamily = productFamily;
            return this;
        }

        public final String getProduct() {
            return product;
        }

        public final void setProduct(String product) {
            this.product = product;
        }

        @Override
        public final Builder product(String product) {
            this.product = product;
            return this;
        }

        public final String getClassification() {
            return classification;
        }

        public final void setClassification(String classification) {
            this.classification = classification;
        }

        @Override
        public final Builder classification(String classification) {
            this.classification = classification;
            return this;
        }

        public final String getMsrcSeverity() {
            return msrcSeverity;
        }

        public final void setMsrcSeverity(String msrcSeverity) {
            this.msrcSeverity = msrcSeverity;
        }

        @Override
        public final Builder msrcSeverity(String msrcSeverity) {
            this.msrcSeverity = msrcSeverity;
            return this;
        }

        public final String getKbNumber() {
            return kbNumber;
        }

        public final void setKbNumber(String kbNumber) {
            this.kbNumber = kbNumber;
        }

        @Override
        public final Builder kbNumber(String kbNumber) {
            this.kbNumber = kbNumber;
            return this;
        }

        public final String getMsrcNumber() {
            return msrcNumber;
        }

        public final void setMsrcNumber(String msrcNumber) {
            this.msrcNumber = msrcNumber;
        }

        @Override
        public final Builder msrcNumber(String msrcNumber) {
            this.msrcNumber = msrcNumber;
            return this;
        }

        public final String getLanguage() {
            return language;
        }

        public final void setLanguage(String language) {
            this.language = language;
        }

        @Override
        public final Builder language(String language) {
            this.language = language;
            return this;
        }

        public final Collection<String> getAdvisoryIds() {
            if (advisoryIds instanceof SdkAutoConstructList) {
                return null;
            }
            return advisoryIds;
        }

        public final void setAdvisoryIds(Collection<String> advisoryIds) {
            this.advisoryIds = PatchAdvisoryIdListCopier.copy(advisoryIds);
        }

        @Override
        public final Builder advisoryIds(Collection<String> advisoryIds) {
            this.advisoryIds = PatchAdvisoryIdListCopier.copy(advisoryIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder advisoryIds(String... advisoryIds) {
            advisoryIds(Arrays.asList(advisoryIds));
            return this;
        }

        public final Collection<String> getBugzillaIds() {
            if (bugzillaIds instanceof SdkAutoConstructList) {
                return null;
            }
            return bugzillaIds;
        }

        public final void setBugzillaIds(Collection<String> bugzillaIds) {
            this.bugzillaIds = PatchBugzillaIdListCopier.copy(bugzillaIds);
        }

        @Override
        public final Builder bugzillaIds(Collection<String> bugzillaIds) {
            this.bugzillaIds = PatchBugzillaIdListCopier.copy(bugzillaIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bugzillaIds(String... bugzillaIds) {
            bugzillaIds(Arrays.asList(bugzillaIds));
            return this;
        }

        public final Collection<String> getCveIds() {
            if (cveIds instanceof SdkAutoConstructList) {
                return null;
            }
            return cveIds;
        }

        public final void setCveIds(Collection<String> cveIds) {
            this.cveIds = PatchCVEIdListCopier.copy(cveIds);
        }

        @Override
        public final Builder cveIds(Collection<String> cveIds) {
            this.cveIds = PatchCVEIdListCopier.copy(cveIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cveIds(String... cveIds) {
            cveIds(Arrays.asList(cveIds));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Integer getEpoch() {
            return epoch;
        }

        public final void setEpoch(Integer epoch) {
            this.epoch = epoch;
        }

        @Override
        public final Builder epoch(Integer epoch) {
            this.epoch = epoch;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getRelease() {
            return release;
        }

        public final void setRelease(String release) {
            this.release = release;
        }

        @Override
        public final Builder release(String release) {
            this.release = release;
            return this;
        }

        public final String getArch() {
            return arch;
        }

        public final void setArch(String arch) {
            this.arch = arch;
        }

        @Override
        public final Builder arch(String arch) {
            this.arch = arch;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final String getRepository() {
            return repository;
        }

        public final void setRepository(String repository) {
            this.repository = repository;
        }

        @Override
        public final Builder repository(String repository) {
            this.repository = repository;
            return this;
        }

        @Override
        public Patch build() {
            return new Patch(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
