/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon Web Services Systems Manager document (SSM document).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentDescription.Builder, DocumentDescription> {
    private static final SdkField<String> SHA1_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Sha1")
            .getter(getter(DocumentDescription::sha1)).setter(setter(Builder::sha1))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sha1").build()).build();

    private static final SdkField<String> HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Hash")
            .getter(getter(DocumentDescription::hash)).setter(setter(Builder::hash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Hash").build()).build();

    private static final SdkField<String> HASH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HashType").getter(getter(DocumentDescription::hashTypeAsString)).setter(setter(Builder::hashType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HashType").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DocumentDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(DocumentDescription::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<String> VERSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionName").getter(getter(DocumentDescription::versionName)).setter(setter(Builder::versionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionName").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Owner")
            .getter(getter(DocumentDescription::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedDate").getter(getter(DocumentDescription::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DocumentDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_INFORMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusInformation").getter(getter(DocumentDescription::statusInformation))
            .setter(setter(Builder::statusInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusInformation").build()).build();

    private static final SdkField<String> DOCUMENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentVersion").getter(getter(DocumentDescription::documentVersion))
            .setter(setter(Builder::documentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentVersion").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(DocumentDescription::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<DocumentParameter>> PARAMETERS_FIELD = SdkField
            .<List<DocumentParameter>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(DocumentDescription::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> PLATFORM_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PlatformTypes")
            .getter(getter(DocumentDescription::platformTypesAsStrings))
            .setter(setter(Builder::platformTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentType").getter(getter(DocumentDescription::documentTypeAsString))
            .setter(setter(Builder::documentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentType").build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SchemaVersion").getter(getter(DocumentDescription::schemaVersion))
            .setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaVersion").build()).build();

    private static final SdkField<String> LATEST_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LatestVersion").getter(getter(DocumentDescription::latestVersion))
            .setter(setter(Builder::latestVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestVersion").build()).build();

    private static final SdkField<String> DEFAULT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultVersion").getter(getter(DocumentDescription::defaultVersion))
            .setter(setter(Builder::defaultVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultVersion").build()).build();

    private static final SdkField<String> DOCUMENT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentFormat").getter(getter(DocumentDescription::documentFormatAsString))
            .setter(setter(Builder::documentFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentFormat").build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetType").getter(getter(DocumentDescription::targetType)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(DocumentDescription::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttachmentInformation>> ATTACHMENTS_INFORMATION_FIELD = SdkField
            .<List<AttachmentInformation>> builder(MarshallingType.LIST)
            .memberName("AttachmentsInformation")
            .getter(getter(DocumentDescription::attachmentsInformation))
            .setter(setter(Builder::attachmentsInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentsInformation").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttachmentInformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttachmentInformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DocumentRequires>> REQUIRES_FIELD = SdkField
            .<List<DocumentRequires>> builder(MarshallingType.LIST)
            .memberName("Requires")
            .getter(getter(DocumentDescription::requires))
            .setter(setter(Builder::requires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Requires").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentRequires> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentRequires::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> AUTHOR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Author")
            .getter(getter(DocumentDescription::author)).setter(setter(Builder::author))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Author").build()).build();

    private static final SdkField<List<ReviewInformation>> REVIEW_INFORMATION_FIELD = SdkField
            .<List<ReviewInformation>> builder(MarshallingType.LIST)
            .memberName("ReviewInformation")
            .getter(getter(DocumentDescription::reviewInformation))
            .setter(setter(Builder::reviewInformation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReviewInformation").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReviewInformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReviewInformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APPROVED_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApprovedVersion").getter(getter(DocumentDescription::approvedVersion))
            .setter(setter(Builder::approvedVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApprovedVersion").build()).build();

    private static final SdkField<String> PENDING_REVIEW_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PendingReviewVersion").getter(getter(DocumentDescription::pendingReviewVersion))
            .setter(setter(Builder::pendingReviewVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PendingReviewVersion").build())
            .build();

    private static final SdkField<String> REVIEW_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReviewStatus").getter(getter(DocumentDescription::reviewStatusAsString))
            .setter(setter(Builder::reviewStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReviewStatus").build()).build();

    private static final SdkField<List<String>> CATEGORY_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Category")
            .getter(getter(DocumentDescription::category))
            .setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Category").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CATEGORY_ENUM_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CategoryEnum")
            .getter(getter(DocumentDescription::categoryEnum))
            .setter(setter(Builder::categoryEnum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CategoryEnum").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SHA1_FIELD, HASH_FIELD,
            HASH_TYPE_FIELD, NAME_FIELD, DISPLAY_NAME_FIELD, VERSION_NAME_FIELD, OWNER_FIELD, CREATED_DATE_FIELD, STATUS_FIELD,
            STATUS_INFORMATION_FIELD, DOCUMENT_VERSION_FIELD, DESCRIPTION_FIELD, PARAMETERS_FIELD, PLATFORM_TYPES_FIELD,
            DOCUMENT_TYPE_FIELD, SCHEMA_VERSION_FIELD, LATEST_VERSION_FIELD, DEFAULT_VERSION_FIELD, DOCUMENT_FORMAT_FIELD,
            TARGET_TYPE_FIELD, TAGS_FIELD, ATTACHMENTS_INFORMATION_FIELD, REQUIRES_FIELD, AUTHOR_FIELD, REVIEW_INFORMATION_FIELD,
            APPROVED_VERSION_FIELD, PENDING_REVIEW_VERSION_FIELD, REVIEW_STATUS_FIELD, CATEGORY_FIELD, CATEGORY_ENUM_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sha1;

    private final String hash;

    private final String hashType;

    private final String name;

    private final String displayName;

    private final String versionName;

    private final String owner;

    private final Instant createdDate;

    private final String status;

    private final String statusInformation;

    private final String documentVersion;

    private final String description;

    private final List<DocumentParameter> parameters;

    private final List<String> platformTypes;

    private final String documentType;

    private final String schemaVersion;

    private final String latestVersion;

    private final String defaultVersion;

    private final String documentFormat;

    private final String targetType;

    private final List<Tag> tags;

    private final List<AttachmentInformation> attachmentsInformation;

    private final List<DocumentRequires> requires;

    private final String author;

    private final List<ReviewInformation> reviewInformation;

    private final String approvedVersion;

    private final String pendingReviewVersion;

    private final String reviewStatus;

    private final List<String> category;

    private final List<String> categoryEnum;

    private DocumentDescription(BuilderImpl builder) {
        this.sha1 = builder.sha1;
        this.hash = builder.hash;
        this.hashType = builder.hashType;
        this.name = builder.name;
        this.displayName = builder.displayName;
        this.versionName = builder.versionName;
        this.owner = builder.owner;
        this.createdDate = builder.createdDate;
        this.status = builder.status;
        this.statusInformation = builder.statusInformation;
        this.documentVersion = builder.documentVersion;
        this.description = builder.description;
        this.parameters = builder.parameters;
        this.platformTypes = builder.platformTypes;
        this.documentType = builder.documentType;
        this.schemaVersion = builder.schemaVersion;
        this.latestVersion = builder.latestVersion;
        this.defaultVersion = builder.defaultVersion;
        this.documentFormat = builder.documentFormat;
        this.targetType = builder.targetType;
        this.tags = builder.tags;
        this.attachmentsInformation = builder.attachmentsInformation;
        this.requires = builder.requires;
        this.author = builder.author;
        this.reviewInformation = builder.reviewInformation;
        this.approvedVersion = builder.approvedVersion;
        this.pendingReviewVersion = builder.pendingReviewVersion;
        this.reviewStatus = builder.reviewStatus;
        this.category = builder.category;
        this.categoryEnum = builder.categoryEnum;
    }

    /**
     * <p>
     * The SHA1 hash of the document, which you can use for verification.
     * </p>
     * 
     * @return The SHA1 hash of the document, which you can use for verification.
     */
    public final String sha1() {
        return sha1;
    }

    /**
     * <p>
     * The Sha256 or Sha1 hash created by the system when the document was created.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * 
     * @return The Sha256 or Sha1 hash created by the system when the document was created. </p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     */
    public final String hash() {
        return hash;
    }

    /**
     * <p>
     * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hashType} will
     * return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hashTypeAsString}.
     * </p>
     * 
     * @return The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     * @see DocumentHashType
     */
    public final DocumentHashType hashType() {
        return DocumentHashType.fromValue(hashType);
    }

    /**
     * <p>
     * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
     * </p>
     * <note>
     * <p>
     * Sha1 hashes have been deprecated.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hashType} will
     * return {@link DocumentHashType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hashTypeAsString}.
     * </p>
     * 
     * @return The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p> <note>
     *         <p>
     *         Sha1 hashes have been deprecated.
     *         </p>
     * @see DocumentHashType
     */
    public final String hashTypeAsString() {
        return hashType;
    }

    /**
     * <p>
     * The name of the SSM document.
     * </p>
     * 
     * @return The name of the SSM document.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The friendly name of the SSM document. This value can differ for each version of the document. If you want to
     * update this value, see <a>UpdateDocument</a>.
     * </p>
     * 
     * @return The friendly name of the SSM document. This value can differ for each version of the document. If you
     *         want to update this value, see <a>UpdateDocument</a>.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The version of the artifact associated with the document.
     * </p>
     * 
     * @return The version of the artifact associated with the document.
     */
    public final String versionName() {
        return versionName;
    }

    /**
     * <p>
     * The Amazon Web Services user that created the document.
     * </p>
     * 
     * @return The Amazon Web Services user that created the document.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * The date when the document was created.
     * </p>
     * 
     * @return The date when the document was created.
     */
    public final Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The status of the SSM document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DocumentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the SSM document.
     * @see DocumentStatus
     */
    public final DocumentStatus status() {
        return DocumentStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the SSM document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DocumentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the SSM document.
     * @see DocumentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A message returned by Amazon Web Services Systems Manager that explains the <code>Status</code> value. For
     * example, a <code>Failed</code> status might be explained by the <code>StatusInformation</code> message,
     * "The specified S3 bucket doesn't exist. Verify that the URL of the S3 bucket is correct."
     * </p>
     * 
     * @return A message returned by Amazon Web Services Systems Manager that explains the <code>Status</code> value.
     *         For example, a <code>Failed</code> status might be explained by the <code>StatusInformation</code>
     *         message, "The specified S3 bucket doesn't exist. Verify that the URL of the S3 bucket is correct."
     */
    public final String statusInformation() {
        return statusInformation;
    }

    /**
     * <p>
     * The document version.
     * </p>
     * 
     * @return The document version.
     */
    public final String documentVersion() {
        return documentVersion;
    }

    /**
     * <p>
     * A description of the document.
     * </p>
     * 
     * @return A description of the document.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A description of the parameters for a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return A description of the parameters for a document.
     */
    public final List<DocumentParameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The list of operating system (OS) platforms compatible with this SSM document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformTypes} method.
     * </p>
     * 
     * @return The list of operating system (OS) platforms compatible with this SSM document.
     */
    public final List<PlatformType> platformTypes() {
        return PlatformTypeListCopier.copyStringToEnum(platformTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the PlatformTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlatformTypes() {
        return platformTypes != null && !(platformTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of operating system (OS) platforms compatible with this SSM document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformTypes} method.
     * </p>
     * 
     * @return The list of operating system (OS) platforms compatible with this SSM document.
     */
    public final List<String> platformTypesAsStrings() {
        return platformTypes;
    }

    /**
     * <p>
     * The type of document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of document.
     * @see DocumentType
     */
    public final DocumentType documentType() {
        return DocumentType.fromValue(documentType);
    }

    /**
     * <p>
     * The type of document.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of document.
     * @see DocumentType
     */
    public final String documentTypeAsString() {
        return documentType;
    }

    /**
     * <p>
     * The schema version.
     * </p>
     * 
     * @return The schema version.
     */
    public final String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The latest version of the document.
     * </p>
     * 
     * @return The latest version of the document.
     */
    public final String latestVersion() {
        return latestVersion;
    }

    /**
     * <p>
     * The default version.
     * </p>
     * 
     * @return The default version.
     */
    public final String defaultVersion() {
        return defaultVersion;
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public final DocumentFormat documentFormat() {
        return DocumentFormat.fromValue(documentFormat);
    }

    /**
     * <p>
     * The document format, either JSON or YAML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentFormat}
     * will return {@link DocumentFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #documentFormatAsString}.
     * </p>
     * 
     * @return The document format, either JSON or YAML.
     * @see DocumentFormat
     */
    public final String documentFormatAsString() {
        return documentFormat;
    }

    /**
     * <p>
     * The target type which defines the kinds of resources the document can run on. For example,
     * <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * 
     * @return The target type which defines the kinds of resources the document can run on. For example,
     *         <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
     */
    public final String targetType() {
        return targetType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags, or metadata, that have been applied to the document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags, or metadata, that have been applied to the document.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttachmentsInformation property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAttachmentsInformation() {
        return attachmentsInformation != null && !(attachmentsInformation instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Details about the document attachments, including names, locations, sizes, and so on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttachmentsInformation} method.
     * </p>
     * 
     * @return Details about the document attachments, including names, locations, sizes, and so on.
     */
    public final List<AttachmentInformation> attachmentsInformation() {
        return attachmentsInformation;
    }

    /**
     * For responses, this returns true if the service returned a value for the Requires property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRequires() {
        return requires != null && !(requires instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code> document
     * requires an <code>ApplicationConfigurationSchema</code> document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequires} method.
     * </p>
     * 
     * @return A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
     *         document requires an <code>ApplicationConfigurationSchema</code> document.
     */
    public final List<DocumentRequires> requires() {
        return requires;
    }

    /**
     * <p>
     * The user in your organization who created the document.
     * </p>
     * 
     * @return The user in your organization who created the document.
     */
    public final String author() {
        return author;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReviewInformation property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReviewInformation() {
        return reviewInformation != null && !(reviewInformation instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Details about the review of a document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReviewInformation} method.
     * </p>
     * 
     * @return Details about the review of a document.
     */
    public final List<ReviewInformation> reviewInformation() {
        return reviewInformation;
    }

    /**
     * <p>
     * The version of the document currently approved for use in the organization.
     * </p>
     * 
     * @return The version of the document currently approved for use in the organization.
     */
    public final String approvedVersion() {
        return approvedVersion;
    }

    /**
     * <p>
     * The version of the document that is currently under review.
     * </p>
     * 
     * @return The version of the document that is currently under review.
     */
    public final String pendingReviewVersion() {
        return pendingReviewVersion;
    }

    /**
     * <p>
     * The current status of the review.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reviewStatus} will
     * return {@link ReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reviewStatusAsString}.
     * </p>
     * 
     * @return The current status of the review.
     * @see ReviewStatus
     */
    public final ReviewStatus reviewStatus() {
        return ReviewStatus.fromValue(reviewStatus);
    }

    /**
     * <p>
     * The current status of the review.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reviewStatus} will
     * return {@link ReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reviewStatusAsString}.
     * </p>
     * 
     * @return The current status of the review.
     * @see ReviewStatus
     */
    public final String reviewStatusAsString() {
        return reviewStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the Category property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCategory() {
        return category != null && !(category instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The classification of a document to help you identify and categorize its use.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCategory} method.
     * </p>
     * 
     * @return The classification of a document to help you identify and categorize its use.
     */
    public final List<String> category() {
        return category;
    }

    /**
     * For responses, this returns true if the service returned a value for the CategoryEnum property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCategoryEnum() {
        return categoryEnum != null && !(categoryEnum instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The value that identifies a document's category.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCategoryEnum} method.
     * </p>
     * 
     * @return The value that identifies a document's category.
     */
    public final List<String> categoryEnum() {
        return categoryEnum;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sha1());
        hashCode = 31 * hashCode + Objects.hashCode(hash());
        hashCode = 31 * hashCode + Objects.hashCode(hashTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(versionName());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusInformation());
        hashCode = 31 * hashCode + Objects.hashCode(documentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatformTypes() ? platformTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(latestVersion());
        hashCode = 31 * hashCode + Objects.hashCode(defaultVersion());
        hashCode = 31 * hashCode + Objects.hashCode(documentFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetType());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachmentsInformation() ? attachmentsInformation() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRequires() ? requires() : null);
        hashCode = 31 * hashCode + Objects.hashCode(author());
        hashCode = 31 * hashCode + Objects.hashCode(hasReviewInformation() ? reviewInformation() : null);
        hashCode = 31 * hashCode + Objects.hashCode(approvedVersion());
        hashCode = 31 * hashCode + Objects.hashCode(pendingReviewVersion());
        hashCode = 31 * hashCode + Objects.hashCode(reviewStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCategory() ? category() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCategoryEnum() ? categoryEnum() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentDescription)) {
            return false;
        }
        DocumentDescription other = (DocumentDescription) obj;
        return Objects.equals(sha1(), other.sha1()) && Objects.equals(hash(), other.hash())
                && Objects.equals(hashTypeAsString(), other.hashTypeAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(versionName(), other.versionName())
                && Objects.equals(owner(), other.owner()) && Objects.equals(createdDate(), other.createdDate())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusInformation(), other.statusInformation())
                && Objects.equals(documentVersion(), other.documentVersion())
                && Objects.equals(description(), other.description()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && hasPlatformTypes() == other.hasPlatformTypes()
                && Objects.equals(platformTypesAsStrings(), other.platformTypesAsStrings())
                && Objects.equals(documentTypeAsString(), other.documentTypeAsString())
                && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(latestVersion(), other.latestVersion())
                && Objects.equals(defaultVersion(), other.defaultVersion())
                && Objects.equals(documentFormatAsString(), other.documentFormatAsString())
                && Objects.equals(targetType(), other.targetType()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasAttachmentsInformation() == other.hasAttachmentsInformation()
                && Objects.equals(attachmentsInformation(), other.attachmentsInformation())
                && hasRequires() == other.hasRequires() && Objects.equals(requires(), other.requires())
                && Objects.equals(author(), other.author()) && hasReviewInformation() == other.hasReviewInformation()
                && Objects.equals(reviewInformation(), other.reviewInformation())
                && Objects.equals(approvedVersion(), other.approvedVersion())
                && Objects.equals(pendingReviewVersion(), other.pendingReviewVersion())
                && Objects.equals(reviewStatusAsString(), other.reviewStatusAsString()) && hasCategory() == other.hasCategory()
                && Objects.equals(category(), other.category()) && hasCategoryEnum() == other.hasCategoryEnum()
                && Objects.equals(categoryEnum(), other.categoryEnum());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentDescription").add("Sha1", sha1()).add("Hash", hash())
                .add("HashType", hashTypeAsString()).add("Name", name()).add("DisplayName", displayName())
                .add("VersionName", versionName()).add("Owner", owner()).add("CreatedDate", createdDate())
                .add("Status", statusAsString()).add("StatusInformation", statusInformation())
                .add("DocumentVersion", documentVersion()).add("Description", description())
                .add("Parameters", hasParameters() ? parameters() : null)
                .add("PlatformTypes", hasPlatformTypes() ? platformTypesAsStrings() : null)
                .add("DocumentType", documentTypeAsString()).add("SchemaVersion", schemaVersion())
                .add("LatestVersion", latestVersion()).add("DefaultVersion", defaultVersion())
                .add("DocumentFormat", documentFormatAsString()).add("TargetType", targetType())
                .add("Tags", hasTags() ? tags() : null)
                .add("AttachmentsInformation", hasAttachmentsInformation() ? attachmentsInformation() : null)
                .add("Requires", hasRequires() ? requires() : null).add("Author", author())
                .add("ReviewInformation", hasReviewInformation() ? reviewInformation() : null)
                .add("ApprovedVersion", approvedVersion()).add("PendingReviewVersion", pendingReviewVersion())
                .add("ReviewStatus", reviewStatusAsString()).add("Category", hasCategory() ? category() : null)
                .add("CategoryEnum", hasCategoryEnum() ? categoryEnum() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Sha1":
            return Optional.ofNullable(clazz.cast(sha1()));
        case "Hash":
            return Optional.ofNullable(clazz.cast(hash()));
        case "HashType":
            return Optional.ofNullable(clazz.cast(hashTypeAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "VersionName":
            return Optional.ofNullable(clazz.cast(versionName()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusInformation":
            return Optional.ofNullable(clazz.cast(statusInformation()));
        case "DocumentVersion":
            return Optional.ofNullable(clazz.cast(documentVersion()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "PlatformTypes":
            return Optional.ofNullable(clazz.cast(platformTypesAsStrings()));
        case "DocumentType":
            return Optional.ofNullable(clazz.cast(documentTypeAsString()));
        case "SchemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "LatestVersion":
            return Optional.ofNullable(clazz.cast(latestVersion()));
        case "DefaultVersion":
            return Optional.ofNullable(clazz.cast(defaultVersion()));
        case "DocumentFormat":
            return Optional.ofNullable(clazz.cast(documentFormatAsString()));
        case "TargetType":
            return Optional.ofNullable(clazz.cast(targetType()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "AttachmentsInformation":
            return Optional.ofNullable(clazz.cast(attachmentsInformation()));
        case "Requires":
            return Optional.ofNullable(clazz.cast(requires()));
        case "Author":
            return Optional.ofNullable(clazz.cast(author()));
        case "ReviewInformation":
            return Optional.ofNullable(clazz.cast(reviewInformation()));
        case "ApprovedVersion":
            return Optional.ofNullable(clazz.cast(approvedVersion()));
        case "PendingReviewVersion":
            return Optional.ofNullable(clazz.cast(pendingReviewVersion()));
        case "ReviewStatus":
            return Optional.ofNullable(clazz.cast(reviewStatusAsString()));
        case "Category":
            return Optional.ofNullable(clazz.cast(category()));
        case "CategoryEnum":
            return Optional.ofNullable(clazz.cast(categoryEnum()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Sha1", SHA1_FIELD);
        map.put("Hash", HASH_FIELD);
        map.put("HashType", HASH_TYPE_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("DisplayName", DISPLAY_NAME_FIELD);
        map.put("VersionName", VERSION_NAME_FIELD);
        map.put("Owner", OWNER_FIELD);
        map.put("CreatedDate", CREATED_DATE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusInformation", STATUS_INFORMATION_FIELD);
        map.put("DocumentVersion", DOCUMENT_VERSION_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Parameters", PARAMETERS_FIELD);
        map.put("PlatformTypes", PLATFORM_TYPES_FIELD);
        map.put("DocumentType", DOCUMENT_TYPE_FIELD);
        map.put("SchemaVersion", SCHEMA_VERSION_FIELD);
        map.put("LatestVersion", LATEST_VERSION_FIELD);
        map.put("DefaultVersion", DEFAULT_VERSION_FIELD);
        map.put("DocumentFormat", DOCUMENT_FORMAT_FIELD);
        map.put("TargetType", TARGET_TYPE_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("AttachmentsInformation", ATTACHMENTS_INFORMATION_FIELD);
        map.put("Requires", REQUIRES_FIELD);
        map.put("Author", AUTHOR_FIELD);
        map.put("ReviewInformation", REVIEW_INFORMATION_FIELD);
        map.put("ApprovedVersion", APPROVED_VERSION_FIELD);
        map.put("PendingReviewVersion", PENDING_REVIEW_VERSION_FIELD);
        map.put("ReviewStatus", REVIEW_STATUS_FIELD);
        map.put("Category", CATEGORY_FIELD);
        map.put("CategoryEnum", CATEGORY_ENUM_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DocumentDescription, T> g) {
        return obj -> g.apply((DocumentDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentDescription> {
        /**
         * <p>
         * The SHA1 hash of the document, which you can use for verification.
         * </p>
         * 
         * @param sha1
         *        The SHA1 hash of the document, which you can use for verification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sha1(String sha1);

        /**
         * <p>
         * The Sha256 or Sha1 hash created by the system when the document was created.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param hash
         *        The Sha256 or Sha1 hash created by the system when the document was created. </p> <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hash(String hash);

        /**
         * <p>
         * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param hashType
         *        The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p>
         *        <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder hashType(String hashType);

        /**
         * <p>
         * The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.
         * </p>
         * <note>
         * <p>
         * Sha1 hashes have been deprecated.
         * </p>
         * </note>
         * 
         * @param hashType
         *        The hash type of the document. Valid values include <code>Sha256</code> or <code>Sha1</code>.</p>
         *        <note>
         *        <p>
         *        Sha1 hashes have been deprecated.
         *        </p>
         * @see DocumentHashType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentHashType
         */
        Builder hashType(DocumentHashType hashType);

        /**
         * <p>
         * The name of the SSM document.
         * </p>
         * 
         * @param name
         *        The name of the SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The friendly name of the SSM document. This value can differ for each version of the document. If you want to
         * update this value, see <a>UpdateDocument</a>.
         * </p>
         * 
         * @param displayName
         *        The friendly name of the SSM document. This value can differ for each version of the document. If you
         *        want to update this value, see <a>UpdateDocument</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The version of the artifact associated with the document.
         * </p>
         * 
         * @param versionName
         *        The version of the artifact associated with the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionName(String versionName);

        /**
         * <p>
         * The Amazon Web Services user that created the document.
         * </p>
         * 
         * @param owner
         *        The Amazon Web Services user that created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The date when the document was created.
         * </p>
         * 
         * @param createdDate
         *        The date when the document was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The status of the SSM document.
         * </p>
         * 
         * @param status
         *        The status of the SSM document.
         * @see DocumentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the SSM document.
         * </p>
         * 
         * @param status
         *        The status of the SSM document.
         * @see DocumentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentStatus
         */
        Builder status(DocumentStatus status);

        /**
         * <p>
         * A message returned by Amazon Web Services Systems Manager that explains the <code>Status</code> value. For
         * example, a <code>Failed</code> status might be explained by the <code>StatusInformation</code> message,
         * "The specified S3 bucket doesn't exist. Verify that the URL of the S3 bucket is correct."
         * </p>
         * 
         * @param statusInformation
         *        A message returned by Amazon Web Services Systems Manager that explains the <code>Status</code> value.
         *        For example, a <code>Failed</code> status might be explained by the <code>StatusInformation</code>
         *        message, "The specified S3 bucket doesn't exist. Verify that the URL of the S3 bucket is correct."
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusInformation(String statusInformation);

        /**
         * <p>
         * The document version.
         * </p>
         * 
         * @param documentVersion
         *        The document version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentVersion(String documentVersion);

        /**
         * <p>
         * A description of the document.
         * </p>
         * 
         * @param description
         *        A description of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<DocumentParameter> parameters);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * 
         * @param parameters
         *        A description of the parameters for a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(DocumentParameter... parameters);

        /**
         * <p>
         * A description of the parameters for a document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.DocumentParameter.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.DocumentParameter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.DocumentParameter.Builder#build()} is called immediately and
         * its result is passed to {@link #parameters(List<DocumentParameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.DocumentParameter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(java.util.Collection<DocumentParameter>)
         */
        Builder parameters(Consumer<DocumentParameter.Builder>... parameters);

        /**
         * <p>
         * The list of operating system (OS) platforms compatible with this SSM document.
         * </p>
         * 
         * @param platformTypes
         *        The list of operating system (OS) platforms compatible with this SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(Collection<String> platformTypes);

        /**
         * <p>
         * The list of operating system (OS) platforms compatible with this SSM document.
         * </p>
         * 
         * @param platformTypes
         *        The list of operating system (OS) platforms compatible with this SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypesWithStrings(String... platformTypes);

        /**
         * <p>
         * The list of operating system (OS) platforms compatible with this SSM document.
         * </p>
         * 
         * @param platformTypes
         *        The list of operating system (OS) platforms compatible with this SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(Collection<PlatformType> platformTypes);

        /**
         * <p>
         * The list of operating system (OS) platforms compatible with this SSM document.
         * </p>
         * 
         * @param platformTypes
         *        The list of operating system (OS) platforms compatible with this SSM document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformTypes(PlatformType... platformTypes);

        /**
         * <p>
         * The type of document.
         * </p>
         * 
         * @param documentType
         *        The type of document.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The type of document.
         * </p>
         * 
         * @param documentType
         *        The type of document.
         * @see DocumentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentType
         */
        Builder documentType(DocumentType documentType);

        /**
         * <p>
         * The schema version.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The latest version of the document.
         * </p>
         * 
         * @param latestVersion
         *        The latest version of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestVersion(String latestVersion);

        /**
         * <p>
         * The default version.
         * </p>
         * 
         * @param defaultVersion
         *        The default version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultVersion(String defaultVersion);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(String documentFormat);

        /**
         * <p>
         * The document format, either JSON or YAML.
         * </p>
         * 
         * @param documentFormat
         *        The document format, either JSON or YAML.
         * @see DocumentFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentFormat
         */
        Builder documentFormat(DocumentFormat documentFormat);

        /**
         * <p>
         * The target type which defines the kinds of resources the document can run on. For example,
         * <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * 
         * @param targetType
         *        The target type which defines the kinds of resources the document can run on. For example,
         *        <code>/AWS::EC2::Instance</code>. For a list of valid resource types, see <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services resource and property types reference</a> in the <i>CloudFormation User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * 
         * @param tags
         *        The tags, or metadata, that have been applied to the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags, or metadata, that have been applied to the document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ssm.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ssm.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ssm.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Details about the document attachments, including names, locations, sizes, and so on.
         * </p>
         * 
         * @param attachmentsInformation
         *        Details about the document attachments, including names, locations, sizes, and so on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentsInformation(Collection<AttachmentInformation> attachmentsInformation);

        /**
         * <p>
         * Details about the document attachments, including names, locations, sizes, and so on.
         * </p>
         * 
         * @param attachmentsInformation
         *        Details about the document attachments, including names, locations, sizes, and so on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentsInformation(AttachmentInformation... attachmentsInformation);

        /**
         * <p>
         * Details about the document attachments, including names, locations, sizes, and so on.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.AttachmentInformation.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ssm.model.AttachmentInformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.AttachmentInformation.Builder#build()} is called immediately
         * and its result is passed to {@link #attachmentsInformation(List<AttachmentInformation>)}.
         * 
         * @param attachmentsInformation
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.AttachmentInformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachmentsInformation(java.util.Collection<AttachmentInformation>)
         */
        Builder attachmentsInformation(Consumer<AttachmentInformation.Builder>... attachmentsInformation);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         *        document requires an <code>ApplicationConfigurationSchema</code> document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(Collection<DocumentRequires> requires);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * 
         * @param requires
         *        A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         *        document requires an <code>ApplicationConfigurationSchema</code> document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requires(DocumentRequires... requires);

        /**
         * <p>
         * A list of SSM documents required by a document. For example, an <code>ApplicationConfiguration</code>
         * document requires an <code>ApplicationConfigurationSchema</code> document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.DocumentRequires#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder#build()} is called immediately and
         * its result is passed to {@link #requires(List<DocumentRequires>)}.
         * 
         * @param requires
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.DocumentRequires.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requires(java.util.Collection<DocumentRequires>)
         */
        Builder requires(Consumer<DocumentRequires.Builder>... requires);

        /**
         * <p>
         * The user in your organization who created the document.
         * </p>
         * 
         * @param author
         *        The user in your organization who created the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder author(String author);

        /**
         * <p>
         * Details about the review of a document.
         * </p>
         * 
         * @param reviewInformation
         *        Details about the review of a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reviewInformation(Collection<ReviewInformation> reviewInformation);

        /**
         * <p>
         * Details about the review of a document.
         * </p>
         * 
         * @param reviewInformation
         *        Details about the review of a document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reviewInformation(ReviewInformation... reviewInformation);

        /**
         * <p>
         * Details about the review of a document.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.ReviewInformation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.ReviewInformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.ReviewInformation.Builder#build()} is called immediately and
         * its result is passed to {@link #reviewInformation(List<ReviewInformation>)}.
         * 
         * @param reviewInformation
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.ReviewInformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #reviewInformation(java.util.Collection<ReviewInformation>)
         */
        Builder reviewInformation(Consumer<ReviewInformation.Builder>... reviewInformation);

        /**
         * <p>
         * The version of the document currently approved for use in the organization.
         * </p>
         * 
         * @param approvedVersion
         *        The version of the document currently approved for use in the organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvedVersion(String approvedVersion);

        /**
         * <p>
         * The version of the document that is currently under review.
         * </p>
         * 
         * @param pendingReviewVersion
         *        The version of the document that is currently under review.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingReviewVersion(String pendingReviewVersion);

        /**
         * <p>
         * The current status of the review.
         * </p>
         * 
         * @param reviewStatus
         *        The current status of the review.
         * @see ReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReviewStatus
         */
        Builder reviewStatus(String reviewStatus);

        /**
         * <p>
         * The current status of the review.
         * </p>
         * 
         * @param reviewStatus
         *        The current status of the review.
         * @see ReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReviewStatus
         */
        Builder reviewStatus(ReviewStatus reviewStatus);

        /**
         * <p>
         * The classification of a document to help you identify and categorize its use.
         * </p>
         * 
         * @param category
         *        The classification of a document to help you identify and categorize its use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(Collection<String> category);

        /**
         * <p>
         * The classification of a document to help you identify and categorize its use.
         * </p>
         * 
         * @param category
         *        The classification of a document to help you identify and categorize its use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(String... category);

        /**
         * <p>
         * The value that identifies a document's category.
         * </p>
         * 
         * @param categoryEnum
         *        The value that identifies a document's category.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categoryEnum(Collection<String> categoryEnum);

        /**
         * <p>
         * The value that identifies a document's category.
         * </p>
         * 
         * @param categoryEnum
         *        The value that identifies a document's category.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categoryEnum(String... categoryEnum);
    }

    static final class BuilderImpl implements Builder {
        private String sha1;

        private String hash;

        private String hashType;

        private String name;

        private String displayName;

        private String versionName;

        private String owner;

        private Instant createdDate;

        private String status;

        private String statusInformation;

        private String documentVersion;

        private String description;

        private List<DocumentParameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private List<String> platformTypes = DefaultSdkAutoConstructList.getInstance();

        private String documentType;

        private String schemaVersion;

        private String latestVersion;

        private String defaultVersion;

        private String documentFormat;

        private String targetType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<AttachmentInformation> attachmentsInformation = DefaultSdkAutoConstructList.getInstance();

        private List<DocumentRequires> requires = DefaultSdkAutoConstructList.getInstance();

        private String author;

        private List<ReviewInformation> reviewInformation = DefaultSdkAutoConstructList.getInstance();

        private String approvedVersion;

        private String pendingReviewVersion;

        private String reviewStatus;

        private List<String> category = DefaultSdkAutoConstructList.getInstance();

        private List<String> categoryEnum = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentDescription model) {
            sha1(model.sha1);
            hash(model.hash);
            hashType(model.hashType);
            name(model.name);
            displayName(model.displayName);
            versionName(model.versionName);
            owner(model.owner);
            createdDate(model.createdDate);
            status(model.status);
            statusInformation(model.statusInformation);
            documentVersion(model.documentVersion);
            description(model.description);
            parameters(model.parameters);
            platformTypesWithStrings(model.platformTypes);
            documentType(model.documentType);
            schemaVersion(model.schemaVersion);
            latestVersion(model.latestVersion);
            defaultVersion(model.defaultVersion);
            documentFormat(model.documentFormat);
            targetType(model.targetType);
            tags(model.tags);
            attachmentsInformation(model.attachmentsInformation);
            requires(model.requires);
            author(model.author);
            reviewInformation(model.reviewInformation);
            approvedVersion(model.approvedVersion);
            pendingReviewVersion(model.pendingReviewVersion);
            reviewStatus(model.reviewStatus);
            category(model.category);
            categoryEnum(model.categoryEnum);
        }

        public final String getSha1() {
            return sha1;
        }

        public final void setSha1(String sha1) {
            this.sha1 = sha1;
        }

        @Override
        public final Builder sha1(String sha1) {
            this.sha1 = sha1;
            return this;
        }

        public final String getHash() {
            return hash;
        }

        public final void setHash(String hash) {
            this.hash = hash;
        }

        @Override
        public final Builder hash(String hash) {
            this.hash = hash;
            return this;
        }

        public final String getHashType() {
            return hashType;
        }

        public final void setHashType(String hashType) {
            this.hashType = hashType;
        }

        @Override
        public final Builder hashType(String hashType) {
            this.hashType = hashType;
            return this;
        }

        @Override
        public final Builder hashType(DocumentHashType hashType) {
            this.hashType(hashType == null ? null : hashType.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getVersionName() {
            return versionName;
        }

        public final void setVersionName(String versionName) {
            this.versionName = versionName;
        }

        @Override
        public final Builder versionName(String versionName) {
            this.versionName = versionName;
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DocumentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusInformation() {
            return statusInformation;
        }

        public final void setStatusInformation(String statusInformation) {
            this.statusInformation = statusInformation;
        }

        @Override
        public final Builder statusInformation(String statusInformation) {
            this.statusInformation = statusInformation;
            return this;
        }

        public final String getDocumentVersion() {
            return documentVersion;
        }

        public final void setDocumentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
        }

        @Override
        public final Builder documentVersion(String documentVersion) {
            this.documentVersion = documentVersion;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<DocumentParameter.Builder> getParameters() {
            List<DocumentParameter.Builder> result = DocumentParameterListCopier.copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParameters(Collection<DocumentParameter.BuilderImpl> parameters) {
            this.parameters = DocumentParameterListCopier.copyFromBuilder(parameters);
        }

        @Override
        public final Builder parameters(Collection<DocumentParameter> parameters) {
            this.parameters = DocumentParameterListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(DocumentParameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<DocumentParameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> DocumentParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getPlatformTypes() {
            if (platformTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return platformTypes;
        }

        public final void setPlatformTypes(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
        }

        @Override
        public final Builder platformTypesWithStrings(Collection<String> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copy(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypesWithStrings(String... platformTypes) {
            platformTypesWithStrings(Arrays.asList(platformTypes));
            return this;
        }

        @Override
        public final Builder platformTypes(Collection<PlatformType> platformTypes) {
            this.platformTypes = PlatformTypeListCopier.copyEnumToString(platformTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformTypes(PlatformType... platformTypes) {
            platformTypes(Arrays.asList(platformTypes));
            return this;
        }

        public final String getDocumentType() {
            return documentType;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentType documentType) {
            this.documentType(documentType == null ? null : documentType.toString());
            return this;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final String getLatestVersion() {
            return latestVersion;
        }

        public final void setLatestVersion(String latestVersion) {
            this.latestVersion = latestVersion;
        }

        @Override
        public final Builder latestVersion(String latestVersion) {
            this.latestVersion = latestVersion;
            return this;
        }

        public final String getDefaultVersion() {
            return defaultVersion;
        }

        public final void setDefaultVersion(String defaultVersion) {
            this.defaultVersion = defaultVersion;
        }

        @Override
        public final Builder defaultVersion(String defaultVersion) {
            this.defaultVersion = defaultVersion;
            return this;
        }

        public final String getDocumentFormat() {
            return documentFormat;
        }

        public final void setDocumentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
        }

        @Override
        public final Builder documentFormat(String documentFormat) {
            this.documentFormat = documentFormat;
            return this;
        }

        @Override
        public final Builder documentFormat(DocumentFormat documentFormat) {
            this.documentFormat(documentFormat == null ? null : documentFormat.toString());
            return this;
        }

        public final String getTargetType() {
            return targetType;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        @Override
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<AttachmentInformation.Builder> getAttachmentsInformation() {
            List<AttachmentInformation.Builder> result = AttachmentInformationListCopier
                    .copyToBuilder(this.attachmentsInformation);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttachmentsInformation(Collection<AttachmentInformation.BuilderImpl> attachmentsInformation) {
            this.attachmentsInformation = AttachmentInformationListCopier.copyFromBuilder(attachmentsInformation);
        }

        @Override
        public final Builder attachmentsInformation(Collection<AttachmentInformation> attachmentsInformation) {
            this.attachmentsInformation = AttachmentInformationListCopier.copy(attachmentsInformation);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentsInformation(AttachmentInformation... attachmentsInformation) {
            attachmentsInformation(Arrays.asList(attachmentsInformation));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentsInformation(Consumer<AttachmentInformation.Builder>... attachmentsInformation) {
            attachmentsInformation(Stream.of(attachmentsInformation)
                    .map(c -> AttachmentInformation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DocumentRequires.Builder> getRequires() {
            List<DocumentRequires.Builder> result = DocumentRequiresListCopier.copyToBuilder(this.requires);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRequires(Collection<DocumentRequires.BuilderImpl> requires) {
            this.requires = DocumentRequiresListCopier.copyFromBuilder(requires);
        }

        @Override
        public final Builder requires(Collection<DocumentRequires> requires) {
            this.requires = DocumentRequiresListCopier.copy(requires);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(DocumentRequires... requires) {
            requires(Arrays.asList(requires));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requires(Consumer<DocumentRequires.Builder>... requires) {
            requires(Stream.of(requires).map(c -> DocumentRequires.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getAuthor() {
            return author;
        }

        public final void setAuthor(String author) {
            this.author = author;
        }

        @Override
        public final Builder author(String author) {
            this.author = author;
            return this;
        }

        public final List<ReviewInformation.Builder> getReviewInformation() {
            List<ReviewInformation.Builder> result = ReviewInformationListCopier.copyToBuilder(this.reviewInformation);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setReviewInformation(Collection<ReviewInformation.BuilderImpl> reviewInformation) {
            this.reviewInformation = ReviewInformationListCopier.copyFromBuilder(reviewInformation);
        }

        @Override
        public final Builder reviewInformation(Collection<ReviewInformation> reviewInformation) {
            this.reviewInformation = ReviewInformationListCopier.copy(reviewInformation);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reviewInformation(ReviewInformation... reviewInformation) {
            reviewInformation(Arrays.asList(reviewInformation));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reviewInformation(Consumer<ReviewInformation.Builder>... reviewInformation) {
            reviewInformation(Stream.of(reviewInformation).map(c -> ReviewInformation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getApprovedVersion() {
            return approvedVersion;
        }

        public final void setApprovedVersion(String approvedVersion) {
            this.approvedVersion = approvedVersion;
        }

        @Override
        public final Builder approvedVersion(String approvedVersion) {
            this.approvedVersion = approvedVersion;
            return this;
        }

        public final String getPendingReviewVersion() {
            return pendingReviewVersion;
        }

        public final void setPendingReviewVersion(String pendingReviewVersion) {
            this.pendingReviewVersion = pendingReviewVersion;
        }

        @Override
        public final Builder pendingReviewVersion(String pendingReviewVersion) {
            this.pendingReviewVersion = pendingReviewVersion;
            return this;
        }

        public final String getReviewStatus() {
            return reviewStatus;
        }

        public final void setReviewStatus(String reviewStatus) {
            this.reviewStatus = reviewStatus;
        }

        @Override
        public final Builder reviewStatus(String reviewStatus) {
            this.reviewStatus = reviewStatus;
            return this;
        }

        @Override
        public final Builder reviewStatus(ReviewStatus reviewStatus) {
            this.reviewStatus(reviewStatus == null ? null : reviewStatus.toString());
            return this;
        }

        public final Collection<String> getCategory() {
            if (category instanceof SdkAutoConstructList) {
                return null;
            }
            return category;
        }

        public final void setCategory(Collection<String> category) {
            this.category = CategoryListCopier.copy(category);
        }

        @Override
        public final Builder category(Collection<String> category) {
            this.category = CategoryListCopier.copy(category);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder category(String... category) {
            category(Arrays.asList(category));
            return this;
        }

        public final Collection<String> getCategoryEnum() {
            if (categoryEnum instanceof SdkAutoConstructList) {
                return null;
            }
            return categoryEnum;
        }

        public final void setCategoryEnum(Collection<String> categoryEnum) {
            this.categoryEnum = CategoryEnumListCopier.copy(categoryEnum);
        }

        @Override
        public final Builder categoryEnum(Collection<String> categoryEnum) {
            this.categoryEnum = CategoryEnumListCopier.copy(categoryEnum);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder categoryEnum(String... categoryEnum) {
            categoryEnum(Arrays.asList(categoryEnum));
            return this;
        }

        @Override
        public DocumentDescription build() {
            return new DocumentDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
