/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines an approval rule for a patch baseline.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PatchRule implements SdkPojo, Serializable, ToCopyableBuilder<PatchRule.Builder, PatchRule> {
    private static final SdkField<PatchFilterGroup> PATCH_FILTER_GROUP_FIELD = SdkField
            .<PatchFilterGroup> builder(MarshallingType.SDK_POJO).memberName("PatchFilterGroup")
            .getter(getter(PatchRule::patchFilterGroup)).setter(setter(Builder::patchFilterGroup))
            .constructor(PatchFilterGroup::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PatchFilterGroup").build()).build();

    private static final SdkField<String> COMPLIANCE_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComplianceLevel").getter(getter(PatchRule::complianceLevelAsString))
            .setter(setter(Builder::complianceLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComplianceLevel").build()).build();

    private static final SdkField<Integer> APPROVE_AFTER_DAYS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ApproveAfterDays").getter(getter(PatchRule::approveAfterDays)).setter(setter(Builder::approveAfterDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApproveAfterDays").build()).build();

    private static final SdkField<String> APPROVE_UNTIL_DATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApproveUntilDate").getter(getter(PatchRule::approveUntilDate)).setter(setter(Builder::approveUntilDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApproveUntilDate").build()).build();

    private static final SdkField<Boolean> ENABLE_NON_SECURITY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableNonSecurity").getter(getter(PatchRule::enableNonSecurity))
            .setter(setter(Builder::enableNonSecurity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableNonSecurity").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATCH_FILTER_GROUP_FIELD,
            COMPLIANCE_LEVEL_FIELD, APPROVE_AFTER_DAYS_FIELD, APPROVE_UNTIL_DATE_FIELD, ENABLE_NON_SECURITY_FIELD));

    private static final long serialVersionUID = 1L;

    private final PatchFilterGroup patchFilterGroup;

    private final String complianceLevel;

    private final Integer approveAfterDays;

    private final String approveUntilDate;

    private final Boolean enableNonSecurity;

    private PatchRule(BuilderImpl builder) {
        this.patchFilterGroup = builder.patchFilterGroup;
        this.complianceLevel = builder.complianceLevel;
        this.approveAfterDays = builder.approveAfterDays;
        this.approveUntilDate = builder.approveUntilDate;
        this.enableNonSecurity = builder.enableNonSecurity;
    }

    /**
     * <p>
     * The patch filter group that defines the criteria for the rule.
     * </p>
     * 
     * @return The patch filter group that defines the criteria for the rule.
     */
    public final PatchFilterGroup patchFilterGroup() {
        return patchFilterGroup;
    }

    /**
     * <p>
     * A compliance severity level for all approved patches in a patch baseline.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceLevel}
     * will return {@link PatchComplianceLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #complianceLevelAsString}.
     * </p>
     * 
     * @return A compliance severity level for all approved patches in a patch baseline.
     * @see PatchComplianceLevel
     */
    public final PatchComplianceLevel complianceLevel() {
        return PatchComplianceLevel.fromValue(complianceLevel);
    }

    /**
     * <p>
     * A compliance severity level for all approved patches in a patch baseline.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceLevel}
     * will return {@link PatchComplianceLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #complianceLevelAsString}.
     * </p>
     * 
     * @return A compliance severity level for all approved patches in a patch baseline.
     * @see PatchComplianceLevel
     */
    public final String complianceLevelAsString() {
        return complianceLevel;
    }

    /**
     * <p>
     * The number of days after the release date of each patch matched by the rule that the patch is marked as approved
     * in the patch baseline. For example, a value of <code>7</code> means that patches are approved seven days after
     * they are released. Not supported on Debian Server or Ubuntu Server.
     * </p>
     * 
     * @return The number of days after the release date of each patch matched by the rule that the patch is marked as
     *         approved in the patch baseline. For example, a value of <code>7</code> means that patches are approved
     *         seven days after they are released. Not supported on Debian Server or Ubuntu Server.
     */
    public final Integer approveAfterDays() {
        return approveAfterDays;
    }

    /**
     * <p>
     * The cutoff date for auto approval of released patches. Any patches released on or before this date are installed
     * automatically. Not supported on Debian Server or Ubuntu Server.
     * </p>
     * <p>
     * Enter dates in the format <code>YYYY-MM-DD</code>. For example, <code>2021-12-31</code>.
     * </p>
     * 
     * @return The cutoff date for auto approval of released patches. Any patches released on or before this date are
     *         installed automatically. Not supported on Debian Server or Ubuntu Server.</p>
     *         <p>
     *         Enter dates in the format <code>YYYY-MM-DD</code>. For example, <code>2021-12-31</code>.
     */
    public final String approveUntilDate() {
        return approveUntilDate;
    }

    /**
     * <p>
     * For managed nodes identified by the approval rule filters, enables a patch baseline to apply non-security updates
     * available in the specified repository. The default value is <code>false</code>. Applies to Linux managed nodes
     * only.
     * </p>
     * 
     * @return For managed nodes identified by the approval rule filters, enables a patch baseline to apply non-security
     *         updates available in the specified repository. The default value is <code>false</code>. Applies to Linux
     *         managed nodes only.
     */
    public final Boolean enableNonSecurity() {
        return enableNonSecurity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(patchFilterGroup());
        hashCode = 31 * hashCode + Objects.hashCode(complianceLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(approveAfterDays());
        hashCode = 31 * hashCode + Objects.hashCode(approveUntilDate());
        hashCode = 31 * hashCode + Objects.hashCode(enableNonSecurity());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PatchRule)) {
            return false;
        }
        PatchRule other = (PatchRule) obj;
        return Objects.equals(patchFilterGroup(), other.patchFilterGroup())
                && Objects.equals(complianceLevelAsString(), other.complianceLevelAsString())
                && Objects.equals(approveAfterDays(), other.approveAfterDays())
                && Objects.equals(approveUntilDate(), other.approveUntilDate())
                && Objects.equals(enableNonSecurity(), other.enableNonSecurity());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PatchRule").add("PatchFilterGroup", patchFilterGroup())
                .add("ComplianceLevel", complianceLevelAsString()).add("ApproveAfterDays", approveAfterDays())
                .add("ApproveUntilDate", approveUntilDate()).add("EnableNonSecurity", enableNonSecurity()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PatchFilterGroup":
            return Optional.ofNullable(clazz.cast(patchFilterGroup()));
        case "ComplianceLevel":
            return Optional.ofNullable(clazz.cast(complianceLevelAsString()));
        case "ApproveAfterDays":
            return Optional.ofNullable(clazz.cast(approveAfterDays()));
        case "ApproveUntilDate":
            return Optional.ofNullable(clazz.cast(approveUntilDate()));
        case "EnableNonSecurity":
            return Optional.ofNullable(clazz.cast(enableNonSecurity()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PatchRule, T> g) {
        return obj -> g.apply((PatchRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PatchRule> {
        /**
         * <p>
         * The patch filter group that defines the criteria for the rule.
         * </p>
         * 
         * @param patchFilterGroup
         *        The patch filter group that defines the criteria for the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patchFilterGroup(PatchFilterGroup patchFilterGroup);

        /**
         * <p>
         * The patch filter group that defines the criteria for the rule.
         * </p>
         * This is a convenience method that creates an instance of the {@link PatchFilterGroup.Builder} avoiding the
         * need to create one manually via {@link PatchFilterGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PatchFilterGroup.Builder#build()} is called immediately and its
         * result is passed to {@link #patchFilterGroup(PatchFilterGroup)}.
         * 
         * @param patchFilterGroup
         *        a consumer that will call methods on {@link PatchFilterGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #patchFilterGroup(PatchFilterGroup)
         */
        default Builder patchFilterGroup(Consumer<PatchFilterGroup.Builder> patchFilterGroup) {
            return patchFilterGroup(PatchFilterGroup.builder().applyMutation(patchFilterGroup).build());
        }

        /**
         * <p>
         * A compliance severity level for all approved patches in a patch baseline.
         * </p>
         * 
         * @param complianceLevel
         *        A compliance severity level for all approved patches in a patch baseline.
         * @see PatchComplianceLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchComplianceLevel
         */
        Builder complianceLevel(String complianceLevel);

        /**
         * <p>
         * A compliance severity level for all approved patches in a patch baseline.
         * </p>
         * 
         * @param complianceLevel
         *        A compliance severity level for all approved patches in a patch baseline.
         * @see PatchComplianceLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PatchComplianceLevel
         */
        Builder complianceLevel(PatchComplianceLevel complianceLevel);

        /**
         * <p>
         * The number of days after the release date of each patch matched by the rule that the patch is marked as
         * approved in the patch baseline. For example, a value of <code>7</code> means that patches are approved seven
         * days after they are released. Not supported on Debian Server or Ubuntu Server.
         * </p>
         * 
         * @param approveAfterDays
         *        The number of days after the release date of each patch matched by the rule that the patch is marked
         *        as approved in the patch baseline. For example, a value of <code>7</code> means that patches are
         *        approved seven days after they are released. Not supported on Debian Server or Ubuntu Server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approveAfterDays(Integer approveAfterDays);

        /**
         * <p>
         * The cutoff date for auto approval of released patches. Any patches released on or before this date are
         * installed automatically. Not supported on Debian Server or Ubuntu Server.
         * </p>
         * <p>
         * Enter dates in the format <code>YYYY-MM-DD</code>. For example, <code>2021-12-31</code>.
         * </p>
         * 
         * @param approveUntilDate
         *        The cutoff date for auto approval of released patches. Any patches released on or before this date are
         *        installed automatically. Not supported on Debian Server or Ubuntu Server.</p>
         *        <p>
         *        Enter dates in the format <code>YYYY-MM-DD</code>. For example, <code>2021-12-31</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approveUntilDate(String approveUntilDate);

        /**
         * <p>
         * For managed nodes identified by the approval rule filters, enables a patch baseline to apply non-security
         * updates available in the specified repository. The default value is <code>false</code>. Applies to Linux
         * managed nodes only.
         * </p>
         * 
         * @param enableNonSecurity
         *        For managed nodes identified by the approval rule filters, enables a patch baseline to apply
         *        non-security updates available in the specified repository. The default value is <code>false</code>.
         *        Applies to Linux managed nodes only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableNonSecurity(Boolean enableNonSecurity);
    }

    static final class BuilderImpl implements Builder {
        private PatchFilterGroup patchFilterGroup;

        private String complianceLevel;

        private Integer approveAfterDays;

        private String approveUntilDate;

        private Boolean enableNonSecurity;

        private BuilderImpl() {
        }

        private BuilderImpl(PatchRule model) {
            patchFilterGroup(model.patchFilterGroup);
            complianceLevel(model.complianceLevel);
            approveAfterDays(model.approveAfterDays);
            approveUntilDate(model.approveUntilDate);
            enableNonSecurity(model.enableNonSecurity);
        }

        public final PatchFilterGroup.Builder getPatchFilterGroup() {
            return patchFilterGroup != null ? patchFilterGroup.toBuilder() : null;
        }

        public final void setPatchFilterGroup(PatchFilterGroup.BuilderImpl patchFilterGroup) {
            this.patchFilterGroup = patchFilterGroup != null ? patchFilterGroup.build() : null;
        }

        @Override
        public final Builder patchFilterGroup(PatchFilterGroup patchFilterGroup) {
            this.patchFilterGroup = patchFilterGroup;
            return this;
        }

        public final String getComplianceLevel() {
            return complianceLevel;
        }

        public final void setComplianceLevel(String complianceLevel) {
            this.complianceLevel = complianceLevel;
        }

        @Override
        public final Builder complianceLevel(String complianceLevel) {
            this.complianceLevel = complianceLevel;
            return this;
        }

        @Override
        public final Builder complianceLevel(PatchComplianceLevel complianceLevel) {
            this.complianceLevel(complianceLevel == null ? null : complianceLevel.toString());
            return this;
        }

        public final Integer getApproveAfterDays() {
            return approveAfterDays;
        }

        public final void setApproveAfterDays(Integer approveAfterDays) {
            this.approveAfterDays = approveAfterDays;
        }

        @Override
        public final Builder approveAfterDays(Integer approveAfterDays) {
            this.approveAfterDays = approveAfterDays;
            return this;
        }

        public final String getApproveUntilDate() {
            return approveUntilDate;
        }

        public final void setApproveUntilDate(String approveUntilDate) {
            this.approveUntilDate = approveUntilDate;
        }

        @Override
        public final Builder approveUntilDate(String approveUntilDate) {
            this.approveUntilDate = approveUntilDate;
            return this;
        }

        public final Boolean getEnableNonSecurity() {
            return enableNonSecurity;
        }

        public final void setEnableNonSecurity(Boolean enableNonSecurity) {
            this.enableNonSecurity = enableNonSecurity;
        }

        @Override
        public final Builder enableNonSecurity(Boolean enableNonSecurity) {
            this.enableNonSecurity = enableNonSecurity;
            return this;
        }

        @Override
        public PatchRule build() {
            return new PatchRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
