/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * When included in a receipt rule, this action saves the received message to an Amazon Simple Storage Service (Amazon
 * S3) bucket and, optionally, publishes a notification to Amazon Simple Notification Service (Amazon SNS).
 * </p>
 * <p>
 * To enable Amazon SES to write emails to your Amazon S3 bucket, use an Amazon Web Services KMS key to encrypt your
 * emails, or publish to an Amazon SNS topic of another account, Amazon SES must have permission to access those
 * resources. For information about granting permissions, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-permissions.html">Amazon SES Developer Guide</a>.
 * </p>
 * <note>
 * <p>
 * When you save your emails to an Amazon S3 bucket, the maximum email size (including headers) is 40 MB. Emails larger
 * than that bounces.
 * </p>
 * </note>
 * <p>
 * For information about specifying Amazon S3 actions in receipt rules, see the <a
 * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-action-s3.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Action implements SdkPojo, Serializable, ToCopyableBuilder<S3Action.Builder, S3Action> {
    private static final SdkField<String> TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TopicArn").getter(getter(S3Action::topicArn)).setter(setter(Builder::topicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopicArn").build()).build();

    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketName").getter(getter(S3Action::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> OBJECT_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ObjectKeyPrefix").getter(getter(S3Action::objectKeyPrefix)).setter(setter(Builder::objectKeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectKeyPrefix").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyArn").getter(getter(S3Action::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(S3Action::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOPIC_ARN_FIELD,
            BUCKET_NAME_FIELD, OBJECT_KEY_PREFIX_FIELD, KMS_KEY_ARN_FIELD, IAM_ROLE_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String topicArn;

    private final String bucketName;

    private final String objectKeyPrefix;

    private final String kmsKeyArn;

    private final String iamRoleArn;

    private S3Action(BuilderImpl builder) {
        this.topicArn = builder.topicArn;
        this.bucketName = builder.bucketName;
        this.objectKeyPrefix = builder.objectKeyPrefix;
        this.kmsKeyArn = builder.kmsKeyArn;
        this.iamRoleArn = builder.iamRoleArn;
    }

    /**
     * <p>
     * The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. You can find the ARN
     * of a topic by using the <a href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a>
     * operation in Amazon SNS.
     * </p>
     * <p>
     * For more information about Amazon SNS topics, see the <a
     * href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     * </p>
     * 
     * @return The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. You can find
     *         the ARN of a topic by using the <a
     *         href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a> operation in Amazon
     *         SNS.</p>
     *         <p>
     *         For more information about Amazon SNS topics, see the <a
     *         href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     */
    public final String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket for incoming email.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket for incoming email.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you to store
     * similar data under the same directory in a bucket.
     * </p>
     * 
     * @return The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you to
     *         store similar data under the same directory in a bucket.
     */
    public final String objectKeyPrefix() {
        return objectKeyPrefix;
    }

    /**
     * <p>
     * The customer managed key that Amazon SES should use to encrypt your emails before saving them to the Amazon S3
     * bucket. You can use the Amazon Web Services managed key or a customer managed key that you created in Amazon Web
     * Services KMS as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To use the Amazon Web Services managed key, provide an ARN in the form of
     * <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your Amazon Web
     * Services account ID is 123456789012 and you want to use the Amazon Web Services managed key in the US West
     * (Oregon) Region, the ARN of the Amazon Web Services managed key would be
     * <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the Amazon Web Services managed key,
     * you don't need to perform any extra steps to give Amazon SES permission to use the key.
     * </p>
     * </li>
     * <li>
     * <p>
     * To use a customer managed key that you created in Amazon Web Services KMS, provide the ARN of the customer
     * managed key and ensure that you add a statement to your key's policy to give Amazon SES permission to use it. For
     * more information about giving permissions, see the <a
     * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-permissions.html">Amazon SES Developer Guide</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about key policies, see the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">Amazon Web Services KMS Developer
     * Guide</a>. If you do not specify an Amazon Web Services KMS key, Amazon SES does not encrypt your emails.
     * </p>
     * <important>
     * <p>
     * Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is submitted to Amazon
     * S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This means that you must use the
     * Amazon S3 encryption client to decrypt the email after retrieving it from Amazon S3, as the service has no access
     * to use your Amazon Web Services KMS keys for decryption. This encryption client is currently available with the
     * <a href="http://aws.amazon.com/sdk-for-java/">Amazon Web Services SDK for Java</a> and <a
     * href="http://aws.amazon.com/sdk-for-ruby/">Amazon Web Services SDK for Ruby</a> only. For more information about
     * client-side encryption using Amazon Web Services KMS managed keys, see the <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3 Developer
     * Guide</a>.
     * </p>
     * </important>
     * 
     * @return The customer managed key that Amazon SES should use to encrypt your emails before saving them to the
     *         Amazon S3 bucket. You can use the Amazon Web Services managed key or a customer managed key that you
     *         created in Amazon Web Services KMS as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To use the Amazon Web Services managed key, provide an ARN in the form of
     *         <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your Amazon Web
     *         Services account ID is 123456789012 and you want to use the Amazon Web Services managed key in the US
     *         West (Oregon) Region, the ARN of the Amazon Web Services managed key would be
     *         <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the Amazon Web Services managed
     *         key, you don't need to perform any extra steps to give Amazon SES permission to use the key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To use a customer managed key that you created in Amazon Web Services KMS, provide the ARN of the
     *         customer managed key and ensure that you add a statement to your key's policy to give Amazon SES
     *         permission to use it. For more information about giving permissions, see the <a
     *         href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-permissions.html">Amazon SES Developer
     *         Guide</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about key policies, see the <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">Amazon Web Services KMS
     *         Developer Guide</a>. If you do not specify an Amazon Web Services KMS key, Amazon SES does not encrypt
     *         your emails.
     *         </p>
     *         <important>
     *         <p>
     *         Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is submitted
     *         to Amazon S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This means that you
     *         must use the Amazon S3 encryption client to decrypt the email after retrieving it from Amazon S3, as the
     *         service has no access to use your Amazon Web Services KMS keys for decryption. This encryption client is
     *         currently available with the <a href="http://aws.amazon.com/sdk-for-java/">Amazon Web Services SDK for
     *         Java</a> and <a href="http://aws.amazon.com/sdk-for-ruby/">Amazon Web Services SDK for Ruby</a> only. For
     *         more information about client-side encryption using Amazon Web Services KMS managed keys, see the <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3 Developer
     *         Guide</a>.
     *         </p>
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    /**
     * <p>
     * The ARN of the IAM role to be used by Amazon Simple Email Service while writing to the Amazon S3 bucket,
     * optionally encrypting your mail via the provided customer managed key, and publishing to the Amazon SNS topic.
     * This role should have access to the following APIs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>s3:PutObject</code>, <code>kms:Encrypt</code> and <code>kms:GenerateDataKey</code> for the given Amazon S3
     * bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>kms:GenerateDataKey</code> for the given Amazon Web Services KMS customer managed key.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sns:Publish</code> for the given Amazon SNS topic.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If an IAM role ARN is provided, the role (and only the role) is used to access all the given resources (Amazon S3
     * bucket, Amazon Web Services KMS customer managed key and Amazon SNS topic). Therefore, setting up individual
     * resource access permissions is not required.
     * </p>
     * </note>
     * 
     * @return The ARN of the IAM role to be used by Amazon Simple Email Service while writing to the Amazon S3 bucket,
     *         optionally encrypting your mail via the provided customer managed key, and publishing to the Amazon SNS
     *         topic. This role should have access to the following APIs: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>s3:PutObject</code>, <code>kms:Encrypt</code> and <code>kms:GenerateDataKey</code> for the given
     *         Amazon S3 bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>kms:GenerateDataKey</code> for the given Amazon Web Services KMS customer managed key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sns:Publish</code> for the given Amazon SNS topic.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If an IAM role ARN is provided, the role (and only the role) is used to access all the given resources
     *         (Amazon S3 bucket, Amazon Web Services KMS customer managed key and Amazon SNS topic). Therefore, setting
     *         up individual resource access permissions is not required.
     *         </p>
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(topicArn());
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(objectKeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Action)) {
            return false;
        }
        S3Action other = (S3Action) obj;
        return Objects.equals(topicArn(), other.topicArn()) && Objects.equals(bucketName(), other.bucketName())
                && Objects.equals(objectKeyPrefix(), other.objectKeyPrefix()) && Objects.equals(kmsKeyArn(), other.kmsKeyArn())
                && Objects.equals(iamRoleArn(), other.iamRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Action").add("TopicArn", topicArn()).add("BucketName", bucketName())
                .add("ObjectKeyPrefix", objectKeyPrefix()).add("KmsKeyArn", kmsKeyArn()).add("IamRoleArn", iamRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TopicArn":
            return Optional.ofNullable(clazz.cast(topicArn()));
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "ObjectKeyPrefix":
            return Optional.ofNullable(clazz.cast(objectKeyPrefix()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("TopicArn", TOPIC_ARN_FIELD);
        map.put("BucketName", BUCKET_NAME_FIELD);
        map.put("ObjectKeyPrefix", OBJECT_KEY_PREFIX_FIELD);
        map.put("KmsKeyArn", KMS_KEY_ARN_FIELD);
        map.put("IamRoleArn", IAM_ROLE_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3Action, T> g) {
        return obj -> g.apply((S3Action) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Action> {
        /**
         * <p>
         * The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. You can find the
         * ARN of a topic by using the <a
         * href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a> operation in Amazon SNS.
         * </p>
         * <p>
         * For more information about Amazon SNS topics, see the <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * </p>
         * 
         * @param topicArn
         *        The ARN of the Amazon SNS topic to notify when the message is saved to the Amazon S3 bucket. You can
         *        find the ARN of a topic by using the <a
         *        href="https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html">ListTopics</a> operation in
         *        Amazon SNS.</p>
         *        <p>
         *        For more information about Amazon SNS topics, see the <a
         *        href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The name of the Amazon S3 bucket for incoming email.
         * </p>
         * 
         * @param bucketName
         *        The name of the Amazon S3 bucket for incoming email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you to
         * store similar data under the same directory in a bucket.
         * </p>
         * 
         * @param objectKeyPrefix
         *        The key prefix of the Amazon S3 bucket. The key prefix is similar to a directory name that enables you
         *        to store similar data under the same directory in a bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectKeyPrefix(String objectKeyPrefix);

        /**
         * <p>
         * The customer managed key that Amazon SES should use to encrypt your emails before saving them to the Amazon
         * S3 bucket. You can use the Amazon Web Services managed key or a customer managed key that you created in
         * Amazon Web Services KMS as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To use the Amazon Web Services managed key, provide an ARN in the form of
         * <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your Amazon Web
         * Services account ID is 123456789012 and you want to use the Amazon Web Services managed key in the US West
         * (Oregon) Region, the ARN of the Amazon Web Services managed key would be
         * <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the Amazon Web Services managed
         * key, you don't need to perform any extra steps to give Amazon SES permission to use the key.
         * </p>
         * </li>
         * <li>
         * <p>
         * To use a customer managed key that you created in Amazon Web Services KMS, provide the ARN of the customer
         * managed key and ensure that you add a statement to your key's policy to give Amazon SES permission to use it.
         * For more information about giving permissions, see the <a
         * href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-permissions.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about key policies, see the <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">Amazon Web Services KMS Developer
         * Guide</a>. If you do not specify an Amazon Web Services KMS key, Amazon SES does not encrypt your emails.
         * </p>
         * <important>
         * <p>
         * Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is submitted to
         * Amazon S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This means that you must
         * use the Amazon S3 encryption client to decrypt the email after retrieving it from Amazon S3, as the service
         * has no access to use your Amazon Web Services KMS keys for decryption. This encryption client is currently
         * available with the <a href="http://aws.amazon.com/sdk-for-java/">Amazon Web Services SDK for Java</a> and <a
         * href="http://aws.amazon.com/sdk-for-ruby/">Amazon Web Services SDK for Ruby</a> only. For more information
         * about client-side encryption using Amazon Web Services KMS managed keys, see the <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3 Developer
         * Guide</a>.
         * </p>
         * </important>
         * 
         * @param kmsKeyArn
         *        The customer managed key that Amazon SES should use to encrypt your emails before saving them to the
         *        Amazon S3 bucket. You can use the Amazon Web Services managed key or a customer managed key that you
         *        created in Amazon Web Services KMS as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To use the Amazon Web Services managed key, provide an ARN in the form of
         *        <code>arn:aws:kms:REGION:ACCOUNT-ID-WITHOUT-HYPHENS:alias/aws/ses</code>. For example, if your Amazon
         *        Web Services account ID is 123456789012 and you want to use the Amazon Web Services managed key in the
         *        US West (Oregon) Region, the ARN of the Amazon Web Services managed key would be
         *        <code>arn:aws:kms:us-west-2:123456789012:alias/aws/ses</code>. If you use the Amazon Web Services
         *        managed key, you don't need to perform any extra steps to give Amazon SES permission to use the key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To use a customer managed key that you created in Amazon Web Services KMS, provide the ARN of the
         *        customer managed key and ensure that you add a statement to your key's policy to give Amazon SES
         *        permission to use it. For more information about giving permissions, see the <a
         *        href="https://docs.aws.amazon.com/ses/latest/dg/receiving-email-permissions.html">Amazon SES Developer
         *        Guide</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about key policies, see the <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html">Amazon Web Services KMS
         *        Developer Guide</a>. If you do not specify an Amazon Web Services KMS key, Amazon SES does not encrypt
         *        your emails.
         *        </p>
         *        <important>
         *        <p>
         *        Your mail is encrypted by Amazon SES using the Amazon S3 encryption client before the mail is
         *        submitted to Amazon S3 for storage. It is not encrypted using Amazon S3 server-side encryption. This
         *        means that you must use the Amazon S3 encryption client to decrypt the email after retrieving it from
         *        Amazon S3, as the service has no access to use your Amazon Web Services KMS keys for decryption. This
         *        encryption client is currently available with the <a href="http://aws.amazon.com/sdk-for-java/">Amazon
         *        Web Services SDK for Java</a> and <a href="http://aws.amazon.com/sdk-for-ruby/">Amazon Web Services
         *        SDK for Ruby</a> only. For more information about client-side encryption using Amazon Web Services KMS
         *        managed keys, see the <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html">Amazon S3
         *        Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);

        /**
         * <p>
         * The ARN of the IAM role to be used by Amazon Simple Email Service while writing to the Amazon S3 bucket,
         * optionally encrypting your mail via the provided customer managed key, and publishing to the Amazon SNS
         * topic. This role should have access to the following APIs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>s3:PutObject</code>, <code>kms:Encrypt</code> and <code>kms:GenerateDataKey</code> for the given Amazon
         * S3 bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>kms:GenerateDataKey</code> for the given Amazon Web Services KMS customer managed key.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sns:Publish</code> for the given Amazon SNS topic.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If an IAM role ARN is provided, the role (and only the role) is used to access all the given resources
         * (Amazon S3 bucket, Amazon Web Services KMS customer managed key and Amazon SNS topic). Therefore, setting up
         * individual resource access permissions is not required.
         * </p>
         * </note>
         * 
         * @param iamRoleArn
         *        The ARN of the IAM role to be used by Amazon Simple Email Service while writing to the Amazon S3
         *        bucket, optionally encrypting your mail via the provided customer managed key, and publishing to the
         *        Amazon SNS topic. This role should have access to the following APIs: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>s3:PutObject</code>, <code>kms:Encrypt</code> and <code>kms:GenerateDataKey</code> for the given
         *        Amazon S3 bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>kms:GenerateDataKey</code> for the given Amazon Web Services KMS customer managed key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sns:Publish</code> for the given Amazon SNS topic.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If an IAM role ARN is provided, the role (and only the role) is used to access all the given resources
         *        (Amazon S3 bucket, Amazon Web Services KMS customer managed key and Amazon SNS topic). Therefore,
         *        setting up individual resource access permissions is not required.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String topicArn;

        private String bucketName;

        private String objectKeyPrefix;

        private String kmsKeyArn;

        private String iamRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Action model) {
            topicArn(model.topicArn);
            bucketName(model.bucketName);
            objectKeyPrefix(model.objectKeyPrefix);
            kmsKeyArn(model.kmsKeyArn);
            iamRoleArn(model.iamRoleArn);
        }

        public final String getTopicArn() {
            return topicArn;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        @Override
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final String getObjectKeyPrefix() {
            return objectKeyPrefix;
        }

        public final void setObjectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
        }

        @Override
        public final Builder objectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        @Override
        public S3Action build() {
            return new S3Action(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
