/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The refresh schedule of a dataset.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RefreshSchedule implements SdkPojo, Serializable, ToCopyableBuilder<RefreshSchedule.Builder, RefreshSchedule> {
    private static final SdkField<String> SCHEDULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScheduleId").getter(getter(RefreshSchedule::scheduleId)).setter(setter(Builder::scheduleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleId").build()).build();

    private static final SdkField<RefreshFrequency> SCHEDULE_FREQUENCY_FIELD = SdkField
            .<RefreshFrequency> builder(MarshallingType.SDK_POJO).memberName("ScheduleFrequency")
            .getter(getter(RefreshSchedule::scheduleFrequency)).setter(setter(Builder::scheduleFrequency))
            .constructor(RefreshFrequency::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleFrequency").build()).build();

    private static final SdkField<Instant> START_AFTER_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartAfterDateTime").getter(getter(RefreshSchedule::startAfterDateTime))
            .setter(setter(Builder::startAfterDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartAfterDateTime").build())
            .build();

    private static final SdkField<String> REFRESH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RefreshType").getter(getter(RefreshSchedule::refreshTypeAsString)).setter(setter(Builder::refreshType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RefreshType").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(RefreshSchedule::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCHEDULE_ID_FIELD,
            SCHEDULE_FREQUENCY_FIELD, START_AFTER_DATE_TIME_FIELD, REFRESH_TYPE_FIELD, ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String scheduleId;

    private final RefreshFrequency scheduleFrequency;

    private final Instant startAfterDateTime;

    private final String refreshType;

    private final String arn;

    private RefreshSchedule(BuilderImpl builder) {
        this.scheduleId = builder.scheduleId;
        this.scheduleFrequency = builder.scheduleFrequency;
        this.startAfterDateTime = builder.startAfterDateTime;
        this.refreshType = builder.refreshType;
        this.arn = builder.arn;
    }

    /**
     * <p>
     * An identifier for the refresh schedule.
     * </p>
     * 
     * @return An identifier for the refresh schedule.
     */
    public final String scheduleId() {
        return scheduleId;
    }

    /**
     * <p>
     * The frequency for the refresh schedule.
     * </p>
     * 
     * @return The frequency for the refresh schedule.
     */
    public final RefreshFrequency scheduleFrequency() {
        return scheduleFrequency;
    }

    /**
     * <p>
     * Time after which the refresh schedule can be started, expressed in <code>YYYY-MM-DDTHH:MM:SS</code> format.
     * </p>
     * 
     * @return Time after which the refresh schedule can be started, expressed in <code>YYYY-MM-DDTHH:MM:SS</code>
     *         format.
     */
    public final Instant startAfterDateTime() {
        return startAfterDateTime;
    }

    /**
     * <p>
     * The type of refresh that a datset undergoes. Valid values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FULL_REFRESH</code>: A complete refresh of a dataset.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time window
     * specified.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on full and incremental refreshes, see <a
     * href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing SPICE data</a>
     * in the <i>Amazon QuickSight User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #refreshType} will
     * return {@link IngestionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #refreshTypeAsString}.
     * </p>
     * 
     * @return The type of refresh that a datset undergoes. Valid values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FULL_REFRESH</code>: A complete refresh of a dataset.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time window
     *         specified.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information on full and incremental refreshes, see <a
     *         href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing SPICE
     *         data</a> in the <i>Amazon QuickSight User Guide</i>.
     * @see IngestionType
     */
    public final IngestionType refreshType() {
        return IngestionType.fromValue(refreshType);
    }

    /**
     * <p>
     * The type of refresh that a datset undergoes. Valid values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FULL_REFRESH</code>: A complete refresh of a dataset.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time window
     * specified.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on full and incremental refreshes, see <a
     * href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing SPICE data</a>
     * in the <i>Amazon QuickSight User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #refreshType} will
     * return {@link IngestionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #refreshTypeAsString}.
     * </p>
     * 
     * @return The type of refresh that a datset undergoes. Valid values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FULL_REFRESH</code>: A complete refresh of a dataset.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time window
     *         specified.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information on full and incremental refreshes, see <a
     *         href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing SPICE
     *         data</a> in the <i>Amazon QuickSight User Guide</i>.
     * @see IngestionType
     */
    public final String refreshTypeAsString() {
        return refreshType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the refresh schedule.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the refresh schedule.
     */
    public final String arn() {
        return arn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(scheduleId());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleFrequency());
        hashCode = 31 * hashCode + Objects.hashCode(startAfterDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(refreshTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RefreshSchedule)) {
            return false;
        }
        RefreshSchedule other = (RefreshSchedule) obj;
        return Objects.equals(scheduleId(), other.scheduleId()) && Objects.equals(scheduleFrequency(), other.scheduleFrequency())
                && Objects.equals(startAfterDateTime(), other.startAfterDateTime())
                && Objects.equals(refreshTypeAsString(), other.refreshTypeAsString()) && Objects.equals(arn(), other.arn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RefreshSchedule").add("ScheduleId", scheduleId()).add("ScheduleFrequency", scheduleFrequency())
                .add("StartAfterDateTime", startAfterDateTime()).add("RefreshType", refreshTypeAsString()).add("Arn", arn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ScheduleId":
            return Optional.ofNullable(clazz.cast(scheduleId()));
        case "ScheduleFrequency":
            return Optional.ofNullable(clazz.cast(scheduleFrequency()));
        case "StartAfterDateTime":
            return Optional.ofNullable(clazz.cast(startAfterDateTime()));
        case "RefreshType":
            return Optional.ofNullable(clazz.cast(refreshTypeAsString()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RefreshSchedule, T> g) {
        return obj -> g.apply((RefreshSchedule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RefreshSchedule> {
        /**
         * <p>
         * An identifier for the refresh schedule.
         * </p>
         * 
         * @param scheduleId
         *        An identifier for the refresh schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleId(String scheduleId);

        /**
         * <p>
         * The frequency for the refresh schedule.
         * </p>
         * 
         * @param scheduleFrequency
         *        The frequency for the refresh schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleFrequency(RefreshFrequency scheduleFrequency);

        /**
         * <p>
         * The frequency for the refresh schedule.
         * </p>
         * This is a convenience method that creates an instance of the {@link RefreshFrequency.Builder} avoiding the
         * need to create one manually via {@link RefreshFrequency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RefreshFrequency.Builder#build()} is called immediately and its
         * result is passed to {@link #scheduleFrequency(RefreshFrequency)}.
         * 
         * @param scheduleFrequency
         *        a consumer that will call methods on {@link RefreshFrequency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scheduleFrequency(RefreshFrequency)
         */
        default Builder scheduleFrequency(Consumer<RefreshFrequency.Builder> scheduleFrequency) {
            return scheduleFrequency(RefreshFrequency.builder().applyMutation(scheduleFrequency).build());
        }

        /**
         * <p>
         * Time after which the refresh schedule can be started, expressed in <code>YYYY-MM-DDTHH:MM:SS</code> format.
         * </p>
         * 
         * @param startAfterDateTime
         *        Time after which the refresh schedule can be started, expressed in <code>YYYY-MM-DDTHH:MM:SS</code>
         *        format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startAfterDateTime(Instant startAfterDateTime);

        /**
         * <p>
         * The type of refresh that a datset undergoes. Valid values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FULL_REFRESH</code>: A complete refresh of a dataset.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time window
         * specified.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information on full and incremental refreshes, see <a
         * href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing SPICE
         * data</a> in the <i>Amazon QuickSight User Guide</i>.
         * </p>
         * 
         * @param refreshType
         *        The type of refresh that a datset undergoes. Valid values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FULL_REFRESH</code>: A complete refresh of a dataset.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time
         *        window specified.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information on full and incremental refreshes, see <a
         *        href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing
         *        SPICE data</a> in the <i>Amazon QuickSight User Guide</i>.
         * @see IngestionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IngestionType
         */
        Builder refreshType(String refreshType);

        /**
         * <p>
         * The type of refresh that a datset undergoes. Valid values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FULL_REFRESH</code>: A complete refresh of a dataset.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time window
         * specified.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information on full and incremental refreshes, see <a
         * href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing SPICE
         * data</a> in the <i>Amazon QuickSight User Guide</i>.
         * </p>
         * 
         * @param refreshType
         *        The type of refresh that a datset undergoes. Valid values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FULL_REFRESH</code>: A complete refresh of a dataset.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCREMENTAL_REFRESH</code>: A partial refresh of some rows of a dataset, based on the time
         *        window specified.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information on full and incremental refreshes, see <a
         *        href="https://docs.aws.amazon.com/quicksight/latest/user/refreshing-imported-data.html">Refreshing
         *        SPICE data</a> in the <i>Amazon QuickSight User Guide</i>.
         * @see IngestionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IngestionType
         */
        Builder refreshType(IngestionType refreshType);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the refresh schedule.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) for the refresh schedule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);
    }

    static final class BuilderImpl implements Builder {
        private String scheduleId;

        private RefreshFrequency scheduleFrequency;

        private Instant startAfterDateTime;

        private String refreshType;

        private String arn;

        private BuilderImpl() {
        }

        private BuilderImpl(RefreshSchedule model) {
            scheduleId(model.scheduleId);
            scheduleFrequency(model.scheduleFrequency);
            startAfterDateTime(model.startAfterDateTime);
            refreshType(model.refreshType);
            arn(model.arn);
        }

        public final String getScheduleId() {
            return scheduleId;
        }

        public final void setScheduleId(String scheduleId) {
            this.scheduleId = scheduleId;
        }

        @Override
        public final Builder scheduleId(String scheduleId) {
            this.scheduleId = scheduleId;
            return this;
        }

        public final RefreshFrequency.Builder getScheduleFrequency() {
            return scheduleFrequency != null ? scheduleFrequency.toBuilder() : null;
        }

        public final void setScheduleFrequency(RefreshFrequency.BuilderImpl scheduleFrequency) {
            this.scheduleFrequency = scheduleFrequency != null ? scheduleFrequency.build() : null;
        }

        @Override
        public final Builder scheduleFrequency(RefreshFrequency scheduleFrequency) {
            this.scheduleFrequency = scheduleFrequency;
            return this;
        }

        public final Instant getStartAfterDateTime() {
            return startAfterDateTime;
        }

        public final void setStartAfterDateTime(Instant startAfterDateTime) {
            this.startAfterDateTime = startAfterDateTime;
        }

        @Override
        public final Builder startAfterDateTime(Instant startAfterDateTime) {
            this.startAfterDateTime = startAfterDateTime;
            return this;
        }

        public final String getRefreshType() {
            return refreshType;
        }

        public final void setRefreshType(String refreshType) {
            this.refreshType = refreshType;
        }

        @Override
        public final Builder refreshType(String refreshType) {
            this.refreshType = refreshType;
            return this;
        }

        @Override
        public final Builder refreshType(IngestionType refreshType) {
            this.refreshType(refreshType == null ? null : refreshType.toString());
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        @Override
        public RefreshSchedule build() {
            return new RefreshSchedule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
