/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancingv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a load balancer.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoadBalancer implements SdkPojo, Serializable, ToCopyableBuilder<LoadBalancer.Builder, LoadBalancer> {
    private static final SdkField<String> LOAD_BALANCER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoadBalancerArn").getter(getter(LoadBalancer::loadBalancerArn)).setter(setter(Builder::loadBalancerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadBalancerArn").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DNSName").getter(getter(LoadBalancer::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DNSName").build()).build();

    private static final SdkField<String> CANONICAL_HOSTED_ZONE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CanonicalHostedZoneId").getter(getter(LoadBalancer::canonicalHostedZoneId))
            .setter(setter(Builder::canonicalHostedZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CanonicalHostedZoneId").build())
            .build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(LoadBalancer::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<String> LOAD_BALANCER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoadBalancerName").getter(getter(LoadBalancer::loadBalancerName))
            .setter(setter(Builder::loadBalancerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadBalancerName").build()).build();

    private static final SdkField<String> SCHEME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Scheme")
            .getter(getter(LoadBalancer::schemeAsString)).setter(setter(Builder::scheme))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scheme").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(LoadBalancer::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<LoadBalancerState> STATE_FIELD = SdkField.<LoadBalancerState> builder(MarshallingType.SDK_POJO)
            .memberName("State").getter(getter(LoadBalancer::state)).setter(setter(Builder::state))
            .constructor(LoadBalancerState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(LoadBalancer::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<List<AvailabilityZone>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<AvailabilityZone>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(LoadBalancer::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AvailabilityZone> builder(MarshallingType.SDK_POJO)
                                            .constructor(AvailabilityZone::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(LoadBalancer::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddressType").getter(getter(LoadBalancer::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressType").build()).build();

    private static final SdkField<String> CUSTOMER_OWNED_IPV4_POOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerOwnedIpv4Pool").getter(getter(LoadBalancer::customerOwnedIpv4Pool))
            .setter(setter(Builder::customerOwnedIpv4Pool))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerOwnedIpv4Pool").build())
            .build();

    private static final SdkField<String> ENFORCE_SECURITY_GROUP_INBOUND_RULES_ON_PRIVATE_LINK_TRAFFIC_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EnforceSecurityGroupInboundRulesOnPrivateLinkTraffic")
            .getter(getter(LoadBalancer::enforceSecurityGroupInboundRulesOnPrivateLinkTraffic))
            .setter(setter(Builder::enforceSecurityGroupInboundRulesOnPrivateLinkTraffic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("EnforceSecurityGroupInboundRulesOnPrivateLinkTraffic").build()).build();

    private static final SdkField<String> ENABLE_PREFIX_FOR_IPV6_SOURCE_NAT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EnablePrefixForIpv6SourceNat")
            .getter(getter(LoadBalancer::enablePrefixForIpv6SourceNatAsString))
            .setter(setter(Builder::enablePrefixForIpv6SourceNat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnablePrefixForIpv6SourceNat")
                    .build()).build();

    private static final SdkField<IpamPools> IPAM_POOLS_FIELD = SdkField.<IpamPools> builder(MarshallingType.SDK_POJO)
            .memberName("IpamPools").getter(getter(LoadBalancer::ipamPools)).setter(setter(Builder::ipamPools))
            .constructor(IpamPools::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPools").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOAD_BALANCER_ARN_FIELD,
            DNS_NAME_FIELD, CANONICAL_HOSTED_ZONE_ID_FIELD, CREATED_TIME_FIELD, LOAD_BALANCER_NAME_FIELD, SCHEME_FIELD,
            VPC_ID_FIELD, STATE_FIELD, TYPE_FIELD, AVAILABILITY_ZONES_FIELD, SECURITY_GROUPS_FIELD, IP_ADDRESS_TYPE_FIELD,
            CUSTOMER_OWNED_IPV4_POOL_FIELD, ENFORCE_SECURITY_GROUP_INBOUND_RULES_ON_PRIVATE_LINK_TRAFFIC_FIELD,
            ENABLE_PREFIX_FOR_IPV6_SOURCE_NAT_FIELD, IPAM_POOLS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String loadBalancerArn;

    private final String dnsName;

    private final String canonicalHostedZoneId;

    private final Instant createdTime;

    private final String loadBalancerName;

    private final String scheme;

    private final String vpcId;

    private final LoadBalancerState state;

    private final String type;

    private final List<AvailabilityZone> availabilityZones;

    private final List<String> securityGroups;

    private final String ipAddressType;

    private final String customerOwnedIpv4Pool;

    private final String enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;

    private final String enablePrefixForIpv6SourceNat;

    private final IpamPools ipamPools;

    private LoadBalancer(BuilderImpl builder) {
        this.loadBalancerArn = builder.loadBalancerArn;
        this.dnsName = builder.dnsName;
        this.canonicalHostedZoneId = builder.canonicalHostedZoneId;
        this.createdTime = builder.createdTime;
        this.loadBalancerName = builder.loadBalancerName;
        this.scheme = builder.scheme;
        this.vpcId = builder.vpcId;
        this.state = builder.state;
        this.type = builder.type;
        this.availabilityZones = builder.availabilityZones;
        this.securityGroups = builder.securityGroups;
        this.ipAddressType = builder.ipAddressType;
        this.customerOwnedIpv4Pool = builder.customerOwnedIpv4Pool;
        this.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic = builder.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;
        this.enablePrefixForIpv6SourceNat = builder.enablePrefixForIpv6SourceNat;
        this.ipamPools = builder.ipamPools;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the load balancer.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the load balancer.
     */
    public final String loadBalancerArn() {
        return loadBalancerArn;
    }

    /**
     * <p>
     * The public DNS name of the load balancer.
     * </p>
     * 
     * @return The public DNS name of the load balancer.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The ID of the Amazon Route 53 hosted zone associated with the load balancer.
     * </p>
     * 
     * @return The ID of the Amazon Route 53 hosted zone associated with the load balancer.
     */
    public final String canonicalHostedZoneId() {
        return canonicalHostedZoneId;
    }

    /**
     * <p>
     * The date and time the load balancer was created.
     * </p>
     * 
     * @return The date and time the load balancer was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The name of the load balancer.
     * </p>
     * 
     * @return The name of the load balancer.
     */
    public final String loadBalancerName() {
        return loadBalancerName;
    }

    /**
     * <p>
     * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing load
     * balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
     * balancers can route requests from clients over the internet.
     * </p>
     * <p>
     * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load balancer
     * is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers can route
     * requests only from clients with access to the VPC for the load balancer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scheme} will
     * return {@link LoadBalancerSchemeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #schemeAsString}.
     * </p>
     * 
     * @return The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
     *         Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore,
     *         Internet-facing load balancers can route requests from clients over the internet.</p>
     *         <p>
     *         The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
     *         balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load
     *         balancers can route requests only from clients with access to the VPC for the load balancer.
     * @see LoadBalancerSchemeEnum
     */
    public final LoadBalancerSchemeEnum scheme() {
        return LoadBalancerSchemeEnum.fromValue(scheme);
    }

    /**
     * <p>
     * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing load
     * balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
     * balancers can route requests from clients over the internet.
     * </p>
     * <p>
     * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load balancer
     * is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers can route
     * requests only from clients with access to the VPC for the load balancer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scheme} will
     * return {@link LoadBalancerSchemeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #schemeAsString}.
     * </p>
     * 
     * @return The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
     *         Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore,
     *         Internet-facing load balancers can route requests from clients over the internet.</p>
     *         <p>
     *         The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
     *         balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load
     *         balancers can route requests only from clients with access to the VPC for the load balancer.
     * @see LoadBalancerSchemeEnum
     */
    public final String schemeAsString() {
        return scheme;
    }

    /**
     * <p>
     * The ID of the VPC for the load balancer.
     * </p>
     * 
     * @return The ID of the VPC for the load balancer.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The state of the load balancer.
     * </p>
     * 
     * @return The state of the load balancer.
     */
    public final LoadBalancerState state() {
        return state;
    }

    /**
     * <p>
     * The type of load balancer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link LoadBalancerTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of load balancer.
     * @see LoadBalancerTypeEnum
     */
    public final LoadBalancerTypeEnum type() {
        return LoadBalancerTypeEnum.fromValue(type);
    }

    /**
     * <p>
     * The type of load balancer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link LoadBalancerTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of load balancer.
     * @see LoadBalancerTypeEnum
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The subnets for the load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return The subnets for the load balancer.
     */
    public final List<AvailabilityZone> availabilityZones() {
        return availabilityZones;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the security groups for the load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return The IDs of the security groups for the load balancer.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * The type of IP addresses used for public or private connections by the subnets attached to your load balancer.
     * </p>
     * <p>
     * [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses), <code>dualstack</code>
     * (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code> (public IPv6 addresses and private IPv4
     * and IPv6 addresses).
     * </p>
     * <p>
     * [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses)
     * and <code>dualstack</code> (IPv4 and IPv6 addresses).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP addresses used for public or private connections by the subnets attached to your load
     *         balancer.</p>
     *         <p>
     *         [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses),
     *         <code>dualstack</code> (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code> (public
     *         IPv6 addresses and private IPv4 and IPv6 addresses).
     *         </p>
     *         <p>
     *         [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4
     *         addresses) and <code>dualstack</code> (IPv4 and IPv6 addresses).
     * @see IpAddressType
     */
    public final IpAddressType ipAddressType() {
        return IpAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The type of IP addresses used for public or private connections by the subnets attached to your load balancer.
     * </p>
     * <p>
     * [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses), <code>dualstack</code>
     * (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code> (public IPv6 addresses and private IPv4
     * and IPv6 addresses).
     * </p>
     * <p>
     * [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses)
     * and <code>dualstack</code> (IPv4 and IPv6 addresses).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP addresses used for public or private connections by the subnets attached to your load
     *         balancer.</p>
     *         <p>
     *         [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses),
     *         <code>dualstack</code> (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code> (public
     *         IPv6 addresses and private IPv4 and IPv6 addresses).
     *         </p>
     *         <p>
     *         [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4
     *         addresses) and <code>dualstack</code> (IPv4 and IPv6 addresses).
     * @see IpAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * <p>
     * [Application Load Balancers on Outposts] The ID of the customer-owned address pool.
     * </p>
     * 
     * @return [Application Load Balancers on Outposts] The ID of the customer-owned address pool.
     */
    public final String customerOwnedIpv4Pool() {
        return customerOwnedIpv4Pool;
    }

    /**
     * <p>
     * Indicates whether to evaluate inbound security group rules for traffic sent to a Network Load Balancer through
     * Amazon Web Services PrivateLink.
     * </p>
     * 
     * @return Indicates whether to evaluate inbound security group rules for traffic sent to a Network Load Balancer
     *         through Amazon Web Services PrivateLink.
     */
    public final String enforceSecurityGroupInboundRulesOnPrivateLinkTraffic() {
        return enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;
    }

    /**
     * <p>
     * [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet for source
     * NAT. The IP address type must be <code>dualstack</code>. The default value is <code>off</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enablePrefixForIpv6SourceNat} will return {@link EnablePrefixForIpv6SourceNatEnum#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #enablePrefixForIpv6SourceNatAsString}.
     * </p>
     * 
     * @return [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet for
     *         source NAT. The IP address type must be <code>dualstack</code>. The default value is <code>off</code>.
     * @see EnablePrefixForIpv6SourceNatEnum
     */
    public final EnablePrefixForIpv6SourceNatEnum enablePrefixForIpv6SourceNat() {
        return EnablePrefixForIpv6SourceNatEnum.fromValue(enablePrefixForIpv6SourceNat);
    }

    /**
     * <p>
     * [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet for source
     * NAT. The IP address type must be <code>dualstack</code>. The default value is <code>off</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enablePrefixForIpv6SourceNat} will return {@link EnablePrefixForIpv6SourceNatEnum#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #enablePrefixForIpv6SourceNatAsString}.
     * </p>
     * 
     * @return [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet for
     *         source NAT. The IP address type must be <code>dualstack</code>. The default value is <code>off</code>.
     * @see EnablePrefixForIpv6SourceNatEnum
     */
    public final String enablePrefixForIpv6SourceNatAsString() {
        return enablePrefixForIpv6SourceNat;
    }

    /**
     * <p>
     * [Application Load Balancers] The IPAM pool in use by the load balancer, if configured.
     * </p>
     * 
     * @return [Application Load Balancers] The IPAM pool in use by the load balancer, if configured.
     */
    public final IpamPools ipamPools() {
        return ipamPools;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerArn());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(canonicalHostedZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerName());
        hashCode = 31 * hashCode + Objects.hashCode(schemeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customerOwnedIpv4Pool());
        hashCode = 31 * hashCode + Objects.hashCode(enforceSecurityGroupInboundRulesOnPrivateLinkTraffic());
        hashCode = 31 * hashCode + Objects.hashCode(enablePrefixForIpv6SourceNatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPools());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoadBalancer)) {
            return false;
        }
        LoadBalancer other = (LoadBalancer) obj;
        return Objects.equals(loadBalancerArn(), other.loadBalancerArn())
                && Objects.equals(dnsName(), other.dnsName())
                && Objects.equals(canonicalHostedZoneId(), other.canonicalHostedZoneId())
                && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(loadBalancerName(), other.loadBalancerName())
                && Objects.equals(schemeAsString(), other.schemeAsString())
                && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(state(), other.state())
                && Objects.equals(typeAsString(), other.typeAsString())
                && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones())
                && hasSecurityGroups() == other.hasSecurityGroups()
                && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString())
                && Objects.equals(customerOwnedIpv4Pool(), other.customerOwnedIpv4Pool())
                && Objects.equals(enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(),
                        other.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic())
                && Objects.equals(enablePrefixForIpv6SourceNatAsString(), other.enablePrefixForIpv6SourceNatAsString())
                && Objects.equals(ipamPools(), other.ipamPools());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("LoadBalancer")
                .add("LoadBalancerArn", loadBalancerArn())
                .add("DNSName", dnsName())
                .add("CanonicalHostedZoneId", canonicalHostedZoneId())
                .add("CreatedTime", createdTime())
                .add("LoadBalancerName", loadBalancerName())
                .add("Scheme", schemeAsString())
                .add("VpcId", vpcId())
                .add("State", state())
                .add("Type", typeAsString())
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null)
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null)
                .add("IpAddressType", ipAddressTypeAsString())
                .add("CustomerOwnedIpv4Pool", customerOwnedIpv4Pool())
                .add("EnforceSecurityGroupInboundRulesOnPrivateLinkTraffic",
                        enforceSecurityGroupInboundRulesOnPrivateLinkTraffic())
                .add("EnablePrefixForIpv6SourceNat", enablePrefixForIpv6SourceNatAsString()).add("IpamPools", ipamPools())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LoadBalancerArn":
            return Optional.ofNullable(clazz.cast(loadBalancerArn()));
        case "DNSName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "CanonicalHostedZoneId":
            return Optional.ofNullable(clazz.cast(canonicalHostedZoneId()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LoadBalancerName":
            return Optional.ofNullable(clazz.cast(loadBalancerName()));
        case "Scheme":
            return Optional.ofNullable(clazz.cast(schemeAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "IpAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "CustomerOwnedIpv4Pool":
            return Optional.ofNullable(clazz.cast(customerOwnedIpv4Pool()));
        case "EnforceSecurityGroupInboundRulesOnPrivateLinkTraffic":
            return Optional.ofNullable(clazz.cast(enforceSecurityGroupInboundRulesOnPrivateLinkTraffic()));
        case "EnablePrefixForIpv6SourceNat":
            return Optional.ofNullable(clazz.cast(enablePrefixForIpv6SourceNatAsString()));
        case "IpamPools":
            return Optional.ofNullable(clazz.cast(ipamPools()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LoadBalancerArn", LOAD_BALANCER_ARN_FIELD);
        map.put("DNSName", DNS_NAME_FIELD);
        map.put("CanonicalHostedZoneId", CANONICAL_HOSTED_ZONE_ID_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("LoadBalancerName", LOAD_BALANCER_NAME_FIELD);
        map.put("Scheme", SCHEME_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("State", STATE_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("AvailabilityZones", AVAILABILITY_ZONES_FIELD);
        map.put("SecurityGroups", SECURITY_GROUPS_FIELD);
        map.put("IpAddressType", IP_ADDRESS_TYPE_FIELD);
        map.put("CustomerOwnedIpv4Pool", CUSTOMER_OWNED_IPV4_POOL_FIELD);
        map.put("EnforceSecurityGroupInboundRulesOnPrivateLinkTraffic",
                ENFORCE_SECURITY_GROUP_INBOUND_RULES_ON_PRIVATE_LINK_TRAFFIC_FIELD);
        map.put("EnablePrefixForIpv6SourceNat", ENABLE_PREFIX_FOR_IPV6_SOURCE_NAT_FIELD);
        map.put("IpamPools", IPAM_POOLS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LoadBalancer, T> g) {
        return obj -> g.apply((LoadBalancer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoadBalancer> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the load balancer.
         * </p>
         * 
         * @param loadBalancerArn
         *        The Amazon Resource Name (ARN) of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerArn(String loadBalancerArn);

        /**
         * <p>
         * The public DNS name of the load balancer.
         * </p>
         * 
         * @param dnsName
         *        The public DNS name of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The ID of the Amazon Route 53 hosted zone associated with the load balancer.
         * </p>
         * 
         * @param canonicalHostedZoneId
         *        The ID of the Amazon Route 53 hosted zone associated with the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder canonicalHostedZoneId(String canonicalHostedZoneId);

        /**
         * <p>
         * The date and time the load balancer was created.
         * </p>
         * 
         * @param createdTime
         *        The date and time the load balancer was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The name of the load balancer.
         * </p>
         * 
         * @param loadBalancerName
         *        The name of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerName(String loadBalancerName);

        /**
         * <p>
         * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing
         * load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
         * balancers can route requests from clients over the internet.
         * </p>
         * <p>
         * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
         * balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers
         * can route requests only from clients with access to the VPC for the load balancer.
         * </p>
         * 
         * @param scheme
         *        The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
         *        Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes.
         *        Therefore, Internet-facing load balancers can route requests from clients over the internet.</p>
         *        <p>
         *        The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal
         *        load balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal
         *        load balancers can route requests only from clients with access to the VPC for the load balancer.
         * @see LoadBalancerSchemeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerSchemeEnum
         */
        Builder scheme(String scheme);

        /**
         * <p>
         * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing
         * load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
         * balancers can route requests from clients over the internet.
         * </p>
         * <p>
         * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
         * balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers
         * can route requests only from clients with access to the VPC for the load balancer.
         * </p>
         * 
         * @param scheme
         *        The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
         *        Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes.
         *        Therefore, Internet-facing load balancers can route requests from clients over the internet.</p>
         *        <p>
         *        The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal
         *        load balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal
         *        load balancers can route requests only from clients with access to the VPC for the load balancer.
         * @see LoadBalancerSchemeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerSchemeEnum
         */
        Builder scheme(LoadBalancerSchemeEnum scheme);

        /**
         * <p>
         * The ID of the VPC for the load balancer.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The state of the load balancer.
         * </p>
         * 
         * @param state
         *        The state of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(LoadBalancerState state);

        /**
         * <p>
         * The state of the load balancer.
         * </p>
         * This is a convenience method that creates an instance of the {@link LoadBalancerState.Builder} avoiding the
         * need to create one manually via {@link LoadBalancerState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LoadBalancerState.Builder#build()} is called immediately and its
         * result is passed to {@link #state(LoadBalancerState)}.
         * 
         * @param state
         *        a consumer that will call methods on {@link LoadBalancerState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #state(LoadBalancerState)
         */
        default Builder state(Consumer<LoadBalancerState.Builder> state) {
            return state(LoadBalancerState.builder().applyMutation(state).build());
        }

        /**
         * <p>
         * The type of load balancer.
         * </p>
         * 
         * @param type
         *        The type of load balancer.
         * @see LoadBalancerTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerTypeEnum
         */
        Builder type(String type);

        /**
         * <p>
         * The type of load balancer.
         * </p>
         * 
         * @param type
         *        The type of load balancer.
         * @see LoadBalancerTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerTypeEnum
         */
        Builder type(LoadBalancerTypeEnum type);

        /**
         * <p>
         * The subnets for the load balancer.
         * </p>
         * 
         * @param availabilityZones
         *        The subnets for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<AvailabilityZone> availabilityZones);

        /**
         * <p>
         * The subnets for the load balancer.
         * </p>
         * 
         * @param availabilityZones
         *        The subnets for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(AvailabilityZone... availabilityZones);

        /**
         * <p>
         * The subnets for the load balancer.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.AvailabilityZone.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.AvailabilityZone#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.AvailabilityZone.Builder#build()} is
         * called immediately and its result is passed to {@link #availabilityZones(List<AvailabilityZone>)}.
         * 
         * @param availabilityZones
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.AvailabilityZone.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availabilityZones(java.util.Collection<AvailabilityZone>)
         */
        Builder availabilityZones(Consumer<AvailabilityZone.Builder>... availabilityZones);

        /**
         * <p>
         * The IDs of the security groups for the load balancer.
         * </p>
         * 
         * @param securityGroups
         *        The IDs of the security groups for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * The IDs of the security groups for the load balancer.
         * </p>
         * 
         * @param securityGroups
         *        The IDs of the security groups for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * <p>
         * The type of IP addresses used for public or private connections by the subnets attached to your load
         * balancer.
         * </p>
         * <p>
         * [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses),
         * <code>dualstack</code> (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code> (public IPv6
         * addresses and private IPv4 and IPv6 addresses).
         * </p>
         * <p>
         * [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4
         * addresses) and <code>dualstack</code> (IPv4 and IPv6 addresses).
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP addresses used for public or private connections by the subnets attached to your load
         *        balancer.</p>
         *        <p>
         *        [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses),
         *        <code>dualstack</code> (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code>
         *        (public IPv6 addresses and private IPv4 and IPv6 addresses).
         *        </p>
         *        <p>
         *        [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4
         *        addresses) and <code>dualstack</code> (IPv4 and IPv6 addresses).
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The type of IP addresses used for public or private connections by the subnets attached to your load
         * balancer.
         * </p>
         * <p>
         * [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses),
         * <code>dualstack</code> (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code> (public IPv6
         * addresses and private IPv4 and IPv6 addresses).
         * </p>
         * <p>
         * [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4
         * addresses) and <code>dualstack</code> (IPv4 and IPv6 addresses).
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP addresses used for public or private connections by the subnets attached to your load
         *        balancer.</p>
         *        <p>
         *        [Application Load Balancers] The possible values are <code>ipv4</code> (IPv4 addresses),
         *        <code>dualstack</code> (IPv4 and IPv6 addresses), and <code>dualstack-without-public-ipv4</code>
         *        (public IPv6 addresses and private IPv4 and IPv6 addresses).
         *        </p>
         *        <p>
         *        [Network Load Balancers and Gateway Load Balancers] The possible values are <code>ipv4</code> (IPv4
         *        addresses) and <code>dualstack</code> (IPv4 and IPv6 addresses).
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(IpAddressType ipAddressType);

        /**
         * <p>
         * [Application Load Balancers on Outposts] The ID of the customer-owned address pool.
         * </p>
         * 
         * @param customerOwnedIpv4Pool
         *        [Application Load Balancers on Outposts] The ID of the customer-owned address pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerOwnedIpv4Pool(String customerOwnedIpv4Pool);

        /**
         * <p>
         * Indicates whether to evaluate inbound security group rules for traffic sent to a Network Load Balancer
         * through Amazon Web Services PrivateLink.
         * </p>
         * 
         * @param enforceSecurityGroupInboundRulesOnPrivateLinkTraffic
         *        Indicates whether to evaluate inbound security group rules for traffic sent to a Network Load Balancer
         *        through Amazon Web Services PrivateLink.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(String enforceSecurityGroupInboundRulesOnPrivateLinkTraffic);

        /**
         * <p>
         * [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet for
         * source NAT. The IP address type must be <code>dualstack</code>. The default value is <code>off</code>.
         * </p>
         * 
         * @param enablePrefixForIpv6SourceNat
         *        [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet
         *        for source NAT. The IP address type must be <code>dualstack</code>. The default value is
         *        <code>off</code>.
         * @see EnablePrefixForIpv6SourceNatEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnablePrefixForIpv6SourceNatEnum
         */
        Builder enablePrefixForIpv6SourceNat(String enablePrefixForIpv6SourceNat);

        /**
         * <p>
         * [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet for
         * source NAT. The IP address type must be <code>dualstack</code>. The default value is <code>off</code>.
         * </p>
         * 
         * @param enablePrefixForIpv6SourceNat
         *        [Network Load Balancers with UDP listeners] Indicates whether to use an IPv6 prefix from each subnet
         *        for source NAT. The IP address type must be <code>dualstack</code>. The default value is
         *        <code>off</code>.
         * @see EnablePrefixForIpv6SourceNatEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnablePrefixForIpv6SourceNatEnum
         */
        Builder enablePrefixForIpv6SourceNat(EnablePrefixForIpv6SourceNatEnum enablePrefixForIpv6SourceNat);

        /**
         * <p>
         * [Application Load Balancers] The IPAM pool in use by the load balancer, if configured.
         * </p>
         * 
         * @param ipamPools
         *        [Application Load Balancers] The IPAM pool in use by the load balancer, if configured.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPools(IpamPools ipamPools);

        /**
         * <p>
         * [Application Load Balancers] The IPAM pool in use by the load balancer, if configured.
         * </p>
         * This is a convenience method that creates an instance of the {@link IpamPools.Builder} avoiding the need to
         * create one manually via {@link IpamPools#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IpamPools.Builder#build()} is called immediately and its result
         * is passed to {@link #ipamPools(IpamPools)}.
         * 
         * @param ipamPools
         *        a consumer that will call methods on {@link IpamPools.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipamPools(IpamPools)
         */
        default Builder ipamPools(Consumer<IpamPools.Builder> ipamPools) {
            return ipamPools(IpamPools.builder().applyMutation(ipamPools).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String loadBalancerArn;

        private String dnsName;

        private String canonicalHostedZoneId;

        private Instant createdTime;

        private String loadBalancerName;

        private String scheme;

        private String vpcId;

        private LoadBalancerState state;

        private String type;

        private List<AvailabilityZone> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private String ipAddressType;

        private String customerOwnedIpv4Pool;

        private String enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;

        private String enablePrefixForIpv6SourceNat;

        private IpamPools ipamPools;

        private BuilderImpl() {
        }

        private BuilderImpl(LoadBalancer model) {
            loadBalancerArn(model.loadBalancerArn);
            dnsName(model.dnsName);
            canonicalHostedZoneId(model.canonicalHostedZoneId);
            createdTime(model.createdTime);
            loadBalancerName(model.loadBalancerName);
            scheme(model.scheme);
            vpcId(model.vpcId);
            state(model.state);
            type(model.type);
            availabilityZones(model.availabilityZones);
            securityGroups(model.securityGroups);
            ipAddressType(model.ipAddressType);
            customerOwnedIpv4Pool(model.customerOwnedIpv4Pool);
            enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(model.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic);
            enablePrefixForIpv6SourceNat(model.enablePrefixForIpv6SourceNat);
            ipamPools(model.ipamPools);
        }

        public final String getLoadBalancerArn() {
            return loadBalancerArn;
        }

        public final void setLoadBalancerArn(String loadBalancerArn) {
            this.loadBalancerArn = loadBalancerArn;
        }

        @Override
        public final Builder loadBalancerArn(String loadBalancerArn) {
            this.loadBalancerArn = loadBalancerArn;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getCanonicalHostedZoneId() {
            return canonicalHostedZoneId;
        }

        public final void setCanonicalHostedZoneId(String canonicalHostedZoneId) {
            this.canonicalHostedZoneId = canonicalHostedZoneId;
        }

        @Override
        public final Builder canonicalHostedZoneId(String canonicalHostedZoneId) {
            this.canonicalHostedZoneId = canonicalHostedZoneId;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final String getLoadBalancerName() {
            return loadBalancerName;
        }

        public final void setLoadBalancerName(String loadBalancerName) {
            this.loadBalancerName = loadBalancerName;
        }

        @Override
        public final Builder loadBalancerName(String loadBalancerName) {
            this.loadBalancerName = loadBalancerName;
            return this;
        }

        public final String getScheme() {
            return scheme;
        }

        public final void setScheme(String scheme) {
            this.scheme = scheme;
        }

        @Override
        public final Builder scheme(String scheme) {
            this.scheme = scheme;
            return this;
        }

        @Override
        public final Builder scheme(LoadBalancerSchemeEnum scheme) {
            this.scheme(scheme == null ? null : scheme.toString());
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final LoadBalancerState.Builder getState() {
            return state != null ? state.toBuilder() : null;
        }

        public final void setState(LoadBalancerState.BuilderImpl state) {
            this.state = state != null ? state.build() : null;
        }

        @Override
        public final Builder state(LoadBalancerState state) {
            this.state = state;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(LoadBalancerTypeEnum type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final List<AvailabilityZone.Builder> getAvailabilityZones() {
            List<AvailabilityZone.Builder> result = AvailabilityZonesCopier.copyToBuilder(this.availabilityZones);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAvailabilityZones(Collection<AvailabilityZone.BuilderImpl> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copyFromBuilder(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<AvailabilityZone> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(AvailabilityZone... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(Consumer<AvailabilityZone.Builder>... availabilityZones) {
            availabilityZones(Stream.of(availabilityZones).map(c -> AvailabilityZone.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupsCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupsCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IpAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final String getCustomerOwnedIpv4Pool() {
            return customerOwnedIpv4Pool;
        }

        public final void setCustomerOwnedIpv4Pool(String customerOwnedIpv4Pool) {
            this.customerOwnedIpv4Pool = customerOwnedIpv4Pool;
        }

        @Override
        public final Builder customerOwnedIpv4Pool(String customerOwnedIpv4Pool) {
            this.customerOwnedIpv4Pool = customerOwnedIpv4Pool;
            return this;
        }

        public final String getEnforceSecurityGroupInboundRulesOnPrivateLinkTraffic() {
            return enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;
        }

        public final void setEnforceSecurityGroupInboundRulesOnPrivateLinkTraffic(
                String enforceSecurityGroupInboundRulesOnPrivateLinkTraffic) {
            this.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic = enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;
        }

        @Override
        public final Builder enforceSecurityGroupInboundRulesOnPrivateLinkTraffic(
                String enforceSecurityGroupInboundRulesOnPrivateLinkTraffic) {
            this.enforceSecurityGroupInboundRulesOnPrivateLinkTraffic = enforceSecurityGroupInboundRulesOnPrivateLinkTraffic;
            return this;
        }

        public final String getEnablePrefixForIpv6SourceNat() {
            return enablePrefixForIpv6SourceNat;
        }

        public final void setEnablePrefixForIpv6SourceNat(String enablePrefixForIpv6SourceNat) {
            this.enablePrefixForIpv6SourceNat = enablePrefixForIpv6SourceNat;
        }

        @Override
        public final Builder enablePrefixForIpv6SourceNat(String enablePrefixForIpv6SourceNat) {
            this.enablePrefixForIpv6SourceNat = enablePrefixForIpv6SourceNat;
            return this;
        }

        @Override
        public final Builder enablePrefixForIpv6SourceNat(EnablePrefixForIpv6SourceNatEnum enablePrefixForIpv6SourceNat) {
            this.enablePrefixForIpv6SourceNat(enablePrefixForIpv6SourceNat == null ? null : enablePrefixForIpv6SourceNat
                    .toString());
            return this;
        }

        public final IpamPools.Builder getIpamPools() {
            return ipamPools != null ? ipamPools.toBuilder() : null;
        }

        public final void setIpamPools(IpamPools.BuilderImpl ipamPools) {
            this.ipamPools = ipamPools != null ? ipamPools.build() : null;
        }

        @Override
        public final Builder ipamPools(IpamPools ipamPools) {
            this.ipamPools = ipamPools;
            return this;
        }

        @Override
        public LoadBalancer build() {
            return new LoadBalancer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
