/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Node group (shard) configuration options when adding or removing replicas. Each node group (shard) configuration has
 * the following members: NodeGroupId, NewReplicaCount, and PreferredAvailabilityZones.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfigureShard implements SdkPojo, Serializable, ToCopyableBuilder<ConfigureShard.Builder, ConfigureShard> {
    private static final SdkField<String> NODE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NodeGroupId").getter(getter(ConfigureShard::nodeGroupId)).setter(setter(Builder::nodeGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupId").build()).build();

    private static final SdkField<Integer> NEW_REPLICA_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NewReplicaCount").getter(getter(ConfigureShard::newReplicaCount))
            .setter(setter(Builder::newReplicaCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewReplicaCount").build()).build();

    private static final SdkField<List<String>> PREFERRED_AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PreferredAvailabilityZones")
            .getter(getter(ConfigureShard::preferredAvailabilityZones))
            .setter(setter(Builder::preferredAvailabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredAvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("PreferredAvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("PreferredAvailabilityZone").build()).build()).build()).build();

    private static final SdkField<List<String>> PREFERRED_OUTPOST_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PreferredOutpostArns")
            .getter(getter(ConfigureShard::preferredOutpostArns))
            .setter(setter(Builder::preferredOutpostArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredOutpostArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("PreferredOutpostArn")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("PreferredOutpostArn").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NODE_GROUP_ID_FIELD,
            NEW_REPLICA_COUNT_FIELD, PREFERRED_AVAILABILITY_ZONES_FIELD, PREFERRED_OUTPOST_ARNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nodeGroupId;

    private final Integer newReplicaCount;

    private final List<String> preferredAvailabilityZones;

    private final List<String> preferredOutpostArns;

    private ConfigureShard(BuilderImpl builder) {
        this.nodeGroupId = builder.nodeGroupId;
        this.newReplicaCount = builder.newReplicaCount;
        this.preferredAvailabilityZones = builder.preferredAvailabilityZones;
        this.preferredOutpostArns = builder.preferredOutpostArns;
    }

    /**
     * <p>
     * The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication groups, the
     * node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's (shard's) id, see <a
     * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding a Shard's Id</a>.
     * </p>
     * 
     * @return The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication
     *         groups, the node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's (shard's)
     *         id, see <a href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding
     *         a Shard's Id</a>.
     */
    public final String nodeGroupId() {
        return nodeGroupId;
    }

    /**
     * <p>
     * The number of replicas you want in this node group at the end of this operation. The maximum value for
     * <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group you are
     * working with.
     * </p>
     * <p>
     * The minimum number of replicas in a shard or replication group is:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Redis (cluster mode disabled)
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Multi-AZ: 1
     * </p>
     * </li>
     * <li>
     * <p>
     * If Multi-AZ: 0
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary node fails)
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of replicas you want in this node group at the end of this operation. The maximum value for
     *         <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group you
     *         are working with.</p>
     *         <p>
     *         The minimum number of replicas in a shard or replication group is:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Redis (cluster mode disabled)
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Multi-AZ: 1
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If Multi-AZ: 0
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary
     *         node fails)
     *         </p>
     *         </li>
     */
    public final Integer newReplicaCount() {
        return newReplicaCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the PreferredAvailabilityZones property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasPreferredAvailabilityZones() {
        return preferredAvailabilityZones != null && !(preferredAvailabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the replication
     * group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must equal the value of
     * <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
     * <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for each of the
     * replicas.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPreferredAvailabilityZones} method.
     * </p>
     * 
     * @return A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
     *         replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must
     *         equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
     *         <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for
     *         each of the replicas.
     */
    public final List<String> preferredAvailabilityZones() {
        return preferredAvailabilityZones;
    }

    /**
     * For responses, this returns true if the service returned a value for the PreferredOutpostArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPreferredOutpostArns() {
        return preferredOutpostArns != null && !(preferredOutpostArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The outpost ARNs in which the cache cluster is created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPreferredOutpostArns} method.
     * </p>
     * 
     * @return The outpost ARNs in which the cache cluster is created.
     */
    public final List<String> preferredOutpostArns() {
        return preferredOutpostArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nodeGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(newReplicaCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasPreferredAvailabilityZones() ? preferredAvailabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPreferredOutpostArns() ? preferredOutpostArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigureShard)) {
            return false;
        }
        ConfigureShard other = (ConfigureShard) obj;
        return Objects.equals(nodeGroupId(), other.nodeGroupId()) && Objects.equals(newReplicaCount(), other.newReplicaCount())
                && hasPreferredAvailabilityZones() == other.hasPreferredAvailabilityZones()
                && Objects.equals(preferredAvailabilityZones(), other.preferredAvailabilityZones())
                && hasPreferredOutpostArns() == other.hasPreferredOutpostArns()
                && Objects.equals(preferredOutpostArns(), other.preferredOutpostArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfigureShard").add("NodeGroupId", nodeGroupId()).add("NewReplicaCount", newReplicaCount())
                .add("PreferredAvailabilityZones", hasPreferredAvailabilityZones() ? preferredAvailabilityZones() : null)
                .add("PreferredOutpostArns", hasPreferredOutpostArns() ? preferredOutpostArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NodeGroupId":
            return Optional.ofNullable(clazz.cast(nodeGroupId()));
        case "NewReplicaCount":
            return Optional.ofNullable(clazz.cast(newReplicaCount()));
        case "PreferredAvailabilityZones":
            return Optional.ofNullable(clazz.cast(preferredAvailabilityZones()));
        case "PreferredOutpostArns":
            return Optional.ofNullable(clazz.cast(preferredOutpostArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConfigureShard, T> g) {
        return obj -> g.apply((ConfigureShard) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfigureShard> {
        /**
         * <p>
         * The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication groups,
         * the node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's (shard's) id, see <a
         * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding a Shard's
         * Id</a>.
         * </p>
         * 
         * @param nodeGroupId
         *        The 4-digit id for the node group you are configuring. For Redis (cluster mode disabled) replication
         *        groups, the node group id is always 0001. To find a Redis (cluster mode enabled)'s node group's
         *        (shard's) id, see <a
         *        href="https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/shard-find-id.html">Finding a
         *        Shard's Id</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupId(String nodeGroupId);

        /**
         * <p>
         * The number of replicas you want in this node group at the end of this operation. The maximum value for
         * <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group you are
         * working with.
         * </p>
         * <p>
         * The minimum number of replicas in a shard or replication group is:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Redis (cluster mode disabled)
         * </p>
         * <ul>
         * <li>
         * <p>
         * If Multi-AZ: 1
         * </p>
         * </li>
         * <li>
         * <p>
         * If Multi-AZ: 0
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary node
         * fails)
         * </p>
         * </li>
         * </ul>
         * 
         * @param newReplicaCount
         *        The number of replicas you want in this node group at the end of this operation. The maximum value for
         *        <code>NewReplicaCount</code> is 5. The minimum value depends upon the type of Redis replication group
         *        you are working with.</p>
         *        <p>
         *        The minimum number of replicas in a shard or replication group is:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Redis (cluster mode disabled)
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If Multi-AZ: 1
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If Multi-AZ: 0
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Redis (cluster mode enabled): 0 (though you will not be able to failover to a replica if your primary
         *        node fails)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newReplicaCount(Integer newReplicaCount);

        /**
         * <p>
         * A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         * replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must
         * equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
         * <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for each of
         * the replicas.
         * </p>
         * 
         * @param preferredAvailabilityZones
         *        A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         *        replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values
         *        must equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this
         *        member of <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability
         *        zone for each of the replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZones(Collection<String> preferredAvailabilityZones);

        /**
         * <p>
         * A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         * replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values must
         * equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this member of
         * <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability zone for each of
         * the replicas.
         * </p>
         * 
         * @param preferredAvailabilityZones
         *        A list of <code>PreferredAvailabilityZone</code> strings that specify which availability zones the
         *        replication group's nodes are to be in. The nummber of <code>PreferredAvailabilityZone</code> values
         *        must equal the value of <code>NewReplicaCount</code> plus 1 to account for the primary node. If this
         *        member of <code>ReplicaConfiguration</code> is omitted, ElastiCache for Redis selects the availability
         *        zone for each of the replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZones(String... preferredAvailabilityZones);

        /**
         * <p>
         * The outpost ARNs in which the cache cluster is created.
         * </p>
         * 
         * @param preferredOutpostArns
         *        The outpost ARNs in which the cache cluster is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredOutpostArns(Collection<String> preferredOutpostArns);

        /**
         * <p>
         * The outpost ARNs in which the cache cluster is created.
         * </p>
         * 
         * @param preferredOutpostArns
         *        The outpost ARNs in which the cache cluster is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredOutpostArns(String... preferredOutpostArns);
    }

    static final class BuilderImpl implements Builder {
        private String nodeGroupId;

        private Integer newReplicaCount;

        private List<String> preferredAvailabilityZones = DefaultSdkAutoConstructList.getInstance();

        private List<String> preferredOutpostArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigureShard model) {
            nodeGroupId(model.nodeGroupId);
            newReplicaCount(model.newReplicaCount);
            preferredAvailabilityZones(model.preferredAvailabilityZones);
            preferredOutpostArns(model.preferredOutpostArns);
        }

        public final String getNodeGroupId() {
            return nodeGroupId;
        }

        public final void setNodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
        }

        @Override
        public final Builder nodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
            return this;
        }

        public final Integer getNewReplicaCount() {
            return newReplicaCount;
        }

        public final void setNewReplicaCount(Integer newReplicaCount) {
            this.newReplicaCount = newReplicaCount;
        }

        @Override
        public final Builder newReplicaCount(Integer newReplicaCount) {
            this.newReplicaCount = newReplicaCount;
            return this;
        }

        public final Collection<String> getPreferredAvailabilityZones() {
            if (preferredAvailabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return preferredAvailabilityZones;
        }

        public final void setPreferredAvailabilityZones(Collection<String> preferredAvailabilityZones) {
            this.preferredAvailabilityZones = PreferredAvailabilityZoneListCopier.copy(preferredAvailabilityZones);
        }

        @Override
        public final Builder preferredAvailabilityZones(Collection<String> preferredAvailabilityZones) {
            this.preferredAvailabilityZones = PreferredAvailabilityZoneListCopier.copy(preferredAvailabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredAvailabilityZones(String... preferredAvailabilityZones) {
            preferredAvailabilityZones(Arrays.asList(preferredAvailabilityZones));
            return this;
        }

        public final Collection<String> getPreferredOutpostArns() {
            if (preferredOutpostArns instanceof SdkAutoConstructList) {
                return null;
            }
            return preferredOutpostArns;
        }

        public final void setPreferredOutpostArns(Collection<String> preferredOutpostArns) {
            this.preferredOutpostArns = PreferredOutpostArnListCopier.copy(preferredOutpostArns);
        }

        @Override
        public final Builder preferredOutpostArns(Collection<String> preferredOutpostArns) {
            this.preferredOutpostArns = PreferredOutpostArnListCopier.copy(preferredOutpostArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredOutpostArns(String... preferredOutpostArns) {
            preferredOutpostArns(Arrays.asList(preferredOutpostArns));
            return this;
        }

        @Override
        public ConfigureShard build() {
            return new ConfigureShard(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
