/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartTaskRequest extends EcsRequest implements ToCopyableBuilder<StartTaskRequest.Builder, StartTaskRequest> {
    private static final SdkField<String> CLUSTER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("cluster")
            .getter(getter(StartTaskRequest::cluster)).setter(setter(Builder::cluster))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cluster").build()).build();

    private static final SdkField<List<String>> CONTAINER_INSTANCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("containerInstances")
            .getter(getter(StartTaskRequest::containerInstances))
            .setter(setter(Builder::containerInstances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerInstances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ENABLE_ECS_MANAGED_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enableECSManagedTags").getter(getter(StartTaskRequest::enableECSManagedTags))
            .setter(setter(Builder::enableECSManagedTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableECSManagedTags").build())
            .build();

    private static final SdkField<Boolean> ENABLE_EXECUTE_COMMAND_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enableExecuteCommand").getter(getter(StartTaskRequest::enableExecuteCommand))
            .setter(setter(Builder::enableExecuteCommand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableExecuteCommand").build())
            .build();

    private static final SdkField<String> GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("group")
            .getter(getter(StartTaskRequest::group)).setter(setter(Builder::group))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("group").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkConfiguration")
            .getter(getter(StartTaskRequest::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<TaskOverride> OVERRIDES_FIELD = SdkField.<TaskOverride> builder(MarshallingType.SDK_POJO)
            .memberName("overrides").getter(getter(StartTaskRequest::overrides)).setter(setter(Builder::overrides))
            .constructor(TaskOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("overrides").build()).build();

    private static final SdkField<String> PROPAGATE_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("propagateTags").getter(getter(StartTaskRequest::propagateTagsAsString))
            .setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propagateTags").build()).build();

    private static final SdkField<String> REFERENCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("referenceId").getter(getter(StartTaskRequest::referenceId)).setter(setter(Builder::referenceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("referenceId").build()).build();

    private static final SdkField<String> STARTED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("startedBy").getter(getter(StartTaskRequest::startedBy)).setter(setter(Builder::startedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedBy").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(StartTaskRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TASK_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskDefinition").getter(getter(StartTaskRequest::taskDefinition))
            .setter(setter(Builder::taskDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskDefinition").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_FIELD,
            CONTAINER_INSTANCES_FIELD, ENABLE_ECS_MANAGED_TAGS_FIELD, ENABLE_EXECUTE_COMMAND_FIELD, GROUP_FIELD,
            NETWORK_CONFIGURATION_FIELD, OVERRIDES_FIELD, PROPAGATE_TAGS_FIELD, REFERENCE_ID_FIELD, STARTED_BY_FIELD, TAGS_FIELD,
            TASK_DEFINITION_FIELD));

    private final String cluster;

    private final List<String> containerInstances;

    private final Boolean enableECSManagedTags;

    private final Boolean enableExecuteCommand;

    private final String group;

    private final NetworkConfiguration networkConfiguration;

    private final TaskOverride overrides;

    private final String propagateTags;

    private final String referenceId;

    private final String startedBy;

    private final List<Tag> tags;

    private final String taskDefinition;

    private StartTaskRequest(BuilderImpl builder) {
        super(builder);
        this.cluster = builder.cluster;
        this.containerInstances = builder.containerInstances;
        this.enableECSManagedTags = builder.enableECSManagedTags;
        this.enableExecuteCommand = builder.enableExecuteCommand;
        this.group = builder.group;
        this.networkConfiguration = builder.networkConfiguration;
        this.overrides = builder.overrides;
        this.propagateTags = builder.propagateTags;
        this.referenceId = builder.referenceId;
        this.startedBy = builder.startedBy;
        this.tags = builder.tags;
        this.taskDefinition = builder.taskDefinition;
    }

    /**
     * <p>
     * The short name or full Amazon Resource Name (ARN) of the cluster where to start your task. If you do not specify
     * a cluster, the default cluster is assumed.
     * </p>
     * 
     * @return The short name or full Amazon Resource Name (ARN) of the cluster where to start your task. If you do not
     *         specify a cluster, the default cluster is assumed.
     */
    public final String cluster() {
        return cluster;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerInstances property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerInstances() {
        return containerInstances != null && !(containerInstances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The container instance IDs or full ARN entries for the container instances where you would like to place your
     * task. You can specify up to 10 container instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerInstances} method.
     * </p>
     * 
     * @return The container instance IDs or full ARN entries for the container instances where you would like to place
     *         your task. You can specify up to 10 container instances.
     */
    public final List<String> containerInstances() {
        return containerInstances;
    }

    /**
     * <p>
     * Specifies whether to use Amazon ECS managed tags for the task. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon ECS
     * Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return Specifies whether to use Amazon ECS managed tags for the task. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your
     *         Amazon ECS Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final Boolean enableECSManagedTags() {
        return enableECSManagedTags;
    }

    /**
     * <p>
     * Whether or not the execute command functionality is turned on for the task. If <code>true</code>, this turns on
     * the execute command functionality on all containers in the task.
     * </p>
     * 
     * @return Whether or not the execute command functionality is turned on for the task. If <code>true</code>, this
     *         turns on the execute command functionality on all containers in the task.
     */
    public final Boolean enableExecuteCommand() {
        return enableExecuteCommand;
    }

    /**
     * <p>
     * The name of the task group to associate with the task. The default value is the family name of the task
     * definition (for example, family:my-family-name).
     * </p>
     * 
     * @return The name of the task group to associate with the task. The default value is the family name of the task
     *         definition (for example, family:my-family-name).
     */
    public final String group() {
        return group;
    }

    /**
     * <p>
     * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
     * using the <code>awsvpc</code> networking mode.
     * </p>
     * 
     * @return The VPC subnet and security group configuration for tasks that receive their own elastic network
     *         interface by using the <code>awsvpc</code> networking mode.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * A list of container overrides in JSON format that specify the name of a container in the specified task
     * definition and the overrides it receives. You can override the default command for a container (that's specified
     * in the task definition or Docker image) with a <code>command</code> override. You can also override existing
     * environment variables (that are specified in the task definition or Docker image) on a container or add new
     * environment variables to it with an <code>environment</code> override.
     * </p>
     * <note>
     * <p>
     * A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters of the
     * override structure.
     * </p>
     * </note>
     * 
     * @return A list of container overrides in JSON format that specify the name of a container in the specified task
     *         definition and the overrides it receives. You can override the default command for a container (that's
     *         specified in the task definition or Docker image) with a <code>command</code> override. You can also
     *         override existing environment variables (that are specified in the task definition or Docker image) on a
     *         container or add new environment variables to it with an <code>environment</code> override.</p> <note>
     *         <p>
     *         A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters
     *         of the override structure.
     *         </p>
     */
    public final TaskOverride overrides() {
        return overrides;
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
     * specified, the tags aren't propagated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the task definition or the service to the task. If no value
     *         is specified, the tags aren't propagated.
     * @see PropagateTags
     */
    public final PropagateTags propagateTags() {
        return PropagateTags.fromValue(propagateTags);
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
     * specified, the tags aren't propagated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the task definition or the service to the task. If no value
     *         is specified, the tags aren't propagated.
     * @see PropagateTags
     */
    public final String propagateTagsAsString() {
        return propagateTags;
    }

    /**
     * <p>
     * The reference ID to use for the task.
     * </p>
     * 
     * @return The reference ID to use for the task.
     */
    public final String referenceId() {
        return referenceId;
    }

    /**
     * <p>
     * An optional tag specified when a task is started. For example, if you automatically trigger a task to run a batch
     * process job, you could apply a unique identifier for that job to your task with the <code>startedBy</code>
     * parameter. You can then identify which tasks belong to that job by filtering the results of a <a>ListTasks</a>
     * call with the <code>startedBy</code> value. Up to 36 letters (uppercase and lowercase), numbers, hyphens (-), and
     * underscores (_) are allowed.
     * </p>
     * <p>
     * If a task is started by an Amazon ECS service, the <code>startedBy</code> parameter contains the deployment ID of
     * the service that starts it.
     * </p>
     * 
     * @return An optional tag specified when a task is started. For example, if you automatically trigger a task to run
     *         a batch process job, you could apply a unique identifier for that job to your task with the
     *         <code>startedBy</code> parameter. You can then identify which tasks belong to that job by filtering the
     *         results of a <a>ListTasks</a> call with the <code>startedBy</code> value. Up to 36 letters (uppercase and
     *         lowercase), numbers, hyphens (-), and underscores (_) are allowed.</p>
     *         <p>
     *         If a task is started by an Amazon ECS service, the <code>startedBy</code> parameter contains the
     *         deployment ID of the service that starts it.
     */
    public final String startedBy() {
        return startedBy;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key and
     * an optional value, both of which you define.
     * </p>
     * <p>
     * The following basic restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per resource - 50
     * </p>
     * </li>
     * <li>
     * <p>
     * For each resource, each tag key must be unique, and each tag key can have only one value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length - 128 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length - 256 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * If your tagging schema is used across multiple services and resources, remember that other services may have
     * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable
     * in UTF-8, and the following characters: + - = . _ : / @.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix for
     * either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys or values
     * with this prefix. Tags with this prefix do not count against your tags per resource limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a
     *         key and an optional value, both of which you define.</p>
     *         <p>
     *         The following basic restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per resource - 50
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For each resource, each tag key must be unique, and each tag key can have only one value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length - 128 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length - 256 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your tagging schema is used across multiple services and resources, remember that other services may
     *         have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
     *         representable in UTF-8, and the following characters: + - = . _ : / @.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case-sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
     *         prefix for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete
     *         tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource
     *         limit.
     *         </p>
     *         </li>
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the task
     * definition to start. If a <code>revision</code> isn't specified, the latest <code>ACTIVE</code> revision is used.
     * </p>
     * 
     * @return The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the task
     *         definition to start. If a <code>revision</code> isn't specified, the latest <code>ACTIVE</code> revision
     *         is used.
     */
    public final String taskDefinition() {
        return taskDefinition;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(cluster());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerInstances() ? containerInstances() : null);
        hashCode = 31 * hashCode + Objects.hashCode(enableECSManagedTags());
        hashCode = 31 * hashCode + Objects.hashCode(enableExecuteCommand());
        hashCode = 31 * hashCode + Objects.hashCode(group());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(overrides());
        hashCode = 31 * hashCode + Objects.hashCode(propagateTagsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(referenceId());
        hashCode = 31 * hashCode + Objects.hashCode(startedBy());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinition());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartTaskRequest)) {
            return false;
        }
        StartTaskRequest other = (StartTaskRequest) obj;
        return Objects.equals(cluster(), other.cluster()) && hasContainerInstances() == other.hasContainerInstances()
                && Objects.equals(containerInstances(), other.containerInstances())
                && Objects.equals(enableECSManagedTags(), other.enableECSManagedTags())
                && Objects.equals(enableExecuteCommand(), other.enableExecuteCommand()) && Objects.equals(group(), other.group())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(overrides(), other.overrides())
                && Objects.equals(propagateTagsAsString(), other.propagateTagsAsString())
                && Objects.equals(referenceId(), other.referenceId()) && Objects.equals(startedBy(), other.startedBy())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(taskDefinition(), other.taskDefinition());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartTaskRequest").add("Cluster", cluster())
                .add("ContainerInstances", hasContainerInstances() ? containerInstances() : null)
                .add("EnableECSManagedTags", enableECSManagedTags()).add("EnableExecuteCommand", enableExecuteCommand())
                .add("Group", group()).add("NetworkConfiguration", networkConfiguration()).add("Overrides", overrides())
                .add("PropagateTags", propagateTagsAsString()).add("ReferenceId", referenceId()).add("StartedBy", startedBy())
                .add("Tags", hasTags() ? tags() : null).add("TaskDefinition", taskDefinition()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cluster":
            return Optional.ofNullable(clazz.cast(cluster()));
        case "containerInstances":
            return Optional.ofNullable(clazz.cast(containerInstances()));
        case "enableECSManagedTags":
            return Optional.ofNullable(clazz.cast(enableECSManagedTags()));
        case "enableExecuteCommand":
            return Optional.ofNullable(clazz.cast(enableExecuteCommand()));
        case "group":
            return Optional.ofNullable(clazz.cast(group()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "overrides":
            return Optional.ofNullable(clazz.cast(overrides()));
        case "propagateTags":
            return Optional.ofNullable(clazz.cast(propagateTagsAsString()));
        case "referenceId":
            return Optional.ofNullable(clazz.cast(referenceId()));
        case "startedBy":
            return Optional.ofNullable(clazz.cast(startedBy()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "taskDefinition":
            return Optional.ofNullable(clazz.cast(taskDefinition()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartTaskRequest, T> g) {
        return obj -> g.apply((StartTaskRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EcsRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartTaskRequest> {
        /**
         * <p>
         * The short name or full Amazon Resource Name (ARN) of the cluster where to start your task. If you do not
         * specify a cluster, the default cluster is assumed.
         * </p>
         * 
         * @param cluster
         *        The short name or full Amazon Resource Name (ARN) of the cluster where to start your task. If you do
         *        not specify a cluster, the default cluster is assumed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cluster(String cluster);

        /**
         * <p>
         * The container instance IDs or full ARN entries for the container instances where you would like to place your
         * task. You can specify up to 10 container instances.
         * </p>
         * 
         * @param containerInstances
         *        The container instance IDs or full ARN entries for the container instances where you would like to
         *        place your task. You can specify up to 10 container instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstances(Collection<String> containerInstances);

        /**
         * <p>
         * The container instance IDs or full ARN entries for the container instances where you would like to place your
         * task. You can specify up to 10 container instances.
         * </p>
         * 
         * @param containerInstances
         *        The container instance IDs or full ARN entries for the container instances where you would like to
         *        place your task. You can specify up to 10 container instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstances(String... containerInstances);

        /**
         * <p>
         * Specifies whether to use Amazon ECS managed tags for the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon
         * ECS Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param enableECSManagedTags
         *        Specifies whether to use Amazon ECS managed tags for the task. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your
         *        Amazon ECS Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableECSManagedTags(Boolean enableECSManagedTags);

        /**
         * <p>
         * Whether or not the execute command functionality is turned on for the task. If <code>true</code>, this turns
         * on the execute command functionality on all containers in the task.
         * </p>
         * 
         * @param enableExecuteCommand
         *        Whether or not the execute command functionality is turned on for the task. If <code>true</code>, this
         *        turns on the execute command functionality on all containers in the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableExecuteCommand(Boolean enableExecuteCommand);

        /**
         * <p>
         * The name of the task group to associate with the task. The default value is the family name of the task
         * definition (for example, family:my-family-name).
         * </p>
         * 
         * @param group
         *        The name of the task group to associate with the task. The default value is the family name of the
         *        task definition (for example, family:my-family-name).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder group(String group);

        /**
         * <p>
         * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
         * using the <code>awsvpc</code> networking mode.
         * </p>
         * 
         * @param networkConfiguration
         *        The VPC subnet and security group configuration for tasks that receive their own elastic network
         *        interface by using the <code>awsvpc</code> networking mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
         * using the <code>awsvpc</code> networking mode.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * A list of container overrides in JSON format that specify the name of a container in the specified task
         * definition and the overrides it receives. You can override the default command for a container (that's
         * specified in the task definition or Docker image) with a <code>command</code> override. You can also override
         * existing environment variables (that are specified in the task definition or Docker image) on a container or
         * add new environment variables to it with an <code>environment</code> override.
         * </p>
         * <note>
         * <p>
         * A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters of
         * the override structure.
         * </p>
         * </note>
         * 
         * @param overrides
         *        A list of container overrides in JSON format that specify the name of a container in the specified
         *        task definition and the overrides it receives. You can override the default command for a container
         *        (that's specified in the task definition or Docker image) with a <code>command</code> override. You
         *        can also override existing environment variables (that are specified in the task definition or Docker
         *        image) on a container or add new environment variables to it with an <code>environment</code>
         *        override.</p> <note>
         *        <p>
         *        A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting
         *        characters of the override structure.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(TaskOverride overrides);

        /**
         * <p>
         * A list of container overrides in JSON format that specify the name of a container in the specified task
         * definition and the overrides it receives. You can override the default command for a container (that's
         * specified in the task definition or Docker image) with a <code>command</code> override. You can also override
         * existing environment variables (that are specified in the task definition or Docker image) on a container or
         * add new environment variables to it with an <code>environment</code> override.
         * </p>
         * <note>
         * <p>
         * A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters of
         * the override structure.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link TaskOverride.Builder} avoiding
         * the need to create one manually via {@link TaskOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TaskOverride.Builder#build()} is called immediately and its
         * result is passed to {@link #overrides(TaskOverride)}.
         * 
         * @param overrides
         *        a consumer that will call methods on {@link TaskOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrides(TaskOverride)
         */
        default Builder overrides(Consumer<TaskOverride.Builder> overrides) {
            return overrides(TaskOverride.builder().applyMutation(overrides).build());
        }

        /**
         * <p>
         * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
         * specified, the tags aren't propagated.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the task definition or the service to the task. If no
         *        value is specified, the tags aren't propagated.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(String propagateTags);

        /**
         * <p>
         * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
         * specified, the tags aren't propagated.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the task definition or the service to the task. If no
         *        value is specified, the tags aren't propagated.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(PropagateTags propagateTags);

        /**
         * <p>
         * The reference ID to use for the task.
         * </p>
         * 
         * @param referenceId
         *        The reference ID to use for the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceId(String referenceId);

        /**
         * <p>
         * An optional tag specified when a task is started. For example, if you automatically trigger a task to run a
         * batch process job, you could apply a unique identifier for that job to your task with the
         * <code>startedBy</code> parameter. You can then identify which tasks belong to that job by filtering the
         * results of a <a>ListTasks</a> call with the <code>startedBy</code> value. Up to 36 letters (uppercase and
         * lowercase), numbers, hyphens (-), and underscores (_) are allowed.
         * </p>
         * <p>
         * If a task is started by an Amazon ECS service, the <code>startedBy</code> parameter contains the deployment
         * ID of the service that starts it.
         * </p>
         * 
         * @param startedBy
         *        An optional tag specified when a task is started. For example, if you automatically trigger a task to
         *        run a batch process job, you could apply a unique identifier for that job to your task with the
         *        <code>startedBy</code> parameter. You can then identify which tasks belong to that job by filtering
         *        the results of a <a>ListTasks</a> call with the <code>startedBy</code> value. Up to 36 letters
         *        (uppercase and lowercase), numbers, hyphens (-), and underscores (_) are allowed.</p>
         *        <p>
         *        If a task is started by an Amazon ECS service, the <code>startedBy</code> parameter contains the
         *        deployment ID of the service that starts it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedBy(String startedBy);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys
         * or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The metadata that you apply to the task to help you categorize and organize them. Each tag consists of
         *        a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or
         *        delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per
         *        resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys
         * or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The metadata that you apply to the task to help you categorize and organize them. Each tag consists of
         *        a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or
         *        delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per
         *        resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys
         * or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the task
         * definition to start. If a <code>revision</code> isn't specified, the latest <code>ACTIVE</code> revision is
         * used.
         * </p>
         * 
         * @param taskDefinition
         *        The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the
         *        task definition to start. If a <code>revision</code> isn't specified, the latest <code>ACTIVE</code>
         *        revision is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinition(String taskDefinition);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EcsRequest.BuilderImpl implements Builder {
        private String cluster;

        private List<String> containerInstances = DefaultSdkAutoConstructList.getInstance();

        private Boolean enableECSManagedTags;

        private Boolean enableExecuteCommand;

        private String group;

        private NetworkConfiguration networkConfiguration;

        private TaskOverride overrides;

        private String propagateTags;

        private String referenceId;

        private String startedBy;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String taskDefinition;

        private BuilderImpl() {
        }

        private BuilderImpl(StartTaskRequest model) {
            super(model);
            cluster(model.cluster);
            containerInstances(model.containerInstances);
            enableECSManagedTags(model.enableECSManagedTags);
            enableExecuteCommand(model.enableExecuteCommand);
            group(model.group);
            networkConfiguration(model.networkConfiguration);
            overrides(model.overrides);
            propagateTags(model.propagateTags);
            referenceId(model.referenceId);
            startedBy(model.startedBy);
            tags(model.tags);
            taskDefinition(model.taskDefinition);
        }

        public final String getCluster() {
            return cluster;
        }

        public final void setCluster(String cluster) {
            this.cluster = cluster;
        }

        @Override
        public final Builder cluster(String cluster) {
            this.cluster = cluster;
            return this;
        }

        public final Collection<String> getContainerInstances() {
            if (containerInstances instanceof SdkAutoConstructList) {
                return null;
            }
            return containerInstances;
        }

        public final void setContainerInstances(Collection<String> containerInstances) {
            this.containerInstances = StringListCopier.copy(containerInstances);
        }

        @Override
        public final Builder containerInstances(Collection<String> containerInstances) {
            this.containerInstances = StringListCopier.copy(containerInstances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerInstances(String... containerInstances) {
            containerInstances(Arrays.asList(containerInstances));
            return this;
        }

        public final Boolean getEnableECSManagedTags() {
            return enableECSManagedTags;
        }

        public final void setEnableECSManagedTags(Boolean enableECSManagedTags) {
            this.enableECSManagedTags = enableECSManagedTags;
        }

        @Override
        public final Builder enableECSManagedTags(Boolean enableECSManagedTags) {
            this.enableECSManagedTags = enableECSManagedTags;
            return this;
        }

        public final Boolean getEnableExecuteCommand() {
            return enableExecuteCommand;
        }

        public final void setEnableExecuteCommand(Boolean enableExecuteCommand) {
            this.enableExecuteCommand = enableExecuteCommand;
        }

        @Override
        public final Builder enableExecuteCommand(Boolean enableExecuteCommand) {
            this.enableExecuteCommand = enableExecuteCommand;
            return this;
        }

        public final String getGroup() {
            return group;
        }

        public final void setGroup(String group) {
            this.group = group;
        }

        @Override
        public final Builder group(String group) {
            this.group = group;
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final TaskOverride.Builder getOverrides() {
            return overrides != null ? overrides.toBuilder() : null;
        }

        public final void setOverrides(TaskOverride.BuilderImpl overrides) {
            this.overrides = overrides != null ? overrides.build() : null;
        }

        @Override
        public final Builder overrides(TaskOverride overrides) {
            this.overrides = overrides;
            return this;
        }

        public final String getPropagateTags() {
            return propagateTags;
        }

        public final void setPropagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public final Builder propagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        @Override
        public final Builder propagateTags(PropagateTags propagateTags) {
            this.propagateTags(propagateTags == null ? null : propagateTags.toString());
            return this;
        }

        public final String getReferenceId() {
            return referenceId;
        }

        public final void setReferenceId(String referenceId) {
            this.referenceId = referenceId;
        }

        @Override
        public final Builder referenceId(String referenceId) {
            this.referenceId = referenceId;
            return this;
        }

        public final String getStartedBy() {
            return startedBy;
        }

        public final void setStartedBy(String startedBy) {
            this.startedBy = startedBy;
        }

        @Override
        public final Builder startedBy(String startedBy) {
            this.startedBy = startedBy;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getTaskDefinition() {
            return taskDefinition;
        }

        public final void setTaskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
        }

        @Override
        public final Builder taskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartTaskRequest build() {
            return new StartTaskRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
