/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a key pair.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeyPairInfo implements SdkPojo, Serializable, ToCopyableBuilder<KeyPairInfo.Builder, KeyPairInfo> {
    private static final SdkField<String> KEY_PAIR_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyPairId")
            .getter(getter(KeyPairInfo::keyPairId))
            .setter(setter(Builder::keyPairId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyPairId")
                    .unmarshallLocationName("keyPairId").build()).build();

    private static final SdkField<String> KEY_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyType")
            .getter(getter(KeyPairInfo::keyTypeAsString))
            .setter(setter(Builder::keyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyType")
                    .unmarshallLocationName("keyType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(KeyPairInfo::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> PUBLIC_KEY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PublicKey")
            .getter(getter(KeyPairInfo::publicKey))
            .setter(setter(Builder::publicKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicKey")
                    .unmarshallLocationName("publicKey").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime")
            .getter(getter(KeyPairInfo::createTime))
            .setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime")
                    .unmarshallLocationName("createTime").build()).build();

    private static final SdkField<String> KEY_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyName")
            .getter(getter(KeyPairInfo::keyName))
            .setter(setter(Builder::keyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyName")
                    .unmarshallLocationName("keyName").build()).build();

    private static final SdkField<String> KEY_FINGERPRINT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KeyFingerprint")
            .getter(getter(KeyPairInfo::keyFingerprint))
            .setter(setter(Builder::keyFingerprint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyFingerprint")
                    .unmarshallLocationName("keyFingerprint").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_PAIR_ID_FIELD,
            KEY_TYPE_FIELD, TAGS_FIELD, PUBLIC_KEY_FIELD, CREATE_TIME_FIELD, KEY_NAME_FIELD, KEY_FINGERPRINT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("KeyPairId", KEY_PAIR_ID_FIELD);
                    put("KeyType", KEY_TYPE_FIELD);
                    put("TagSet", TAGS_FIELD);
                    put("PublicKey", PUBLIC_KEY_FIELD);
                    put("CreateTime", CREATE_TIME_FIELD);
                    put("KeyName", KEY_NAME_FIELD);
                    put("KeyFingerprint", KEY_FINGERPRINT_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String keyPairId;

    private final String keyType;

    private final List<Tag> tags;

    private final String publicKey;

    private final Instant createTime;

    private final String keyName;

    private final String keyFingerprint;

    private KeyPairInfo(BuilderImpl builder) {
        this.keyPairId = builder.keyPairId;
        this.keyType = builder.keyType;
        this.tags = builder.tags;
        this.publicKey = builder.publicKey;
        this.createTime = builder.createTime;
        this.keyName = builder.keyName;
        this.keyFingerprint = builder.keyFingerprint;
    }

    /**
     * <p>
     * The ID of the key pair.
     * </p>
     * 
     * @return The ID of the key pair.
     */
    public final String keyPairId() {
        return keyPairId;
    }

    /**
     * <p>
     * The type of key pair.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyType} will
     * return {@link KeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyTypeAsString}.
     * </p>
     * 
     * @return The type of key pair.
     * @see KeyType
     */
    public final KeyType keyType() {
        return KeyType.fromValue(keyType);
    }

    /**
     * <p>
     * The type of key pair.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyType} will
     * return {@link KeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyTypeAsString}.
     * </p>
     * 
     * @return The type of key pair.
     * @see KeyType
     */
    public final String keyTypeAsString() {
        return keyType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags applied to the key pair.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags applied to the key pair.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The public key material.
     * </p>
     * 
     * @return The public key material.
     */
    public final String publicKey() {
        return publicKey;
    }

    /**
     * <p>
     * If you used Amazon EC2 to create the key pair, this is the date and time when the key was created, in <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the UTC time
     * zone.
     * </p>
     * <p>
     * If you imported an existing key pair to Amazon EC2, this is the date and time the key was imported, in <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the UTC time
     * zone.
     * </p>
     * 
     * @return If you used Amazon EC2 to create the key pair, this is the date and time when the key was created, in <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the UTC
     *         time zone.</p>
     *         <p>
     *         If you imported an existing key pair to Amazon EC2, this is the date and time the key was imported, in <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the UTC
     *         time zone.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The name of the key pair.
     * </p>
     * 
     * @return The name of the key pair.
     */
    public final String keyName() {
        return keyName;
    }

    /**
     * <p>
     * If you used <a>CreateKeyPair</a> to create the key pair:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For RSA key pairs, the key fingerprint is the SHA-1 digest of the DER encoded private key.
     * </p>
     * </li>
     * <li>
     * <p>
     * For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default for
     * OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For RSA key pairs, the key fingerprint is the MD5 public key fingerprint as specified in section 4 of RFC4716.
     * </p>
     * </li>
     * <li>
     * <p>
     * For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default for
     * OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return If you used <a>CreateKeyPair</a> to create the key pair:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For RSA key pairs, the key fingerprint is the SHA-1 digest of the DER encoded private key.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default for
     *         OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For RSA key pairs, the key fingerprint is the MD5 public key fingerprint as specified in section 4 of
     *         RFC4716.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default for
     *         OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
     *         </p>
     *         </li>
     */
    public final String keyFingerprint() {
        return keyFingerprint;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyPairId());
        hashCode = 31 * hashCode + Objects.hashCode(keyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(publicKey());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(keyName());
        hashCode = 31 * hashCode + Objects.hashCode(keyFingerprint());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeyPairInfo)) {
            return false;
        }
        KeyPairInfo other = (KeyPairInfo) obj;
        return Objects.equals(keyPairId(), other.keyPairId()) && Objects.equals(keyTypeAsString(), other.keyTypeAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(publicKey(), other.publicKey()) && Objects.equals(createTime(), other.createTime())
                && Objects.equals(keyName(), other.keyName()) && Objects.equals(keyFingerprint(), other.keyFingerprint());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KeyPairInfo").add("KeyPairId", keyPairId()).add("KeyType", keyTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("PublicKey", publicKey()).add("CreateTime", createTime())
                .add("KeyName", keyName()).add("KeyFingerprint", keyFingerprint()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyPairId":
            return Optional.ofNullable(clazz.cast(keyPairId()));
        case "KeyType":
            return Optional.ofNullable(clazz.cast(keyTypeAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "PublicKey":
            return Optional.ofNullable(clazz.cast(publicKey()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "KeyName":
            return Optional.ofNullable(clazz.cast(keyName()));
        case "KeyFingerprint":
            return Optional.ofNullable(clazz.cast(keyFingerprint()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<KeyPairInfo, T> g) {
        return obj -> g.apply((KeyPairInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeyPairInfo> {
        /**
         * <p>
         * The ID of the key pair.
         * </p>
         * 
         * @param keyPairId
         *        The ID of the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyPairId(String keyPairId);

        /**
         * <p>
         * The type of key pair.
         * </p>
         * 
         * @param keyType
         *        The type of key pair.
         * @see KeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyType
         */
        Builder keyType(String keyType);

        /**
         * <p>
         * The type of key pair.
         * </p>
         * 
         * @param keyType
         *        The type of key pair.
         * @see KeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyType
         */
        Builder keyType(KeyType keyType);

        /**
         * <p>
         * Any tags applied to the key pair.
         * </p>
         * 
         * @param tags
         *        Any tags applied to the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags applied to the key pair.
         * </p>
         * 
         * @param tags
         *        Any tags applied to the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags applied to the key pair.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The public key material.
         * </p>
         * 
         * @param publicKey
         *        The public key material.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicKey(String publicKey);

        /**
         * <p>
         * If you used Amazon EC2 to create the key pair, this is the date and time when the key was created, in <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the UTC time
         * zone.
         * </p>
         * <p>
         * If you imported an existing key pair to Amazon EC2, this is the date and time the key was imported, in <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the UTC time
         * zone.
         * </p>
         * 
         * @param createTime
         *        If you used Amazon EC2 to create the key pair, this is the date and time when the key was created, in
         *        <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the
         *        UTC time zone.</p>
         *        <p>
         *        If you imported an existing key pair to Amazon EC2, this is the date and time the key was imported, in
         *        <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601 date-time format</a>, in the
         *        UTC time zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The name of the key pair.
         * </p>
         * 
         * @param keyName
         *        The name of the key pair.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyName(String keyName);

        /**
         * <p>
         * If you used <a>CreateKeyPair</a> to create the key pair:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For RSA key pairs, the key fingerprint is the SHA-1 digest of the DER encoded private key.
         * </p>
         * </li>
         * <li>
         * <p>
         * For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default for
         * OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For RSA key pairs, the key fingerprint is the MD5 public key fingerprint as specified in section 4 of
         * RFC4716.
         * </p>
         * </li>
         * <li>
         * <p>
         * For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default for
         * OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyFingerprint
         *        If you used <a>CreateKeyPair</a> to create the key pair:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For RSA key pairs, the key fingerprint is the SHA-1 digest of the DER encoded private key.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default
         *        for OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you used <a>ImportKeyPair</a> to provide Amazon Web Services the public key:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For RSA key pairs, the key fingerprint is the MD5 public key fingerprint as specified in section 4 of
         *        RFC4716.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For ED25519 key pairs, the key fingerprint is the base64-encoded SHA-256 digest, which is the default
         *        for OpenSSH, starting with <a href="http://www.openssh.com/txt/release-6.8">OpenSSH 6.8</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyFingerprint(String keyFingerprint);
    }

    static final class BuilderImpl implements Builder {
        private String keyPairId;

        private String keyType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String publicKey;

        private Instant createTime;

        private String keyName;

        private String keyFingerprint;

        private BuilderImpl() {
        }

        private BuilderImpl(KeyPairInfo model) {
            keyPairId(model.keyPairId);
            keyType(model.keyType);
            tags(model.tags);
            publicKey(model.publicKey);
            createTime(model.createTime);
            keyName(model.keyName);
            keyFingerprint(model.keyFingerprint);
        }

        public final String getKeyPairId() {
            return keyPairId;
        }

        public final void setKeyPairId(String keyPairId) {
            this.keyPairId = keyPairId;
        }

        @Override
        public final Builder keyPairId(String keyPairId) {
            this.keyPairId = keyPairId;
            return this;
        }

        public final String getKeyType() {
            return keyType;
        }

        public final void setKeyType(String keyType) {
            this.keyType = keyType;
        }

        @Override
        public final Builder keyType(String keyType) {
            this.keyType = keyType;
            return this;
        }

        @Override
        public final Builder keyType(KeyType keyType) {
            this.keyType(keyType == null ? null : keyType.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPublicKey() {
            return publicKey;
        }

        public final void setPublicKey(String publicKey) {
            this.publicKey = publicKey;
        }

        @Override
        public final Builder publicKey(String publicKey) {
            this.publicKey = publicKey;
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final String getKeyName() {
            return keyName;
        }

        public final void setKeyName(String keyName) {
            this.keyName = keyName;
        }

        @Override
        public final Builder keyName(String keyName) {
            this.keyName = keyName;
            return this;
        }

        public final String getKeyFingerprint() {
            return keyFingerprint;
        }

        public final void setKeyFingerprint(String keyFingerprint) {
            this.keyFingerprint = keyFingerprint;
        }

        @Override
        public final Builder keyFingerprint(String keyFingerprint) {
            this.keyFingerprint = keyFingerprint;
            return this;
        }

        @Override
        public KeyPairInfo build() {
            return new KeyPairInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
