/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a VPC endpoint connection to a service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpcEndpointConnection implements SdkPojo, Serializable,
        ToCopyableBuilder<VpcEndpointConnection.Builder, VpcEndpointConnection> {
    private static final SdkField<String> SERVICE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ServiceId")
            .getter(getter(VpcEndpointConnection::serviceId))
            .setter(setter(Builder::serviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceId")
                    .unmarshallLocationName("serviceId").build()).build();

    private static final SdkField<String> VPC_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointId")
            .getter(getter(VpcEndpointConnection::vpcEndpointId))
            .setter(setter(Builder::vpcEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointId")
                    .unmarshallLocationName("vpcEndpointId").build()).build();

    private static final SdkField<String> VPC_ENDPOINT_OWNER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointOwner")
            .getter(getter(VpcEndpointConnection::vpcEndpointOwner))
            .setter(setter(Builder::vpcEndpointOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointOwner")
                    .unmarshallLocationName("vpcEndpointOwner").build()).build();

    private static final SdkField<String> VPC_ENDPOINT_STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointState")
            .getter(getter(VpcEndpointConnection::vpcEndpointStateAsString))
            .setter(setter(Builder::vpcEndpointState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointState")
                    .unmarshallLocationName("vpcEndpointState").build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTimestamp")
            .getter(getter(VpcEndpointConnection::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp")
                    .unmarshallLocationName("creationTimestamp").build()).build();

    private static final SdkField<List<DnsEntry>> DNS_ENTRIES_FIELD = SdkField
            .<List<DnsEntry>> builder(MarshallingType.LIST)
            .memberName("DnsEntries")
            .getter(getter(VpcEndpointConnection::dnsEntries))
            .setter(setter(Builder::dnsEntries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsEntrySet")
                    .unmarshallLocationName("dnsEntrySet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<DnsEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(DnsEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> NETWORK_LOAD_BALANCER_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkLoadBalancerArns")
            .getter(getter(VpcEndpointConnection::networkLoadBalancerArns))
            .setter(setter(Builder::networkLoadBalancerArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkLoadBalancerArnSet")
                    .unmarshallLocationName("networkLoadBalancerArnSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> GATEWAY_LOAD_BALANCER_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GatewayLoadBalancerArns")
            .getter(getter(VpcEndpointConnection::gatewayLoadBalancerArns))
            .setter(setter(Builder::gatewayLoadBalancerArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GatewayLoadBalancerArnSet")
                    .unmarshallLocationName("gatewayLoadBalancerArnSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpAddressType")
            .getter(getter(VpcEndpointConnection::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressType")
                    .unmarshallLocationName("ipAddressType").build()).build();

    private static final SdkField<String> VPC_ENDPOINT_CONNECTION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointConnectionId")
            .getter(getter(VpcEndpointConnection::vpcEndpointConnectionId))
            .setter(setter(Builder::vpcEndpointConnectionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointConnectionId")
                    .unmarshallLocationName("vpcEndpointConnectionId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(VpcEndpointConnection::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_ID_FIELD,
            VPC_ENDPOINT_ID_FIELD, VPC_ENDPOINT_OWNER_FIELD, VPC_ENDPOINT_STATE_FIELD, CREATION_TIMESTAMP_FIELD,
            DNS_ENTRIES_FIELD, NETWORK_LOAD_BALANCER_ARNS_FIELD, GATEWAY_LOAD_BALANCER_ARNS_FIELD, IP_ADDRESS_TYPE_FIELD,
            VPC_ENDPOINT_CONNECTION_ID_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String serviceId;

    private final String vpcEndpointId;

    private final String vpcEndpointOwner;

    private final String vpcEndpointState;

    private final Instant creationTimestamp;

    private final List<DnsEntry> dnsEntries;

    private final List<String> networkLoadBalancerArns;

    private final List<String> gatewayLoadBalancerArns;

    private final String ipAddressType;

    private final String vpcEndpointConnectionId;

    private final List<Tag> tags;

    private VpcEndpointConnection(BuilderImpl builder) {
        this.serviceId = builder.serviceId;
        this.vpcEndpointId = builder.vpcEndpointId;
        this.vpcEndpointOwner = builder.vpcEndpointOwner;
        this.vpcEndpointState = builder.vpcEndpointState;
        this.creationTimestamp = builder.creationTimestamp;
        this.dnsEntries = builder.dnsEntries;
        this.networkLoadBalancerArns = builder.networkLoadBalancerArns;
        this.gatewayLoadBalancerArns = builder.gatewayLoadBalancerArns;
        this.ipAddressType = builder.ipAddressType;
        this.vpcEndpointConnectionId = builder.vpcEndpointConnectionId;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the service to which the endpoint is connected.
     * </p>
     * 
     * @return The ID of the service to which the endpoint is connected.
     */
    public final String serviceId() {
        return serviceId;
    }

    /**
     * <p>
     * The ID of the VPC endpoint.
     * </p>
     * 
     * @return The ID of the VPC endpoint.
     */
    public final String vpcEndpointId() {
        return vpcEndpointId;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that owns the VPC endpoint.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that owns the VPC endpoint.
     */
    public final String vpcEndpointOwner() {
        return vpcEndpointOwner;
    }

    /**
     * <p>
     * The state of the VPC endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpcEndpointState}
     * will return {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vpcEndpointStateAsString}.
     * </p>
     * 
     * @return The state of the VPC endpoint.
     * @see State
     */
    public final State vpcEndpointState() {
        return State.fromValue(vpcEndpointState);
    }

    /**
     * <p>
     * The state of the VPC endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpcEndpointState}
     * will return {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vpcEndpointStateAsString}.
     * </p>
     * 
     * @return The state of the VPC endpoint.
     * @see State
     */
    public final String vpcEndpointStateAsString() {
        return vpcEndpointState;
    }

    /**
     * <p>
     * The date and time that the VPC endpoint was created.
     * </p>
     * 
     * @return The date and time that the VPC endpoint was created.
     */
    public final Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsEntries property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsEntries() {
        return dnsEntries != null && !(dnsEntries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The DNS entries for the VPC endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsEntries} method.
     * </p>
     * 
     * @return The DNS entries for the VPC endpoint.
     */
    public final List<DnsEntry> dnsEntries() {
        return dnsEntries;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkLoadBalancerArns property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasNetworkLoadBalancerArns() {
        return networkLoadBalancerArns != null && !(networkLoadBalancerArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the network load balancers for the service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkLoadBalancerArns} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the network load balancers for the service.
     */
    public final List<String> networkLoadBalancerArns() {
        return networkLoadBalancerArns;
    }

    /**
     * For responses, this returns true if the service returned a value for the GatewayLoadBalancerArns property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasGatewayLoadBalancerArns() {
        return gatewayLoadBalancerArns != null && !(gatewayLoadBalancerArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the Gateway Load Balancers for the service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGatewayLoadBalancerArns} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the Gateway Load Balancers for the service.
     */
    public final List<String> gatewayLoadBalancerArns() {
        return gatewayLoadBalancerArns;
    }

    /**
     * <p>
     * The IP address type for the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The IP address type for the endpoint.
     * @see IpAddressType
     */
    public final IpAddressType ipAddressType() {
        return IpAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The IP address type for the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The IP address type for the endpoint.
     * @see IpAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * <p>
     * The ID of the VPC endpoint connection.
     * </p>
     * 
     * @return The ID of the VPC endpoint connection.
     */
    public final String vpcEndpointConnectionId() {
        return vpcEndpointConnectionId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointOwner());
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsEntries() ? dnsEntries() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkLoadBalancerArns() ? networkLoadBalancerArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasGatewayLoadBalancerArns() ? gatewayLoadBalancerArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointConnectionId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpcEndpointConnection)) {
            return false;
        }
        VpcEndpointConnection other = (VpcEndpointConnection) obj;
        return Objects.equals(serviceId(), other.serviceId()) && Objects.equals(vpcEndpointId(), other.vpcEndpointId())
                && Objects.equals(vpcEndpointOwner(), other.vpcEndpointOwner())
                && Objects.equals(vpcEndpointStateAsString(), other.vpcEndpointStateAsString())
                && Objects.equals(creationTimestamp(), other.creationTimestamp()) && hasDnsEntries() == other.hasDnsEntries()
                && Objects.equals(dnsEntries(), other.dnsEntries())
                && hasNetworkLoadBalancerArns() == other.hasNetworkLoadBalancerArns()
                && Objects.equals(networkLoadBalancerArns(), other.networkLoadBalancerArns())
                && hasGatewayLoadBalancerArns() == other.hasGatewayLoadBalancerArns()
                && Objects.equals(gatewayLoadBalancerArns(), other.gatewayLoadBalancerArns())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString())
                && Objects.equals(vpcEndpointConnectionId(), other.vpcEndpointConnectionId()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VpcEndpointConnection").add("ServiceId", serviceId()).add("VpcEndpointId", vpcEndpointId())
                .add("VpcEndpointOwner", vpcEndpointOwner()).add("VpcEndpointState", vpcEndpointStateAsString())
                .add("CreationTimestamp", creationTimestamp()).add("DnsEntries", hasDnsEntries() ? dnsEntries() : null)
                .add("NetworkLoadBalancerArns", hasNetworkLoadBalancerArns() ? networkLoadBalancerArns() : null)
                .add("GatewayLoadBalancerArns", hasGatewayLoadBalancerArns() ? gatewayLoadBalancerArns() : null)
                .add("IpAddressType", ipAddressTypeAsString()).add("VpcEndpointConnectionId", vpcEndpointConnectionId())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServiceId":
            return Optional.ofNullable(clazz.cast(serviceId()));
        case "VpcEndpointId":
            return Optional.ofNullable(clazz.cast(vpcEndpointId()));
        case "VpcEndpointOwner":
            return Optional.ofNullable(clazz.cast(vpcEndpointOwner()));
        case "VpcEndpointState":
            return Optional.ofNullable(clazz.cast(vpcEndpointStateAsString()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "DnsEntries":
            return Optional.ofNullable(clazz.cast(dnsEntries()));
        case "NetworkLoadBalancerArns":
            return Optional.ofNullable(clazz.cast(networkLoadBalancerArns()));
        case "GatewayLoadBalancerArns":
            return Optional.ofNullable(clazz.cast(gatewayLoadBalancerArns()));
        case "IpAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "VpcEndpointConnectionId":
            return Optional.ofNullable(clazz.cast(vpcEndpointConnectionId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VpcEndpointConnection, T> g) {
        return obj -> g.apply((VpcEndpointConnection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpcEndpointConnection> {
        /**
         * <p>
         * The ID of the service to which the endpoint is connected.
         * </p>
         * 
         * @param serviceId
         *        The ID of the service to which the endpoint is connected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceId(String serviceId);

        /**
         * <p>
         * The ID of the VPC endpoint.
         * </p>
         * 
         * @param vpcEndpointId
         *        The ID of the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpointId(String vpcEndpointId);

        /**
         * <p>
         * The ID of the Amazon Web Services account that owns the VPC endpoint.
         * </p>
         * 
         * @param vpcEndpointOwner
         *        The ID of the Amazon Web Services account that owns the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpointOwner(String vpcEndpointOwner);

        /**
         * <p>
         * The state of the VPC endpoint.
         * </p>
         * 
         * @param vpcEndpointState
         *        The state of the VPC endpoint.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder vpcEndpointState(String vpcEndpointState);

        /**
         * <p>
         * The state of the VPC endpoint.
         * </p>
         * 
         * @param vpcEndpointState
         *        The state of the VPC endpoint.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder vpcEndpointState(State vpcEndpointState);

        /**
         * <p>
         * The date and time that the VPC endpoint was created.
         * </p>
         * 
         * @param creationTimestamp
         *        The date and time that the VPC endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * The DNS entries for the VPC endpoint.
         * </p>
         * 
         * @param dnsEntries
         *        The DNS entries for the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsEntries(Collection<DnsEntry> dnsEntries);

        /**
         * <p>
         * The DNS entries for the VPC endpoint.
         * </p>
         * 
         * @param dnsEntries
         *        The DNS entries for the VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsEntries(DnsEntry... dnsEntries);

        /**
         * <p>
         * The DNS entries for the VPC endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.DnsEntry.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ec2.model.DnsEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.DnsEntry.Builder#build()} is called immediately and its
         * result is passed to {@link #dnsEntries(List<DnsEntry>)}.
         * 
         * @param dnsEntries
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.DnsEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dnsEntries(java.util.Collection<DnsEntry>)
         */
        Builder dnsEntries(Consumer<DnsEntry.Builder>... dnsEntries);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the network load balancers for the service.
         * </p>
         * 
         * @param networkLoadBalancerArns
         *        The Amazon Resource Names (ARNs) of the network load balancers for the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkLoadBalancerArns(Collection<String> networkLoadBalancerArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the network load balancers for the service.
         * </p>
         * 
         * @param networkLoadBalancerArns
         *        The Amazon Resource Names (ARNs) of the network load balancers for the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkLoadBalancerArns(String... networkLoadBalancerArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the Gateway Load Balancers for the service.
         * </p>
         * 
         * @param gatewayLoadBalancerArns
         *        The Amazon Resource Names (ARNs) of the Gateway Load Balancers for the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gatewayLoadBalancerArns(Collection<String> gatewayLoadBalancerArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the Gateway Load Balancers for the service.
         * </p>
         * 
         * @param gatewayLoadBalancerArns
         *        The Amazon Resource Names (ARNs) of the Gateway Load Balancers for the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gatewayLoadBalancerArns(String... gatewayLoadBalancerArns);

        /**
         * <p>
         * The IP address type for the endpoint.
         * </p>
         * 
         * @param ipAddressType
         *        The IP address type for the endpoint.
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The IP address type for the endpoint.
         * </p>
         * 
         * @param ipAddressType
         *        The IP address type for the endpoint.
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(IpAddressType ipAddressType);

        /**
         * <p>
         * The ID of the VPC endpoint connection.
         * </p>
         * 
         * @param vpcEndpointConnectionId
         *        The ID of the VPC endpoint connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpointConnectionId(String vpcEndpointConnectionId);

        /**
         * <p>
         * The tags.
         * </p>
         * 
         * @param tags
         *        The tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags.
         * </p>
         * 
         * @param tags
         *        The tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String serviceId;

        private String vpcEndpointId;

        private String vpcEndpointOwner;

        private String vpcEndpointState;

        private Instant creationTimestamp;

        private List<DnsEntry> dnsEntries = DefaultSdkAutoConstructList.getInstance();

        private List<String> networkLoadBalancerArns = DefaultSdkAutoConstructList.getInstance();

        private List<String> gatewayLoadBalancerArns = DefaultSdkAutoConstructList.getInstance();

        private String ipAddressType;

        private String vpcEndpointConnectionId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(VpcEndpointConnection model) {
            serviceId(model.serviceId);
            vpcEndpointId(model.vpcEndpointId);
            vpcEndpointOwner(model.vpcEndpointOwner);
            vpcEndpointState(model.vpcEndpointState);
            creationTimestamp(model.creationTimestamp);
            dnsEntries(model.dnsEntries);
            networkLoadBalancerArns(model.networkLoadBalancerArns);
            gatewayLoadBalancerArns(model.gatewayLoadBalancerArns);
            ipAddressType(model.ipAddressType);
            vpcEndpointConnectionId(model.vpcEndpointConnectionId);
            tags(model.tags);
        }

        public final String getServiceId() {
            return serviceId;
        }

        public final void setServiceId(String serviceId) {
            this.serviceId = serviceId;
        }

        @Override
        public final Builder serviceId(String serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public final String getVpcEndpointId() {
            return vpcEndpointId;
        }

        public final void setVpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
        }

        @Override
        public final Builder vpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
            return this;
        }

        public final String getVpcEndpointOwner() {
            return vpcEndpointOwner;
        }

        public final void setVpcEndpointOwner(String vpcEndpointOwner) {
            this.vpcEndpointOwner = vpcEndpointOwner;
        }

        @Override
        public final Builder vpcEndpointOwner(String vpcEndpointOwner) {
            this.vpcEndpointOwner = vpcEndpointOwner;
            return this;
        }

        public final String getVpcEndpointState() {
            return vpcEndpointState;
        }

        public final void setVpcEndpointState(String vpcEndpointState) {
            this.vpcEndpointState = vpcEndpointState;
        }

        @Override
        public final Builder vpcEndpointState(String vpcEndpointState) {
            this.vpcEndpointState = vpcEndpointState;
            return this;
        }

        @Override
        public final Builder vpcEndpointState(State vpcEndpointState) {
            this.vpcEndpointState(vpcEndpointState == null ? null : vpcEndpointState.toString());
            return this;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final List<DnsEntry.Builder> getDnsEntries() {
            List<DnsEntry.Builder> result = DnsEntrySetCopier.copyToBuilder(this.dnsEntries);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDnsEntries(Collection<DnsEntry.BuilderImpl> dnsEntries) {
            this.dnsEntries = DnsEntrySetCopier.copyFromBuilder(dnsEntries);
        }

        @Override
        public final Builder dnsEntries(Collection<DnsEntry> dnsEntries) {
            this.dnsEntries = DnsEntrySetCopier.copy(dnsEntries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsEntries(DnsEntry... dnsEntries) {
            dnsEntries(Arrays.asList(dnsEntries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsEntries(Consumer<DnsEntry.Builder>... dnsEntries) {
            dnsEntries(Stream.of(dnsEntries).map(c -> DnsEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getNetworkLoadBalancerArns() {
            if (networkLoadBalancerArns instanceof SdkAutoConstructList) {
                return null;
            }
            return networkLoadBalancerArns;
        }

        public final void setNetworkLoadBalancerArns(Collection<String> networkLoadBalancerArns) {
            this.networkLoadBalancerArns = ValueStringListCopier.copy(networkLoadBalancerArns);
        }

        @Override
        public final Builder networkLoadBalancerArns(Collection<String> networkLoadBalancerArns) {
            this.networkLoadBalancerArns = ValueStringListCopier.copy(networkLoadBalancerArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkLoadBalancerArns(String... networkLoadBalancerArns) {
            networkLoadBalancerArns(Arrays.asList(networkLoadBalancerArns));
            return this;
        }

        public final Collection<String> getGatewayLoadBalancerArns() {
            if (gatewayLoadBalancerArns instanceof SdkAutoConstructList) {
                return null;
            }
            return gatewayLoadBalancerArns;
        }

        public final void setGatewayLoadBalancerArns(Collection<String> gatewayLoadBalancerArns) {
            this.gatewayLoadBalancerArns = ValueStringListCopier.copy(gatewayLoadBalancerArns);
        }

        @Override
        public final Builder gatewayLoadBalancerArns(Collection<String> gatewayLoadBalancerArns) {
            this.gatewayLoadBalancerArns = ValueStringListCopier.copy(gatewayLoadBalancerArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gatewayLoadBalancerArns(String... gatewayLoadBalancerArns) {
            gatewayLoadBalancerArns(Arrays.asList(gatewayLoadBalancerArns));
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IpAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final String getVpcEndpointConnectionId() {
            return vpcEndpointConnectionId;
        }

        public final void setVpcEndpointConnectionId(String vpcEndpointConnectionId) {
            this.vpcEndpointConnectionId = vpcEndpointConnectionId;
        }

        @Override
        public final Builder vpcEndpointConnectionId(String vpcEndpointConnectionId) {
            this.vpcEndpointConnectionId = vpcEndpointConnectionId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public VpcEndpointConnection build() {
            return new VpcEndpointConnection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
