/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The information that DataSync Discovery collects about a storage virtual machine (SVM) in your on-premises storage
 * system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetAppONTAPSVM implements SdkPojo, Serializable, ToCopyableBuilder<NetAppONTAPSVM.Builder, NetAppONTAPSVM> {
    private static final SdkField<String> CLUSTER_UUID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClusterUuid").getter(getter(NetAppONTAPSVM::clusterUuid)).setter(setter(Builder::clusterUuid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterUuid").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(NetAppONTAPSVM::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> SVM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SvmName").getter(getter(NetAppONTAPSVM::svmName)).setter(setter(Builder::svmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SvmName").build()).build();

    private static final SdkField<Long> CIFS_SHARE_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CifsShareCount").getter(getter(NetAppONTAPSVM::cifsShareCount)).setter(setter(Builder::cifsShareCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CifsShareCount").build()).build();

    private static final SdkField<List<String>> ENABLED_PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnabledProtocols")
            .getter(getter(NetAppONTAPSVM::enabledProtocols))
            .setter(setter(Builder::enabledProtocols))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnabledProtocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> TOTAL_CAPACITY_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("TotalCapacityUsed").getter(getter(NetAppONTAPSVM::totalCapacityUsed))
            .setter(setter(Builder::totalCapacityUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalCapacityUsed").build()).build();

    private static final SdkField<Long> TOTAL_CAPACITY_PROVISIONED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("TotalCapacityProvisioned").getter(getter(NetAppONTAPSVM::totalCapacityProvisioned))
            .setter(setter(Builder::totalCapacityProvisioned))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalCapacityProvisioned").build())
            .build();

    private static final SdkField<Long> TOTAL_LOGICAL_CAPACITY_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("TotalLogicalCapacityUsed").getter(getter(NetAppONTAPSVM::totalLogicalCapacityUsed))
            .setter(setter(Builder::totalLogicalCapacityUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalLogicalCapacityUsed").build())
            .build();

    private static final SdkField<MaxP95Performance> MAX_P95_PERFORMANCE_FIELD = SdkField
            .<MaxP95Performance> builder(MarshallingType.SDK_POJO).memberName("MaxP95Performance")
            .getter(getter(NetAppONTAPSVM::maxP95Performance)).setter(setter(Builder::maxP95Performance))
            .constructor(MaxP95Performance::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxP95Performance").build()).build();

    private static final SdkField<List<Recommendation>> RECOMMENDATIONS_FIELD = SdkField
            .<List<Recommendation>> builder(MarshallingType.LIST)
            .memberName("Recommendations")
            .getter(getter(NetAppONTAPSVM::recommendations))
            .setter(setter(Builder::recommendations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recommendations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Recommendation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Recommendation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> NFS_EXPORTED_VOLUMES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("NfsExportedVolumes").getter(getter(NetAppONTAPSVM::nfsExportedVolumes))
            .setter(setter(Builder::nfsExportedVolumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NfsExportedVolumes").build())
            .build();

    private static final SdkField<String> RECOMMENDATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecommendationStatus").getter(getter(NetAppONTAPSVM::recommendationStatusAsString))
            .setter(setter(Builder::recommendationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecommendationStatus").build())
            .build();

    private static final SdkField<Long> TOTAL_SNAPSHOT_CAPACITY_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("TotalSnapshotCapacityUsed").getter(getter(NetAppONTAPSVM::totalSnapshotCapacityUsed))
            .setter(setter(Builder::totalSnapshotCapacityUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalSnapshotCapacityUsed").build())
            .build();

    private static final SdkField<Long> LUN_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("LunCount")
            .getter(getter(NetAppONTAPSVM::lunCount)).setter(setter(Builder::lunCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LunCount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_UUID_FIELD,
            RESOURCE_ID_FIELD, SVM_NAME_FIELD, CIFS_SHARE_COUNT_FIELD, ENABLED_PROTOCOLS_FIELD, TOTAL_CAPACITY_USED_FIELD,
            TOTAL_CAPACITY_PROVISIONED_FIELD, TOTAL_LOGICAL_CAPACITY_USED_FIELD, MAX_P95_PERFORMANCE_FIELD,
            RECOMMENDATIONS_FIELD, NFS_EXPORTED_VOLUMES_FIELD, RECOMMENDATION_STATUS_FIELD, TOTAL_SNAPSHOT_CAPACITY_USED_FIELD,
            LUN_COUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clusterUuid;

    private final String resourceId;

    private final String svmName;

    private final Long cifsShareCount;

    private final List<String> enabledProtocols;

    private final Long totalCapacityUsed;

    private final Long totalCapacityProvisioned;

    private final Long totalLogicalCapacityUsed;

    private final MaxP95Performance maxP95Performance;

    private final List<Recommendation> recommendations;

    private final Long nfsExportedVolumes;

    private final String recommendationStatus;

    private final Long totalSnapshotCapacityUsed;

    private final Long lunCount;

    private NetAppONTAPSVM(BuilderImpl builder) {
        this.clusterUuid = builder.clusterUuid;
        this.resourceId = builder.resourceId;
        this.svmName = builder.svmName;
        this.cifsShareCount = builder.cifsShareCount;
        this.enabledProtocols = builder.enabledProtocols;
        this.totalCapacityUsed = builder.totalCapacityUsed;
        this.totalCapacityProvisioned = builder.totalCapacityProvisioned;
        this.totalLogicalCapacityUsed = builder.totalLogicalCapacityUsed;
        this.maxP95Performance = builder.maxP95Performance;
        this.recommendations = builder.recommendations;
        this.nfsExportedVolumes = builder.nfsExportedVolumes;
        this.recommendationStatus = builder.recommendationStatus;
        this.totalSnapshotCapacityUsed = builder.totalSnapshotCapacityUsed;
        this.lunCount = builder.lunCount;
    }

    /**
     * <p>
     * The universally unique identifier (UUID) of the cluster associated with the SVM.
     * </p>
     * 
     * @return The universally unique identifier (UUID) of the cluster associated with the SVM.
     */
    public final String clusterUuid() {
        return clusterUuid;
    }

    /**
     * <p>
     * The UUID of the SVM.
     * </p>
     * 
     * @return The UUID of the SVM.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The name of the SVM
     * </p>
     * 
     * @return The name of the SVM
     */
    public final String svmName() {
        return svmName;
    }

    /**
     * <p>
     * The number of CIFS shares in the SVM.
     * </p>
     * 
     * @return The number of CIFS shares in the SVM.
     */
    public final Long cifsShareCount() {
        return cifsShareCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnabledProtocols property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnabledProtocols() {
        return enabledProtocols != null && !(enabledProtocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The data transfer protocols (such as NFS) configured for the SVM.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnabledProtocols} method.
     * </p>
     * 
     * @return The data transfer protocols (such as NFS) configured for the SVM.
     */
    public final List<String> enabledProtocols() {
        return enabledProtocols;
    }

    /**
     * <p>
     * The storage space that's being used in the SVM.
     * </p>
     * 
     * @return The storage space that's being used in the SVM.
     */
    public final Long totalCapacityUsed() {
        return totalCapacityUsed;
    }

    /**
     * <p>
     * The total storage space that's available in the SVM.
     * </p>
     * 
     * @return The total storage space that's available in the SVM.
     */
    public final Long totalCapacityProvisioned() {
        return totalCapacityProvisioned;
    }

    /**
     * <p>
     * The storage space that's being used in the SVM without accounting for compression or deduplication.
     * </p>
     * 
     * @return The storage space that's being used in the SVM without accounting for compression or deduplication.
     */
    public final Long totalLogicalCapacityUsed() {
        return totalLogicalCapacityUsed;
    }

    /**
     * <p>
     * The performance data that DataSync Discovery collects about the SVM.
     * </p>
     * 
     * @return The performance data that DataSync Discovery collects about the SVM.
     */
    public final MaxP95Performance maxP95Performance() {
        return maxP95Performance;
    }

    /**
     * For responses, this returns true if the service returned a value for the Recommendations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendations() {
        return recommendations != null && !(recommendations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more information,
     * see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html">
     * Recommendations provided by DataSync Discovery</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendations} method.
     * </p>
     * 
     * @return The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
     *         >Recommendations provided by DataSync Discovery</a>.
     */
    public final List<Recommendation> recommendations() {
        return recommendations;
    }

    /**
     * <p>
     * The number of NFS volumes in the SVM.
     * </p>
     * 
     * @return The number of NFS volumes in the SVM.
     */
    public final Long nfsExportedVolumes() {
        return nfsExportedVolumes;
    }

    /**
     * <p>
     * Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or can't be
     * determined.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     * >Recommendation statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recommendationStatus} will return {@link RecommendationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #recommendationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or can't
     *         be determined.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     *         >Recommendation statuses</a>.
     * @see RecommendationStatus
     */
    public final RecommendationStatus recommendationStatus() {
        return RecommendationStatus.fromValue(recommendationStatus);
    }

    /**
     * <p>
     * Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or can't be
     * determined.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     * >Recommendation statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recommendationStatus} will return {@link RecommendationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #recommendationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or can't
     *         be determined.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     *         >Recommendation statuses</a>.
     * @see RecommendationStatus
     */
    public final String recommendationStatusAsString() {
        return recommendationStatus;
    }

    /**
     * <p>
     * The amount of storage in the SVM that's being used for snapshots.
     * </p>
     * 
     * @return The amount of storage in the SVM that's being used for snapshots.
     */
    public final Long totalSnapshotCapacityUsed() {
        return totalSnapshotCapacityUsed;
    }

    /**
     * <p>
     * The number of LUNs (logical unit numbers) in the SVM.
     * </p>
     * 
     * @return The number of LUNs (logical unit numbers) in the SVM.
     */
    public final Long lunCount() {
        return lunCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterUuid());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(svmName());
        hashCode = 31 * hashCode + Objects.hashCode(cifsShareCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnabledProtocols() ? enabledProtocols() : null);
        hashCode = 31 * hashCode + Objects.hashCode(totalCapacityUsed());
        hashCode = 31 * hashCode + Objects.hashCode(totalCapacityProvisioned());
        hashCode = 31 * hashCode + Objects.hashCode(totalLogicalCapacityUsed());
        hashCode = 31 * hashCode + Objects.hashCode(maxP95Performance());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendations() ? recommendations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nfsExportedVolumes());
        hashCode = 31 * hashCode + Objects.hashCode(recommendationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(totalSnapshotCapacityUsed());
        hashCode = 31 * hashCode + Objects.hashCode(lunCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetAppONTAPSVM)) {
            return false;
        }
        NetAppONTAPSVM other = (NetAppONTAPSVM) obj;
        return Objects.equals(clusterUuid(), other.clusterUuid()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(svmName(), other.svmName()) && Objects.equals(cifsShareCount(), other.cifsShareCount())
                && hasEnabledProtocols() == other.hasEnabledProtocols()
                && Objects.equals(enabledProtocols(), other.enabledProtocols())
                && Objects.equals(totalCapacityUsed(), other.totalCapacityUsed())
                && Objects.equals(totalCapacityProvisioned(), other.totalCapacityProvisioned())
                && Objects.equals(totalLogicalCapacityUsed(), other.totalLogicalCapacityUsed())
                && Objects.equals(maxP95Performance(), other.maxP95Performance())
                && hasRecommendations() == other.hasRecommendations()
                && Objects.equals(recommendations(), other.recommendations())
                && Objects.equals(nfsExportedVolumes(), other.nfsExportedVolumes())
                && Objects.equals(recommendationStatusAsString(), other.recommendationStatusAsString())
                && Objects.equals(totalSnapshotCapacityUsed(), other.totalSnapshotCapacityUsed())
                && Objects.equals(lunCount(), other.lunCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetAppONTAPSVM").add("ClusterUuid", clusterUuid()).add("ResourceId", resourceId())
                .add("SvmName", svmName()).add("CifsShareCount", cifsShareCount())
                .add("EnabledProtocols", hasEnabledProtocols() ? enabledProtocols() : null)
                .add("TotalCapacityUsed", totalCapacityUsed()).add("TotalCapacityProvisioned", totalCapacityProvisioned())
                .add("TotalLogicalCapacityUsed", totalLogicalCapacityUsed()).add("MaxP95Performance", maxP95Performance())
                .add("Recommendations", hasRecommendations() ? recommendations() : null)
                .add("NfsExportedVolumes", nfsExportedVolumes()).add("RecommendationStatus", recommendationStatusAsString())
                .add("TotalSnapshotCapacityUsed", totalSnapshotCapacityUsed()).add("LunCount", lunCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClusterUuid":
            return Optional.ofNullable(clazz.cast(clusterUuid()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "SvmName":
            return Optional.ofNullable(clazz.cast(svmName()));
        case "CifsShareCount":
            return Optional.ofNullable(clazz.cast(cifsShareCount()));
        case "EnabledProtocols":
            return Optional.ofNullable(clazz.cast(enabledProtocols()));
        case "TotalCapacityUsed":
            return Optional.ofNullable(clazz.cast(totalCapacityUsed()));
        case "TotalCapacityProvisioned":
            return Optional.ofNullable(clazz.cast(totalCapacityProvisioned()));
        case "TotalLogicalCapacityUsed":
            return Optional.ofNullable(clazz.cast(totalLogicalCapacityUsed()));
        case "MaxP95Performance":
            return Optional.ofNullable(clazz.cast(maxP95Performance()));
        case "Recommendations":
            return Optional.ofNullable(clazz.cast(recommendations()));
        case "NfsExportedVolumes":
            return Optional.ofNullable(clazz.cast(nfsExportedVolumes()));
        case "RecommendationStatus":
            return Optional.ofNullable(clazz.cast(recommendationStatusAsString()));
        case "TotalSnapshotCapacityUsed":
            return Optional.ofNullable(clazz.cast(totalSnapshotCapacityUsed()));
        case "LunCount":
            return Optional.ofNullable(clazz.cast(lunCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NetAppONTAPSVM, T> g) {
        return obj -> g.apply((NetAppONTAPSVM) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetAppONTAPSVM> {
        /**
         * <p>
         * The universally unique identifier (UUID) of the cluster associated with the SVM.
         * </p>
         * 
         * @param clusterUuid
         *        The universally unique identifier (UUID) of the cluster associated with the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterUuid(String clusterUuid);

        /**
         * <p>
         * The UUID of the SVM.
         * </p>
         * 
         * @param resourceId
         *        The UUID of the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The name of the SVM
         * </p>
         * 
         * @param svmName
         *        The name of the SVM
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder svmName(String svmName);

        /**
         * <p>
         * The number of CIFS shares in the SVM.
         * </p>
         * 
         * @param cifsShareCount
         *        The number of CIFS shares in the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cifsShareCount(Long cifsShareCount);

        /**
         * <p>
         * The data transfer protocols (such as NFS) configured for the SVM.
         * </p>
         * 
         * @param enabledProtocols
         *        The data transfer protocols (such as NFS) configured for the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledProtocols(Collection<String> enabledProtocols);

        /**
         * <p>
         * The data transfer protocols (such as NFS) configured for the SVM.
         * </p>
         * 
         * @param enabledProtocols
         *        The data transfer protocols (such as NFS) configured for the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledProtocols(String... enabledProtocols);

        /**
         * <p>
         * The storage space that's being used in the SVM.
         * </p>
         * 
         * @param totalCapacityUsed
         *        The storage space that's being used in the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalCapacityUsed(Long totalCapacityUsed);

        /**
         * <p>
         * The total storage space that's available in the SVM.
         * </p>
         * 
         * @param totalCapacityProvisioned
         *        The total storage space that's available in the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalCapacityProvisioned(Long totalCapacityProvisioned);

        /**
         * <p>
         * The storage space that's being used in the SVM without accounting for compression or deduplication.
         * </p>
         * 
         * @param totalLogicalCapacityUsed
         *        The storage space that's being used in the SVM without accounting for compression or deduplication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalLogicalCapacityUsed(Long totalLogicalCapacityUsed);

        /**
         * <p>
         * The performance data that DataSync Discovery collects about the SVM.
         * </p>
         * 
         * @param maxP95Performance
         *        The performance data that DataSync Discovery collects about the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxP95Performance(MaxP95Performance maxP95Performance);

        /**
         * <p>
         * The performance data that DataSync Discovery collects about the SVM.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaxP95Performance.Builder} avoiding the
         * need to create one manually via {@link MaxP95Performance#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaxP95Performance.Builder#build()} is called immediately and its
         * result is passed to {@link #maxP95Performance(MaxP95Performance)}.
         * 
         * @param maxP95Performance
         *        a consumer that will call methods on {@link MaxP95Performance.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maxP95Performance(MaxP95Performance)
         */
        default Builder maxP95Performance(Consumer<MaxP95Performance.Builder> maxP95Performance) {
            return maxP95Performance(MaxP95Performance.builder().applyMutation(maxP95Performance).build());
        }

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * 
         * @param recommendations
         *        The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         *        >Recommendations provided by DataSync Discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Collection<Recommendation> recommendations);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * 
         * @param recommendations
         *        The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         *        >Recommendations provided by DataSync Discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Recommendation... recommendations);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the SVM. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.Recommendation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder#build()} is called immediately
         * and its result is passed to {@link #recommendations(List<Recommendation>)}.
         * 
         * @param recommendations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendations(java.util.Collection<Recommendation>)
         */
        Builder recommendations(Consumer<Recommendation.Builder>... recommendations);

        /**
         * <p>
         * The number of NFS volumes in the SVM.
         * </p>
         * 
         * @param nfsExportedVolumes
         *        The number of NFS volumes in the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExportedVolumes(Long nfsExportedVolumes);

        /**
         * <p>
         * Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or can't be
         * determined.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         * >Recommendation statuses</a>.
         * </p>
         * 
         * @param recommendationStatus
         *        Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or
         *        can't be determined.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         *        >Recommendation statuses</a>.
         * @see RecommendationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationStatus
         */
        Builder recommendationStatus(String recommendationStatus);

        /**
         * <p>
         * Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or can't be
         * determined.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         * >Recommendation statuses</a>.
         * </p>
         * 
         * @param recommendationStatus
         *        Indicates whether DataSync Discovery recommendations for the SVM are ready to view, incomplete, or
         *        can't be determined.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         *        >Recommendation statuses</a>.
         * @see RecommendationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationStatus
         */
        Builder recommendationStatus(RecommendationStatus recommendationStatus);

        /**
         * <p>
         * The amount of storage in the SVM that's being used for snapshots.
         * </p>
         * 
         * @param totalSnapshotCapacityUsed
         *        The amount of storage in the SVM that's being used for snapshots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalSnapshotCapacityUsed(Long totalSnapshotCapacityUsed);

        /**
         * <p>
         * The number of LUNs (logical unit numbers) in the SVM.
         * </p>
         * 
         * @param lunCount
         *        The number of LUNs (logical unit numbers) in the SVM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lunCount(Long lunCount);
    }

    static final class BuilderImpl implements Builder {
        private String clusterUuid;

        private String resourceId;

        private String svmName;

        private Long cifsShareCount;

        private List<String> enabledProtocols = DefaultSdkAutoConstructList.getInstance();

        private Long totalCapacityUsed;

        private Long totalCapacityProvisioned;

        private Long totalLogicalCapacityUsed;

        private MaxP95Performance maxP95Performance;

        private List<Recommendation> recommendations = DefaultSdkAutoConstructList.getInstance();

        private Long nfsExportedVolumes;

        private String recommendationStatus;

        private Long totalSnapshotCapacityUsed;

        private Long lunCount;

        private BuilderImpl() {
        }

        private BuilderImpl(NetAppONTAPSVM model) {
            clusterUuid(model.clusterUuid);
            resourceId(model.resourceId);
            svmName(model.svmName);
            cifsShareCount(model.cifsShareCount);
            enabledProtocols(model.enabledProtocols);
            totalCapacityUsed(model.totalCapacityUsed);
            totalCapacityProvisioned(model.totalCapacityProvisioned);
            totalLogicalCapacityUsed(model.totalLogicalCapacityUsed);
            maxP95Performance(model.maxP95Performance);
            recommendations(model.recommendations);
            nfsExportedVolumes(model.nfsExportedVolumes);
            recommendationStatus(model.recommendationStatus);
            totalSnapshotCapacityUsed(model.totalSnapshotCapacityUsed);
            lunCount(model.lunCount);
        }

        public final String getClusterUuid() {
            return clusterUuid;
        }

        public final void setClusterUuid(String clusterUuid) {
            this.clusterUuid = clusterUuid;
        }

        @Override
        public final Builder clusterUuid(String clusterUuid) {
            this.clusterUuid = clusterUuid;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getSvmName() {
            return svmName;
        }

        public final void setSvmName(String svmName) {
            this.svmName = svmName;
        }

        @Override
        public final Builder svmName(String svmName) {
            this.svmName = svmName;
            return this;
        }

        public final Long getCifsShareCount() {
            return cifsShareCount;
        }

        public final void setCifsShareCount(Long cifsShareCount) {
            this.cifsShareCount = cifsShareCount;
        }

        @Override
        public final Builder cifsShareCount(Long cifsShareCount) {
            this.cifsShareCount = cifsShareCount;
            return this;
        }

        public final Collection<String> getEnabledProtocols() {
            if (enabledProtocols instanceof SdkAutoConstructList) {
                return null;
            }
            return enabledProtocols;
        }

        public final void setEnabledProtocols(Collection<String> enabledProtocols) {
            this.enabledProtocols = EnabledProtocolsCopier.copy(enabledProtocols);
        }

        @Override
        public final Builder enabledProtocols(Collection<String> enabledProtocols) {
            this.enabledProtocols = EnabledProtocolsCopier.copy(enabledProtocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledProtocols(String... enabledProtocols) {
            enabledProtocols(Arrays.asList(enabledProtocols));
            return this;
        }

        public final Long getTotalCapacityUsed() {
            return totalCapacityUsed;
        }

        public final void setTotalCapacityUsed(Long totalCapacityUsed) {
            this.totalCapacityUsed = totalCapacityUsed;
        }

        @Override
        public final Builder totalCapacityUsed(Long totalCapacityUsed) {
            this.totalCapacityUsed = totalCapacityUsed;
            return this;
        }

        public final Long getTotalCapacityProvisioned() {
            return totalCapacityProvisioned;
        }

        public final void setTotalCapacityProvisioned(Long totalCapacityProvisioned) {
            this.totalCapacityProvisioned = totalCapacityProvisioned;
        }

        @Override
        public final Builder totalCapacityProvisioned(Long totalCapacityProvisioned) {
            this.totalCapacityProvisioned = totalCapacityProvisioned;
            return this;
        }

        public final Long getTotalLogicalCapacityUsed() {
            return totalLogicalCapacityUsed;
        }

        public final void setTotalLogicalCapacityUsed(Long totalLogicalCapacityUsed) {
            this.totalLogicalCapacityUsed = totalLogicalCapacityUsed;
        }

        @Override
        public final Builder totalLogicalCapacityUsed(Long totalLogicalCapacityUsed) {
            this.totalLogicalCapacityUsed = totalLogicalCapacityUsed;
            return this;
        }

        public final MaxP95Performance.Builder getMaxP95Performance() {
            return maxP95Performance != null ? maxP95Performance.toBuilder() : null;
        }

        public final void setMaxP95Performance(MaxP95Performance.BuilderImpl maxP95Performance) {
            this.maxP95Performance = maxP95Performance != null ? maxP95Performance.build() : null;
        }

        @Override
        public final Builder maxP95Performance(MaxP95Performance maxP95Performance) {
            this.maxP95Performance = maxP95Performance;
            return this;
        }

        public final List<Recommendation.Builder> getRecommendations() {
            List<Recommendation.Builder> result = RecommendationsCopier.copyToBuilder(this.recommendations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendations(Collection<Recommendation.BuilderImpl> recommendations) {
            this.recommendations = RecommendationsCopier.copyFromBuilder(recommendations);
        }

        @Override
        public final Builder recommendations(Collection<Recommendation> recommendations) {
            this.recommendations = RecommendationsCopier.copy(recommendations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Recommendation... recommendations) {
            recommendations(Arrays.asList(recommendations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Consumer<Recommendation.Builder>... recommendations) {
            recommendations(Stream.of(recommendations).map(c -> Recommendation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Long getNfsExportedVolumes() {
            return nfsExportedVolumes;
        }

        public final void setNfsExportedVolumes(Long nfsExportedVolumes) {
            this.nfsExportedVolumes = nfsExportedVolumes;
        }

        @Override
        public final Builder nfsExportedVolumes(Long nfsExportedVolumes) {
            this.nfsExportedVolumes = nfsExportedVolumes;
            return this;
        }

        public final String getRecommendationStatus() {
            return recommendationStatus;
        }

        public final void setRecommendationStatus(String recommendationStatus) {
            this.recommendationStatus = recommendationStatus;
        }

        @Override
        public final Builder recommendationStatus(String recommendationStatus) {
            this.recommendationStatus = recommendationStatus;
            return this;
        }

        @Override
        public final Builder recommendationStatus(RecommendationStatus recommendationStatus) {
            this.recommendationStatus(recommendationStatus == null ? null : recommendationStatus.toString());
            return this;
        }

        public final Long getTotalSnapshotCapacityUsed() {
            return totalSnapshotCapacityUsed;
        }

        public final void setTotalSnapshotCapacityUsed(Long totalSnapshotCapacityUsed) {
            this.totalSnapshotCapacityUsed = totalSnapshotCapacityUsed;
        }

        @Override
        public final Builder totalSnapshotCapacityUsed(Long totalSnapshotCapacityUsed) {
            this.totalSnapshotCapacityUsed = totalSnapshotCapacityUsed;
            return this;
        }

        public final Long getLunCount() {
            return lunCount;
        }

        public final void setLunCount(Long lunCount) {
            this.lunCount = lunCount;
        }

        @Override
        public final Builder lunCount(Long lunCount) {
            this.lunCount = lunCount;
            return this;
        }

        @Override
        public NetAppONTAPSVM build() {
            return new NetAppONTAPSVM(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
