/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A cache policy configuration.
 * </p>
 * <p>
 * This configuration determines the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The values that CloudFront includes in the cache key. These values can include HTTP headers, cookies, and URL query
 * strings. CloudFront uses the cache key to find an object in its cache that it can return to the viewer.
 * </p>
 * </li>
 * <li>
 * <p>
 * The default, minimum, and maximum time to live (TTL) values that you want objects to stay in the CloudFront cache.
 * </p>
 * </li>
 * </ul>
 * <p>
 * The headers, cookies, and query strings that are included in the cache key are also included in requests that
 * CloudFront sends to the origin. CloudFront sends a request when it can't find a valid object in its cache that
 * matches the request's cache key. If you want to send values to the origin but <i>not</i> include them in the cache
 * key, use <code>OriginRequestPolicy</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CachePolicyConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<CachePolicyConfig.Builder, CachePolicyConfig> {
    private static final SdkField<String> COMMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Comment")
            .getter(getter(CachePolicyConfig::comment))
            .setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment")
                    .unmarshallLocationName("Comment").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(CachePolicyConfig::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<Long> DEFAULT_TTL_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("DefaultTTL")
            .getter(getter(CachePolicyConfig::defaultTTL))
            .setter(setter(Builder::defaultTTL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultTTL")
                    .unmarshallLocationName("DefaultTTL").build()).build();

    private static final SdkField<Long> MAX_TTL_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MaxTTL")
            .getter(getter(CachePolicyConfig::maxTTL))
            .setter(setter(Builder::maxTTL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxTTL")
                    .unmarshallLocationName("MaxTTL").build()).build();

    private static final SdkField<Long> MIN_TTL_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MinTTL")
            .getter(getter(CachePolicyConfig::minTTL))
            .setter(setter(Builder::minTTL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinTTL")
                    .unmarshallLocationName("MinTTL").build()).build();

    private static final SdkField<ParametersInCacheKeyAndForwardedToOrigin> PARAMETERS_IN_CACHE_KEY_AND_FORWARDED_TO_ORIGIN_FIELD = SdkField
            .<ParametersInCacheKeyAndForwardedToOrigin> builder(MarshallingType.SDK_POJO)
            .memberName("ParametersInCacheKeyAndForwardedToOrigin")
            .getter(getter(CachePolicyConfig::parametersInCacheKeyAndForwardedToOrigin))
            .setter(setter(Builder::parametersInCacheKeyAndForwardedToOrigin))
            .constructor(ParametersInCacheKeyAndForwardedToOrigin::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("ParametersInCacheKeyAndForwardedToOrigin")
                    .unmarshallLocationName("ParametersInCacheKeyAndForwardedToOrigin").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMENT_FIELD, NAME_FIELD,
            DEFAULT_TTL_FIELD, MAX_TTL_FIELD, MIN_TTL_FIELD, PARAMETERS_IN_CACHE_KEY_AND_FORWARDED_TO_ORIGIN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String comment;

    private final String name;

    private final Long defaultTTL;

    private final Long maxTTL;

    private final Long minTTL;

    private final ParametersInCacheKeyAndForwardedToOrigin parametersInCacheKeyAndForwardedToOrigin;

    private CachePolicyConfig(BuilderImpl builder) {
        this.comment = builder.comment;
        this.name = builder.name;
        this.defaultTTL = builder.defaultTTL;
        this.maxTTL = builder.maxTTL;
        this.minTTL = builder.minTTL;
        this.parametersInCacheKeyAndForwardedToOrigin = builder.parametersInCacheKeyAndForwardedToOrigin;
    }

    /**
     * <p>
     * A comment to describe the cache policy. The comment cannot be longer than 128 characters.
     * </p>
     * 
     * @return A comment to describe the cache policy. The comment cannot be longer than 128 characters.
     */
    public final String comment() {
        return comment;
    }

    /**
     * <p>
     * A unique name to identify the cache policy.
     * </p>
     * 
     * @return A unique name to identify the cache policy.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The default amount of time, in seconds, that you want objects to stay in the CloudFront cache before CloudFront
     * sends another request to the origin to see if the object has been updated. CloudFront uses this value as the
     * object's time to live (TTL) only when the origin does <i>not</i> send <code>Cache-Control</code> or
     * <code>Expires</code> headers with the object. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How Long
     * Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * The default value for this field is 86400 seconds (one day). If the value of <code>MinTTL</code> is more than
     * 86400 seconds, then the default value for this field is the same as the value of <code>MinTTL</code>.
     * </p>
     * 
     * @return The default amount of time, in seconds, that you want objects to stay in the CloudFront cache before
     *         CloudFront sends another request to the origin to see if the object has been updated. CloudFront uses
     *         this value as the object's time to live (TTL) only when the origin does <i>not</i> send
     *         <code>Cache-Control</code> or <code>Expires</code> headers with the object. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How
     *         Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.</p>
     *         <p>
     *         The default value for this field is 86400 seconds (one day). If the value of <code>MinTTL</code> is more
     *         than 86400 seconds, then the default value for this field is the same as the value of <code>MinTTL</code>.
     */
    public final Long defaultTTL() {
        return defaultTTL;
    }

    /**
     * <p>
     * The maximum amount of time, in seconds, that objects stay in the CloudFront cache before CloudFront sends another
     * request to the origin to see if the object has been updated. CloudFront uses this value only when the origin
     * sends <code>Cache-Control</code> or <code>Expires</code> headers with the object. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How Long
     * Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * <p>
     * The default value for this field is 31536000 seconds (one year). If the value of <code>MinTTL</code> or
     * <code>DefaultTTL</code> is more than 31536000 seconds, then the default value for this field is the same as the
     * value of <code>DefaultTTL</code>.
     * </p>
     * 
     * @return The maximum amount of time, in seconds, that objects stay in the CloudFront cache before CloudFront sends
     *         another request to the origin to see if the object has been updated. CloudFront uses this value only when
     *         the origin sends <code>Cache-Control</code> or <code>Expires</code> headers with the object. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How
     *         Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.</p>
     *         <p>
     *         The default value for this field is 31536000 seconds (one year). If the value of <code>MinTTL</code> or
     *         <code>DefaultTTL</code> is more than 31536000 seconds, then the default value for this field is the same
     *         as the value of <code>DefaultTTL</code>.
     */
    public final Long maxTTL() {
        return maxTTL;
    }

    /**
     * <p>
     * The minimum amount of time, in seconds, that you want objects to stay in the CloudFront cache before CloudFront
     * sends another request to the origin to see if the object has been updated. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How Long
     * Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The minimum amount of time, in seconds, that you want objects to stay in the CloudFront cache before
     *         CloudFront sends another request to the origin to see if the object has been updated. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How
     *         Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final Long minTTL() {
        return minTTL;
    }

    /**
     * <p>
     * The HTTP headers, cookies, and URL query strings to include in the cache key. The values included in the cache
     * key are also included in requests that CloudFront sends to the origin.
     * </p>
     * 
     * @return The HTTP headers, cookies, and URL query strings to include in the cache key. The values included in the
     *         cache key are also included in requests that CloudFront sends to the origin.
     */
    public final ParametersInCacheKeyAndForwardedToOrigin parametersInCacheKeyAndForwardedToOrigin() {
        return parametersInCacheKeyAndForwardedToOrigin;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(defaultTTL());
        hashCode = 31 * hashCode + Objects.hashCode(maxTTL());
        hashCode = 31 * hashCode + Objects.hashCode(minTTL());
        hashCode = 31 * hashCode + Objects.hashCode(parametersInCacheKeyAndForwardedToOrigin());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CachePolicyConfig)) {
            return false;
        }
        CachePolicyConfig other = (CachePolicyConfig) obj;
        return Objects.equals(comment(), other.comment()) && Objects.equals(name(), other.name())
                && Objects.equals(defaultTTL(), other.defaultTTL()) && Objects.equals(maxTTL(), other.maxTTL())
                && Objects.equals(minTTL(), other.minTTL())
                && Objects.equals(parametersInCacheKeyAndForwardedToOrigin(), other.parametersInCacheKeyAndForwardedToOrigin());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CachePolicyConfig").add("Comment", comment()).add("Name", name())
                .add("DefaultTTL", defaultTTL()).add("MaxTTL", maxTTL()).add("MinTTL", minTTL())
                .add("ParametersInCacheKeyAndForwardedToOrigin", parametersInCacheKeyAndForwardedToOrigin()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DefaultTTL":
            return Optional.ofNullable(clazz.cast(defaultTTL()));
        case "MaxTTL":
            return Optional.ofNullable(clazz.cast(maxTTL()));
        case "MinTTL":
            return Optional.ofNullable(clazz.cast(minTTL()));
        case "ParametersInCacheKeyAndForwardedToOrigin":
            return Optional.ofNullable(clazz.cast(parametersInCacheKeyAndForwardedToOrigin()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CachePolicyConfig, T> g) {
        return obj -> g.apply((CachePolicyConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CachePolicyConfig> {
        /**
         * <p>
         * A comment to describe the cache policy. The comment cannot be longer than 128 characters.
         * </p>
         * 
         * @param comment
         *        A comment to describe the cache policy. The comment cannot be longer than 128 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * A unique name to identify the cache policy.
         * </p>
         * 
         * @param name
         *        A unique name to identify the cache policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The default amount of time, in seconds, that you want objects to stay in the CloudFront cache before
         * CloudFront sends another request to the origin to see if the object has been updated. CloudFront uses this
         * value as the object's time to live (TTL) only when the origin does <i>not</i> send <code>Cache-Control</code>
         * or <code>Expires</code> headers with the object. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How Long
         * Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * <p>
         * The default value for this field is 86400 seconds (one day). If the value of <code>MinTTL</code> is more than
         * 86400 seconds, then the default value for this field is the same as the value of <code>MinTTL</code>.
         * </p>
         * 
         * @param defaultTTL
         *        The default amount of time, in seconds, that you want objects to stay in the CloudFront cache before
         *        CloudFront sends another request to the origin to see if the object has been updated. CloudFront uses
         *        this value as the object's time to live (TTL) only when the origin does <i>not</i> send
         *        <code>Cache-Control</code> or <code>Expires</code> headers with the object. For more information, see
         *        <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing
         *        How Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer
         *        Guide</i>.</p>
         *        <p>
         *        The default value for this field is 86400 seconds (one day). If the value of <code>MinTTL</code> is
         *        more than 86400 seconds, then the default value for this field is the same as the value of
         *        <code>MinTTL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultTTL(Long defaultTTL);

        /**
         * <p>
         * The maximum amount of time, in seconds, that objects stay in the CloudFront cache before CloudFront sends
         * another request to the origin to see if the object has been updated. CloudFront uses this value only when the
         * origin sends <code>Cache-Control</code> or <code>Expires</code> headers with the object. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How Long
         * Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * <p>
         * The default value for this field is 31536000 seconds (one year). If the value of <code>MinTTL</code> or
         * <code>DefaultTTL</code> is more than 31536000 seconds, then the default value for this field is the same as
         * the value of <code>DefaultTTL</code>.
         * </p>
         * 
         * @param maxTTL
         *        The maximum amount of time, in seconds, that objects stay in the CloudFront cache before CloudFront
         *        sends another request to the origin to see if the object has been updated. CloudFront uses this value
         *        only when the origin sends <code>Cache-Control</code> or <code>Expires</code> headers with the object.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How
         *        Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer
         *        Guide</i>.</p>
         *        <p>
         *        The default value for this field is 31536000 seconds (one year). If the value of <code>MinTTL</code>
         *        or <code>DefaultTTL</code> is more than 31536000 seconds, then the default value for this field is the
         *        same as the value of <code>DefaultTTL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTTL(Long maxTTL);

        /**
         * <p>
         * The minimum amount of time, in seconds, that you want objects to stay in the CloudFront cache before
         * CloudFront sends another request to the origin to see if the object has been updated. For more information,
         * see <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How
         * Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param minTTL
         *        The minimum amount of time, in seconds, that you want objects to stay in the CloudFront cache before
         *        CloudFront sends another request to the origin to see if the object has been updated. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html">Managing How
         *        Long Content Stays in an Edge Cache (Expiration)</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minTTL(Long minTTL);

        /**
         * <p>
         * The HTTP headers, cookies, and URL query strings to include in the cache key. The values included in the
         * cache key are also included in requests that CloudFront sends to the origin.
         * </p>
         * 
         * @param parametersInCacheKeyAndForwardedToOrigin
         *        The HTTP headers, cookies, and URL query strings to include in the cache key. The values included in
         *        the cache key are also included in requests that CloudFront sends to the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parametersInCacheKeyAndForwardedToOrigin(
                ParametersInCacheKeyAndForwardedToOrigin parametersInCacheKeyAndForwardedToOrigin);

        /**
         * <p>
         * The HTTP headers, cookies, and URL query strings to include in the cache key. The values included in the
         * cache key are also included in requests that CloudFront sends to the origin.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ParametersInCacheKeyAndForwardedToOrigin.Builder} avoiding the need to create one manually via
         * {@link ParametersInCacheKeyAndForwardedToOrigin#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ParametersInCacheKeyAndForwardedToOrigin.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #parametersInCacheKeyAndForwardedToOrigin(ParametersInCacheKeyAndForwardedToOrigin)}.
         * 
         * @param parametersInCacheKeyAndForwardedToOrigin
         *        a consumer that will call methods on {@link ParametersInCacheKeyAndForwardedToOrigin.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parametersInCacheKeyAndForwardedToOrigin(ParametersInCacheKeyAndForwardedToOrigin)
         */
        default Builder parametersInCacheKeyAndForwardedToOrigin(
                Consumer<ParametersInCacheKeyAndForwardedToOrigin.Builder> parametersInCacheKeyAndForwardedToOrigin) {
            return parametersInCacheKeyAndForwardedToOrigin(ParametersInCacheKeyAndForwardedToOrigin.builder()
                    .applyMutation(parametersInCacheKeyAndForwardedToOrigin).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String comment;

        private String name;

        private Long defaultTTL;

        private Long maxTTL;

        private Long minTTL;

        private ParametersInCacheKeyAndForwardedToOrigin parametersInCacheKeyAndForwardedToOrigin;

        private BuilderImpl() {
        }

        private BuilderImpl(CachePolicyConfig model) {
            comment(model.comment);
            name(model.name);
            defaultTTL(model.defaultTTL);
            maxTTL(model.maxTTL);
            minTTL(model.minTTL);
            parametersInCacheKeyAndForwardedToOrigin(model.parametersInCacheKeyAndForwardedToOrigin);
        }

        public final String getComment() {
            return comment;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Long getDefaultTTL() {
            return defaultTTL;
        }

        public final void setDefaultTTL(Long defaultTTL) {
            this.defaultTTL = defaultTTL;
        }

        @Override
        public final Builder defaultTTL(Long defaultTTL) {
            this.defaultTTL = defaultTTL;
            return this;
        }

        public final Long getMaxTTL() {
            return maxTTL;
        }

        public final void setMaxTTL(Long maxTTL) {
            this.maxTTL = maxTTL;
        }

        @Override
        public final Builder maxTTL(Long maxTTL) {
            this.maxTTL = maxTTL;
            return this;
        }

        public final Long getMinTTL() {
            return minTTL;
        }

        public final void setMinTTL(Long minTTL) {
            this.minTTL = minTTL;
        }

        @Override
        public final Builder minTTL(Long minTTL) {
            this.minTTL = minTTL;
            return this;
        }

        public final ParametersInCacheKeyAndForwardedToOrigin.Builder getParametersInCacheKeyAndForwardedToOrigin() {
            return parametersInCacheKeyAndForwardedToOrigin != null ? parametersInCacheKeyAndForwardedToOrigin.toBuilder() : null;
        }

        public final void setParametersInCacheKeyAndForwardedToOrigin(
                ParametersInCacheKeyAndForwardedToOrigin.BuilderImpl parametersInCacheKeyAndForwardedToOrigin) {
            this.parametersInCacheKeyAndForwardedToOrigin = parametersInCacheKeyAndForwardedToOrigin != null ? parametersInCacheKeyAndForwardedToOrigin
                    .build() : null;
        }

        @Override
        public final Builder parametersInCacheKeyAndForwardedToOrigin(
                ParametersInCacheKeyAndForwardedToOrigin parametersInCacheKeyAndForwardedToOrigin) {
            this.parametersInCacheKeyAndForwardedToOrigin = parametersInCacheKeyAndForwardedToOrigin;
            return this;
        }

        @Override
        public CachePolicyConfig build() {
            return new CachePolicyConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
