/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigatewaymanagementapi;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.apigatewaymanagementapi.internal.ApiGatewayManagementApiServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.ApiGatewayManagementApiException;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.ForbiddenException;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.GetConnectionRequest;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.GetConnectionResponse;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.GoneException;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.LimitExceededException;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.PayloadTooLargeException;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.PostToConnectionRequest;
import software.amazon.awssdk.services.apigatewaymanagementapi.model.PostToConnectionResponse;
import software.amazon.awssdk.services.apigatewaymanagementapi.transform.DeleteConnectionRequestMarshaller;
import software.amazon.awssdk.services.apigatewaymanagementapi.transform.GetConnectionRequestMarshaller;
import software.amazon.awssdk.services.apigatewaymanagementapi.transform.PostToConnectionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ApiGatewayManagementApiClient}.
 *
 * @see ApiGatewayManagementApiClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApiGatewayManagementApiClient implements ApiGatewayManagementApiClient {
    private static final Logger log = Logger.loggerFor(DefaultApiGatewayManagementApiClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ApiGatewayManagementApiServiceClientConfiguration serviceClientConfiguration;

    protected DefaultApiGatewayManagementApiClient(ApiGatewayManagementApiServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Delete the connection with the provided id.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return Result of the DeleteConnection operation returned by the service.
     * @throws GoneException
     *         The connection with the provided id no longer exists.
     * @throws LimitExceededException
     *         The client is sending more than the allowed number of requests per unit of time or the WebSocket client
     *         side buffer is full.
     * @throws ForbiddenException
     *         The caller is not authorized to invoke this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApiGatewayManagementApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApiGatewayManagementApiClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apigatewaymanagementapi-2018-11-29/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConnectionResponse deleteConnection(DeleteConnectionRequest deleteConnectionRequest) throws GoneException,
            LimitExceededException, ForbiddenException, AwsServiceException, SdkClientException, ApiGatewayManagementApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApiGatewayManagementApi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnection");

            return clientHandler.execute(new ClientExecutionParams<DeleteConnectionRequest, DeleteConnectionResponse>()
                    .withOperationName("DeleteConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteConnectionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get information about the connection with the provided id.
     * </p>
     *
     * @param getConnectionRequest
     * @return Result of the GetConnection operation returned by the service.
     * @throws GoneException
     *         The connection with the provided id no longer exists.
     * @throws LimitExceededException
     *         The client is sending more than the allowed number of requests per unit of time or the WebSocket client
     *         side buffer is full.
     * @throws ForbiddenException
     *         The caller is not authorized to invoke this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApiGatewayManagementApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApiGatewayManagementApiClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apigatewaymanagementapi-2018-11-29/GetConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConnectionResponse getConnection(GetConnectionRequest getConnectionRequest) throws GoneException,
            LimitExceededException, ForbiddenException, AwsServiceException, SdkClientException, ApiGatewayManagementApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApiGatewayManagementApi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnection");

            return clientHandler.execute(new ClientExecutionParams<GetConnectionRequest, GetConnectionResponse>()
                    .withOperationName("GetConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getConnectionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends the provided data to the specified connection.
     * </p>
     *
     * @param postToConnectionRequest
     * @return Result of the PostToConnection operation returned by the service.
     * @throws GoneException
     *         The connection with the provided id no longer exists.
     * @throws LimitExceededException
     *         The client is sending more than the allowed number of requests per unit of time or the WebSocket client
     *         side buffer is full.
     * @throws PayloadTooLargeException
     *         The data has exceeded the maximum size allowed.
     * @throws ForbiddenException
     *         The caller is not authorized to invoke this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApiGatewayManagementApiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApiGatewayManagementApiClient.PostToConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apigatewaymanagementapi-2018-11-29/PostToConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PostToConnectionResponse postToConnection(PostToConnectionRequest postToConnectionRequest) throws GoneException,
            LimitExceededException, PayloadTooLargeException, ForbiddenException, AwsServiceException, SdkClientException,
            ApiGatewayManagementApiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PostToConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PostToConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postToConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postToConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApiGatewayManagementApi");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostToConnection");

            return clientHandler.execute(new ClientExecutionParams<PostToConnectionRequest, PostToConnectionResponse>()
                    .withOperationName("PostToConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(postToConnectionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PostToConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        ApiGatewayManagementApiServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = ApiGatewayManagementApiServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApiGatewayManagementApiException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PayloadTooLargeException")
                                .exceptionBuilderSupplier(PayloadTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GoneException").exceptionBuilderSupplier(GoneException::builder)
                                .httpStatusCode(410).build());
    }

    @Override
    public final ApiGatewayManagementApiServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
