package cdm.product.asset.validation.datarule;

import cdm.product.asset.VarianceCapFloor;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.math.BigDecimal;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.0.0
 */
@RosettaDataRule("VarianceCapFloorPositiveUnadjustedVarianceCap")
@ImplementedBy(VarianceCapFloorPositiveUnadjustedVarianceCap.Default.class)
public interface VarianceCapFloorPositiveUnadjustedVarianceCap extends Validator<VarianceCapFloor> {
	
	String NAME = "VarianceCapFloorPositiveUnadjustedVarianceCap";
	String DEFINITION = "if unadjustedVarianceCap exists then unadjustedVarianceCap > 0";
	
	ValidationResult<VarianceCapFloor> validate(RosettaPath path, VarianceCapFloor varianceCapFloor);
	
	class Default implements VarianceCapFloorPositiveUnadjustedVarianceCap {
	
		@Override
		public ValidationResult<VarianceCapFloor> validate(RosettaPath path, VarianceCapFloor varianceCapFloor) {
			ComparisonResult result = executeDataRule(varianceCapFloor);
			if (result.get()) {
				return ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "VarianceCapFloor", path, DEFINITION);
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null) {
				failureMessage = "Condition " + NAME + " failed.";
			}
			return ValidationResult.failure(NAME, ValidationType.DATA_RULE, "VarianceCapFloor", path, DEFINITION, failureMessage);
		}
		
		private ComparisonResult executeDataRule(VarianceCapFloor varianceCapFloor) {
			try {
				ComparisonResult result = MapperUtils.toComparisonResult(MapperUtils.runSingle(() -> {
					if (exists(MapperS.of(varianceCapFloor).<BigDecimal>map("getUnadjustedVarianceCap", _varianceCapFloor -> _varianceCapFloor.getUnadjustedVarianceCap())).getOrDefault(false)) {
						return greaterThan(MapperS.of(varianceCapFloor).<BigDecimal>map("getUnadjustedVarianceCap", _varianceCapFloor -> _varianceCapFloor.getUnadjustedVarianceCap()), MapperS.of(Integer.valueOf(0)), CardinalityOperator.All);
					}
					else {
						return null;
					}
				}));
				return result.get() == null ? ComparisonResult.success() : result;
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements VarianceCapFloorPositiveUnadjustedVarianceCap {
	
		@Override
		public ValidationResult<VarianceCapFloor> validate(RosettaPath path, VarianceCapFloor varianceCapFloor) {
			return ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "VarianceCapFloor", path, DEFINITION);
		}
	}
}
