package cdm.product.asset.floatingrate.functions;

import cdm.base.math.Rounding;
import cdm.base.math.RoundingDirectionEnum;
import cdm.base.math.functions.RoundToPrecision;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.math.BigDecimal;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(ApplyFinalRateRounding.ApplyFinalRateRoundingDefault.class)
public abstract class ApplyFinalRateRounding implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected RoundToPrecision roundToPrecision;

	/**
	* @param baseRate Rate before rounding.
	* @param finalRateRounding type of rounding (precision and direction).
	* @return roundedRate rate after rounding.
	*/
	public BigDecimal evaluate(BigDecimal baseRate, Rounding finalRateRounding) {
		BigDecimal roundedRate = doEvaluate(baseRate, finalRateRounding);
		
		return roundedRate;
	}

	protected abstract BigDecimal doEvaluate(BigDecimal baseRate, Rounding finalRateRounding);

	protected abstract Mapper<Integer> precision(BigDecimal baseRate, Rounding finalRateRounding);

	protected abstract Mapper<RoundingDirectionEnum> direction(BigDecimal baseRate, Rounding finalRateRounding);

	public static class ApplyFinalRateRoundingDefault extends ApplyFinalRateRounding {
		@Override
		protected BigDecimal doEvaluate(BigDecimal baseRate, Rounding finalRateRounding) {
			BigDecimal roundedRate = null;
			return assignOutput(roundedRate, baseRate, finalRateRounding);
		}
		
		protected BigDecimal assignOutput(BigDecimal roundedRate, BigDecimal baseRate, Rounding finalRateRounding) {
			roundedRate = MapperS.of(roundToPrecision.evaluate(MapperS.of(baseRate).get(), MapperS.of(precision(baseRate, finalRateRounding).get()).get(), MapperS.of(direction(baseRate, finalRateRounding).get()).get())).get();
			
			return roundedRate;
		}
		
		@Override
		protected Mapper<Integer> precision(BigDecimal baseRate, Rounding finalRateRounding) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(finalRateRounding).<Integer>map("getPrecision", rounding -> rounding.getPrecision())).getOrDefault(false)) {
					return MapperS.of(finalRateRounding).<Integer>map("getPrecision", rounding -> rounding.getPrecision());
				}
				else {
					return MapperS.of(Integer.valueOf(7));
				}
			});
		}
		
		@Override
		protected Mapper<RoundingDirectionEnum> direction(BigDecimal baseRate, Rounding finalRateRounding) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(finalRateRounding).<RoundingDirectionEnum>map("getRoundingDirection", rounding -> rounding.getRoundingDirection())).getOrDefault(false)) {
					return MapperS.of(finalRateRounding).<RoundingDirectionEnum>map("getRoundingDirection", rounding -> rounding.getRoundingDirection());
				}
				else {
					return MapperS.of(RoundingDirectionEnum.NEAREST);
				}
			});
		}
	}
}
