package cdm.observable.asset.calculatedrate.functions;

import cdm.base.math.ArithmeticOperationEnum;
import cdm.base.math.functions.VectorScalarOperation;
import cdm.observable.asset.calculatedrate.FloatingRateCalculationParameters;
import cdm.observable.asset.calculatedrate.ObservationParameters;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(ProcessObservations.ProcessObservationsDefault.class)
public abstract class ProcessObservations implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected VectorScalarOperation vectorScalarOperation;

	/**
	* @param calculationParameters Floating rate definition for the calculated rate.
	* @param rawObservations 
	* @return processedObservations 
	*/
	public List<BigDecimal> evaluate(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
		List<BigDecimal> processedObservations = doEvaluate(calculationParameters, rawObservations);
		
		return processedObservations;
	}

	protected abstract List<BigDecimal> doEvaluate(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations);

	protected abstract Mapper<? extends ObservationParameters> params(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations);

	protected abstract Mapper<BigDecimal> cap(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations);

	protected abstract Mapper<BigDecimal> floor(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations);

	protected abstract Mapper<BigDecimal> cappedObservations(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations);

	protected abstract Mapper<BigDecimal> flooredObservations(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations);

	public static class ProcessObservationsDefault extends ProcessObservations {
		@Override
		protected List<BigDecimal> doEvaluate(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			List<BigDecimal> processedObservations = new ArrayList<>();
			return assignOutput(processedObservations, calculationParameters, rawObservations);
		}
		
		protected List<BigDecimal> assignOutput(List<BigDecimal> processedObservations, FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			List<BigDecimal> addVar = MapperC.<BigDecimal>of(flooredObservations(calculationParameters, rawObservations).getMulti()).getMulti();
			processedObservations.addAll(addVar);
			
			return processedObservations;
		}
		
		@Override
		protected Mapper<? extends ObservationParameters> params(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			return MapperS.of(calculationParameters).<ObservationParameters>map("getObservationParameters", floatingRateCalculationParameters -> floatingRateCalculationParameters.getObservationParameters());
		}
		
		@Override
		protected Mapper<BigDecimal> cap(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(params(calculationParameters, rawObservations).get())).getOrDefault(false)) {
					return MapperS.of(params(calculationParameters, rawObservations).get()).<BigDecimal>map("getObservationCapRate", observationParameters -> observationParameters.getObservationCapRate());
				}
				else {
					return null;
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> floor(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(params(calculationParameters, rawObservations).get())).getOrDefault(false)) {
					return MapperS.of(params(calculationParameters, rawObservations).get()).<BigDecimal>map("getObservationFloorRate", observationParameters -> observationParameters.getObservationFloorRate());
				}
				else {
					return null;
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> cappedObservations(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			return MapperUtils.runMulti(() -> {
				if (exists(MapperS.of(cap(calculationParameters, rawObservations).get())).getOrDefault(false)) {
					return MapperC.<BigDecimal>of(vectorScalarOperation.evaluate(MapperS.of(ArithmeticOperationEnum.MIN).get(), MapperC.<BigDecimal>of(rawObservations).getMulti(), MapperS.of(cap(calculationParameters, rawObservations).get()).get()));
				}
				else {
					return MapperC.<BigDecimal>of(rawObservations);
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> flooredObservations(FloatingRateCalculationParameters calculationParameters, List<BigDecimal> rawObservations) {
			return MapperUtils.runMulti(() -> {
				if (exists(MapperS.of(floor(calculationParameters, rawObservations).get())).getOrDefault(false)) {
					return MapperC.<BigDecimal>of(vectorScalarOperation.evaluate(MapperS.of(ArithmeticOperationEnum.MAX).get(), MapperC.<BigDecimal>of(cappedObservations(calculationParameters, rawObservations).getMulti()).getMulti(), MapperS.of(floor(calculationParameters, rawObservations).get()).get()));
				}
				else {
					return MapperC.<BigDecimal>of(cappedObservations(calculationParameters, rawObservations).getMulti());
				}
			});
		}
	}
}
