package cdm.observable.asset.calculatedrate.functions;

import cdm.base.datetime.BusinessCenterEnum;
import cdm.base.datetime.BusinessCenters;
import cdm.base.datetime.functions.GetAllBusinessCenters;
import cdm.observable.asset.calculatedrate.CalculatedRateObservationDatesAndWeights;
import cdm.observable.asset.calculatedrate.CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder;
import cdm.observable.asset.calculatedrate.FloatingRateCalculationParameters;
import cdm.observable.asset.calculatedrate.ObservationPeriodDatesEnum;
import cdm.observable.asset.calculatedrate.ObservationShiftCalculation;
import cdm.observable.asset.calculatedrate.OffsetCalculation;
import cdm.product.common.schedule.CalculationPeriodBase;
import cdm.product.common.schedule.ResetDates;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import com.rosetta.model.lib.records.Date;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GenerateObservationDatesAndWeights.GenerateObservationDatesAndWeightsDefault.class)
public abstract class GenerateObservationDatesAndWeights implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected ComputeCalculationPeriod computeCalculationPeriod;
	@Inject protected DetermineObservationPeriod determineObservationPeriod;
	@Inject protected GenerateObservationDates generateObservationDates;
	@Inject protected GenerateWeightings generateWeightings;
	@Inject protected GetAllBusinessCenters getAllBusinessCenters;

	/**
	* @param calculationParams Floating rate definition for the calculated rate.
	* @param resetDates Reset structure (needed only for fallback rates, otherwise will be empty.
	* @param calculationPeriod Calculation period for which we want to determine the rate.
	* @param priorCalculationPeriod The prior calculation period (needed only for set in advance observation shift rate.
	* @return results observation dates and corresponding weight.
	*/
	public CalculatedRateObservationDatesAndWeights evaluate(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
		CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder resultsBuilder = doEvaluate(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod);
		
		final CalculatedRateObservationDatesAndWeights results;
		if (resultsBuilder == null) {
			results = null;
		} else {
			results = resultsBuilder.build();
			objectValidator.validate(CalculatedRateObservationDatesAndWeights.class, results);
		}
		
		return results;
	}

	protected abstract CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder doEvaluate(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<? extends ObservationShiftCalculation> obsShift(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<? extends OffsetCalculation> lockout(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<Integer> specifiedLockout(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<Integer> lockoutDays(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<BusinessCenterEnum> businessDays(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<ObservationPeriodDatesEnum> calculateRelative(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<? extends CalculationPeriodBase> adjustedCalculationPeriod(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<? extends CalculationPeriodBase> observationPeriod(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	protected abstract Mapper<Date> observationDates(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod);

	public static class GenerateObservationDatesAndWeightsDefault extends GenerateObservationDatesAndWeights {
		@Override
		protected CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder doEvaluate(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder results = CalculatedRateObservationDatesAndWeights.builder();
			return assignOutput(results, calculationParams, resetDates, calculationPeriod, priorCalculationPeriod);
		}
		
		protected CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder assignOutput(CalculatedRateObservationDatesAndWeights.CalculatedRateObservationDatesAndWeightsBuilder results, FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			results
				.addObservationDates(MapperC.<Date>of(observationDates(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).getMulti()).getMulti());
			
			results
				.addWeights(MapperC.<BigDecimal>of(generateWeightings.evaluate(MapperS.of(calculationParams).get(), MapperS.of(results).<Date>mapC("getObservationDates", calculatedRateObservationDatesAndWeights -> calculatedRateObservationDatesAndWeights.getObservationDates()).getMulti(), MapperS.of(observationPeriod(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get(), MapperS.of(adjustedCalculationPeriod(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get(), MapperS.of(lockoutDays(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get())).getMulti());
			
			return Optional.ofNullable(results)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<? extends ObservationShiftCalculation> obsShift(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperS.of(calculationParams).<ObservationShiftCalculation>map("getObservationShiftCalculation", floatingRateCalculationParameters -> floatingRateCalculationParameters.getObservationShiftCalculation());
		}
		
		@Override
		protected Mapper<? extends OffsetCalculation> lockout(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperS.of(calculationParams).<OffsetCalculation>map("getLockoutCalculation", floatingRateCalculationParameters -> floatingRateCalculationParameters.getLockoutCalculation());
		}
		
		@Override
		protected Mapper<Integer> specifiedLockout(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(lockout(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).<Integer>map("getOffsetDays", offsetCalculation -> offsetCalculation.getOffsetDays())).getOrDefault(false)) {
					return MapperS.of(lockout(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).<Integer>map("getOffsetDays", offsetCalculation -> offsetCalculation.getOffsetDays());
				}
				else {
					return MapperS.of(Integer.valueOf(5));
				}
			});
		}
		
		@Override
		protected Mapper<Integer> lockoutDays(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(lockout(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get())).getOrDefault(false)) {
					return MapperS.of(specifiedLockout(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get());
				}
				else {
					return MapperS.of(Integer.valueOf(0));
				}
			});
		}
		
		@Override
		protected Mapper<BusinessCenterEnum> businessDays(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperC.<BusinessCenterEnum>of(getAllBusinessCenters.evaluate(MapperS.of(calculationParams).<BusinessCenters>map("getApplicableBusinessDays", floatingRateCalculationParameters -> floatingRateCalculationParameters.getApplicableBusinessDays()).get()));
		}
		
		@Override
		protected Mapper<ObservationPeriodDatesEnum> calculateRelative(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(obsShift(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).<ObservationPeriodDatesEnum>map("getCalculationBase", observationShiftCalculation -> observationShiftCalculation.getCalculationBase())).getOrDefault(false)) {
					return MapperS.of(obsShift(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).<ObservationPeriodDatesEnum>map("getCalculationBase", observationShiftCalculation -> observationShiftCalculation.getCalculationBase());
				}
				else {
					return MapperS.of(ObservationPeriodDatesEnum.STANDARD);
				}
			});
		}
		
		@Override
		protected Mapper<? extends CalculationPeriodBase> adjustedCalculationPeriod(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperS.of(computeCalculationPeriod.evaluate(MapperS.of(calculationPeriod).get(), MapperS.of(priorCalculationPeriod).get(), MapperS.of(calculateRelative(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get(), MapperS.of(resetDates).get()));
		}
		
		@Override
		protected Mapper<? extends CalculationPeriodBase> observationPeriod(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperS.of(determineObservationPeriod.evaluate(MapperS.of(adjustedCalculationPeriod(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get(), MapperS.of(calculationParams).get()));
		}
		
		@Override
		protected Mapper<Date> observationDates(FloatingRateCalculationParameters calculationParams, ResetDates resetDates, CalculationPeriodBase calculationPeriod, CalculationPeriodBase priorCalculationPeriod) {
			return MapperC.<Date>of(generateObservationDates.evaluate(MapperS.of(observationPeriod(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get(), MapperC.<BusinessCenterEnum>of(businessDays(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).getMulti()).getMulti(), MapperS.of(lockoutDays(calculationParams, resetDates, calculationPeriod, priorCalculationPeriod).get()).get()));
		}
	}
}
