package cdm.event.common.functions;

import cdm.base.math.ArithmeticOperationEnum;
import cdm.base.math.DatedValue;
import cdm.base.math.QuantityChangeDirectionEnum;
import cdm.base.math.UnitType;
import cdm.base.staticdata.identifier.Identifier;
import cdm.event.common.QuantityChangeInstruction;
import cdm.event.common.QuantityChangeInstruction.QuantityChangeInstructionBuilder;
import cdm.observable.asset.CashPrice;
import cdm.observable.asset.Price;
import cdm.observable.asset.PriceComposite;
import cdm.observable.asset.PriceExpressionEnum;
import cdm.observable.asset.PriceSchedule;
import cdm.observable.asset.PriceTypeEnum;
import cdm.observable.asset.metafields.FieldWithMetaPriceSchedule;
import cdm.product.common.settlement.PriceQuantity;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.ConditionValidator;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(Create_OnDemandRateChangePriceChangeInstruction.Create_OnDemandRateChangePriceChangeInstructionDefault.class)
public abstract class Create_OnDemandRateChangePriceChangeInstruction implements RosettaFunction {
	
	@Inject protected ConditionValidator conditionValidator;
	
	@Inject protected ModelObjectValidator objectValidator;

	/**
	* @param priceQuantity The original price / quantity to be modified with the new rate.
	* @param newRate The new rate value, provided as a single number.
	* @return quantityChangeInstruction 
	*/
	public QuantityChangeInstruction evaluate(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate) {
		// pre-conditions
		conditionValidator.validate(() -> 
			exists(MapperS.of(currentRatePrice(priceQuantity, newRate).get())), 
				"There should be 1 and only 1 rate type price in the current price.");
		
		QuantityChangeInstruction.QuantityChangeInstructionBuilder quantityChangeInstructionBuilder = doEvaluate(priceQuantity, newRate);
		
		final QuantityChangeInstruction quantityChangeInstruction;
		if (quantityChangeInstructionBuilder == null) {
			quantityChangeInstruction = null;
		} else {
			quantityChangeInstruction = quantityChangeInstructionBuilder.build();
			objectValidator.validate(QuantityChangeInstruction.class, quantityChangeInstruction);
		}
		
		return quantityChangeInstruction;
	}

	protected abstract QuantityChangeInstruction.QuantityChangeInstructionBuilder doEvaluate(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate);

	protected abstract Mapper<? extends PriceSchedule> currentRatePrice(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate);

	protected abstract Mapper<? extends Price> newPrice(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate);

	protected abstract Mapper<? extends PriceQuantity> newPriceQuantity(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate);

	public static class Create_OnDemandRateChangePriceChangeInstructionDefault extends Create_OnDemandRateChangePriceChangeInstruction {
		@Override
		protected QuantityChangeInstruction.QuantityChangeInstructionBuilder doEvaluate(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate) {
			QuantityChangeInstruction.QuantityChangeInstructionBuilder quantityChangeInstruction = QuantityChangeInstruction.builder();
			return assignOutput(quantityChangeInstruction, priceQuantity, newRate);
		}
		
		protected QuantityChangeInstruction.QuantityChangeInstructionBuilder assignOutput(QuantityChangeInstruction.QuantityChangeInstructionBuilder quantityChangeInstruction, List<? extends PriceQuantity> priceQuantity, BigDecimal newRate) {
			quantityChangeInstruction = toBuilder(MapperS.of(QuantityChangeInstruction.builder()
				.setChange(MapperS.of(newPriceQuantity(priceQuantity, newRate).get()).getMulti())
				.setDirection(MapperS.of(QuantityChangeDirectionEnum.REPLACE).get())
				.setLotIdentifier(MapperC.<Identifier>ofNull().getMulti())
				.build())
			.get());
			
			return Optional.ofNullable(quantityChangeInstruction)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<? extends PriceSchedule> currentRatePrice(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate) {
			return MapperC.<PriceQuantity>of(priceQuantity)
				.mapItemToList(item -> (MapperC<PriceSchedule>)item.<FieldWithMetaPriceSchedule>mapC("getPrice", _priceQuantity -> _priceQuantity.getPrice()).<PriceSchedule>map("getValue", _f->_f.getValue()))
				.apply(item -> item
					.flattenList())
				.apply(item -> item
					.filterItemNullSafe(_item -> (Boolean)areEqual(_item.<PriceTypeEnum>map("getPriceType", priceSchedule -> priceSchedule.getPriceType()), MapperS.of(PriceTypeEnum.INTEREST_RATE), CardinalityOperator.All).get()))
				.apply(item -> MapperS.of(item.get()));
		}
		
		@Override
		protected Mapper<? extends Price> newPrice(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate) {
			return MapperS.of(Price.builder()
				.setValue(MapperS.of(newRate).get())
				.setUnit(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<UnitType>map("getUnit", measureBase -> measureBase.getUnit()).get())
				.setPerUnitOf(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<UnitType>map("getPerUnitOf", priceSchedule -> priceSchedule.getPerUnitOf()).get())
				.setPriceType(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<PriceTypeEnum>map("getPriceType", priceSchedule -> priceSchedule.getPriceType()).get())
				.setPriceExpression(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<PriceExpressionEnum>map("getPriceExpression", priceSchedule -> priceSchedule.getPriceExpression()).get())
				.setComposite(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<PriceComposite>map("getComposite", priceSchedule -> priceSchedule.getComposite()).get())
				.setArithmeticOperator(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<ArithmeticOperationEnum>map("getArithmeticOperator", priceSchedule -> priceSchedule.getArithmeticOperator()).get())
				.setCashPrice(MapperS.of(currentRatePrice(priceQuantity, newRate).get()).<CashPrice>map("getCashPrice", priceSchedule -> priceSchedule.getCashPrice()).get())
				.setDatedValue(MapperC.<DatedValue>ofNull().getMulti())
				.build())
			;
		}
		
		@Override
		protected Mapper<? extends PriceQuantity> newPriceQuantity(List<? extends PriceQuantity> priceQuantity, BigDecimal newRate) {
			return MapperS.of(PriceQuantity.builder()
				.setPriceValue(MapperS.of(newPrice(priceQuantity, newRate).get()).getMulti())
				.build())
			;
		}
	}
}
