package cdm.base.datetime.daycount.functions;

import cdm.base.datetime.daycount.DayCountFractionEnum;
import cdm.base.datetime.functions.DateDifference;
import cdm.base.datetime.functions.LeapYearDateDifference;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.MapperMaths;
import com.rosetta.model.lib.functions.IsLeapYear;
import com.rosetta.model.lib.functions.Min;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import com.rosetta.model.lib.records.Date;
import java.math.BigDecimal;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * The fraction of a year represented by a date range
 * @version 5.0.0
 */
public class YearFraction implements RosettaFunction {
	
	@Inject protected YearFraction.YearFraction_1_1 yearFraction_1_1;
	@Inject protected YearFraction.YearFractionACT_ACT_ISDA yearFractionACT_ACT_ISDA;
	@Inject protected YearFraction.YearFractionACT_ACT_ICMA yearFractionACT_ACT_ICMA;
	@Inject protected YearFraction.YearFractionACT_365_FIXED yearFractionACT_365_FIXED;
	@Inject protected YearFraction.YearFractionACT_360 yearFractionACT_360;
	@Inject protected YearFraction.YearFraction_30_360 yearFraction_30_360;
	@Inject protected YearFraction.YearFraction_30E_360 yearFraction_30E_360;
	@Inject protected YearFraction.YearFraction_30E_360_ISDA yearFraction_30E_360_ISDA;
	@Inject protected YearFraction.YearFractionACT_364 yearFractionACT_364;
	@Inject protected YearFraction.YearFractionACT_365L yearFractionACT_365L;
	@Inject protected YearFraction.YearFractionCAL_252 yearFractionCAL_252;
	
	public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
		switch (dayCountFractionEnum) {
			case _1_1:
				return yearFraction_1_1.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case ACT_ACT_ISDA:
				return yearFractionACT_ACT_ISDA.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case ACT_ACT_ICMA:
				return yearFractionACT_ACT_ICMA.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case ACT_365_FIXED:
				return yearFractionACT_365_FIXED.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case ACT_360:
				return yearFractionACT_360.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case _30_360:
				return yearFraction_30_360.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case _30E_360:
				return yearFraction_30E_360.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case _30E_360_ISDA:
				return yearFraction_30E_360_ISDA.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case ACT_364:
				return yearFractionACT_364.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case ACT_365L:
				return yearFractionACT_365L.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			case CAL_252:
				return yearFractionCAL_252.evaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			default:
				throw new IllegalArgumentException("Enum value not implemented: " + dayCountFractionEnum);
		}
	}
	
	@ImplementedBy(YearFraction_1_1.YearFraction_1_1Default.class)
	public static abstract class YearFraction_1_1 implements RosettaFunction {
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFraction_1_1Default extends YearFraction_1_1 {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperS.of(new BigDecimal("1.0")).get();
				
				return result;
			}
		}
	}
	@ImplementedBy(YearFractionACT_ACT_ISDA.YearFractionACT_ACT_ISDADefault.class)
	public static abstract class YearFractionACT_ACT_ISDA implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
		@Inject protected LeapYearDateDifference leapYearDateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInLeapYearPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInNonLeapPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionACT_ACT_ISDADefault extends YearFractionACT_ACT_ISDA {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, BigDecimal, BigDecimal>add(MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInNonLeapPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Integer.valueOf(365))), MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInLeapYearPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Integer.valueOf(366)))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
			
			@Override
			protected Mapper<Integer> daysInLeapYearPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(leapYearDateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
			
			@Override
			protected Mapper<Integer> daysInNonLeapPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(daysInLeapYearPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()));
			}
		}
	}
	@ImplementedBy(YearFractionACT_ACT_ICMA.YearFractionACT_ACT_ICMADefault.class)
	public static abstract class YearFractionACT_ACT_ICMA implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionACT_ACT_ICMADefault extends YearFractionACT_ACT_ICMA {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(periodsInYear))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
		}
	}
	@ImplementedBy(YearFractionACT_365_FIXED.YearFractionACT_365_FIXEDDefault.class)
	public static abstract class YearFractionACT_365_FIXED implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionACT_365_FIXEDDefault extends YearFractionACT_365_FIXED {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Integer.valueOf(365))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
		}
	}
	@ImplementedBy(YearFractionACT_360.YearFractionACT_360Default.class)
	public static abstract class YearFractionACT_360 implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionACT_360Default extends YearFractionACT_360 {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Integer.valueOf(360))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
		}
	}
	@ImplementedBy(YearFraction_30_360.YearFraction_30_360Default.class)
	public static abstract class YearFraction_30_360 implements RosettaFunction {
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFraction_30_360Default extends YearFraction_30_360 {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperMaths.<Integer, Integer, Integer>add(MapperMaths.<Integer, Integer, Integer>add(MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(Integer.valueOf(360)), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()))), MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(Integer.valueOf(30)), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endMonth(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startMonth(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get())))), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endDay(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startDay(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()))), MapperS.of(Integer.valueOf(360))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> startYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(startDate).<Integer>map("Year", Date::getYear);
			}
			
			@Override
			protected Mapper<Integer> endYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(endDate).<Integer>map("Year", Date::getYear);
			}
			
			@Override
			protected Mapper<Integer> startMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(startDate).<Integer>map("Month", Date::getMonth);
			}
			
			@Override
			protected Mapper<Integer> endMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(endDate).<Integer>map("Month", Date::getMonth);
			}
			
			@Override
			protected Mapper<Integer> endDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperUtils.runSingle(() -> {
					if (greaterThan(MapperS.of(startDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(29)), CardinalityOperator.All).getOrDefault(false)) {
						return MapperS.of(new Min().execute(MapperS.of(endDate).<Integer>map("Day", Date::getDay).get(), MapperS.of(Integer.valueOf(30)).get()));
					}
					else {
						return MapperS.of(endDate).<Integer>map("Day", Date::getDay);
					}
				});
			}
			
			@Override
			protected Mapper<Integer> startDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(new Min().execute(MapperS.of(startDate).<Integer>map("Day", Date::getDay).get(), MapperS.of(Integer.valueOf(30)).get()));
			}
		}
	}
	@ImplementedBy(YearFraction_30E_360.YearFraction_30E_360Default.class)
	public static abstract class YearFraction_30E_360 implements RosettaFunction {
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFraction_30E_360Default extends YearFraction_30E_360 {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperMaths.<Integer, Integer, Integer>add(MapperMaths.<Integer, Integer, Integer>add(MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(Integer.valueOf(360)), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()))), MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(Integer.valueOf(30)), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endMonth(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startMonth(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get())))), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endDay(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startDay(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()))), MapperS.of(Integer.valueOf(360))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> startYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(startDate).<Integer>map("Year", Date::getYear);
			}
			
			@Override
			protected Mapper<Integer> endYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(endDate).<Integer>map("Year", Date::getYear);
			}
			
			@Override
			protected Mapper<Integer> startMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(startDate).<Integer>map("Month", Date::getMonth);
			}
			
			@Override
			protected Mapper<Integer> endMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(endDate).<Integer>map("Month", Date::getMonth);
			}
			
			@Override
			protected Mapper<Integer> endDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(new Min().execute(MapperS.of(endDate).<Integer>map("Day", Date::getDay).get(), MapperS.of(Integer.valueOf(30)).get()));
			}
			
			@Override
			protected Mapper<Integer> startDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(new Min().execute(MapperS.of(startDate).<Integer>map("Day", Date::getDay).get(), MapperS.of(Integer.valueOf(30)).get()));
			}
		}
	}
	@ImplementedBy(YearFraction_30E_360_ISDA.YearFraction_30E_360_ISDADefault.class)
	public static abstract class YearFraction_30E_360_ISDA implements RosettaFunction {
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Boolean> startDateIsInLeapYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Boolean> endDateIsInLeapYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> startDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> endDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFraction_30E_360_ISDADefault extends YearFraction_30E_360_ISDA {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperMaths.<Integer, Integer, Integer>add(MapperMaths.<Integer, Integer, Integer>add(MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(Integer.valueOf(360)), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()))), MapperMaths.<Integer, Integer, Integer>multiply(MapperS.of(Integer.valueOf(30)), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endMonth(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startMonth(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get())))), MapperMaths.<Integer, Integer, Integer>subtract(MapperS.of(endDay(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(startDay(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()))), MapperS.of(Integer.valueOf(360))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Boolean> startDateIsInLeapYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(new IsLeapYear().execute(MapperS.of(startDate).<Integer>map("Year", Date::getYear).get()));
			}
			
			@Override
			protected Mapper<Boolean> endDateIsInLeapYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(new IsLeapYear().execute(MapperS.of(endDate).<Integer>map("Year", Date::getYear).get()));
			}
			
			@Override
			protected Mapper<Integer> startYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(startDate).<Integer>map("Year", Date::getYear);
			}
			
			@Override
			protected Mapper<Integer> endYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(endDate).<Integer>map("Year", Date::getYear);
			}
			
			@Override
			protected Mapper<Integer> startMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(startDate).<Integer>map("Month", Date::getMonth);
			}
			
			@Override
			protected Mapper<Integer> endMonth(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(endDate).<Integer>map("Month", Date::getMonth);
			}
			
			@Override
			protected Mapper<Integer> startDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperUtils.runSingle(() -> {
					if (areEqual(MapperS.of(startDateIsInLeapYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Boolean.valueOf(false)), CardinalityOperator.All).and(areEqual(MapperS.of(startDate).<Integer>map("Month", Date::getMonth), MapperS.of(Integer.valueOf(2)), CardinalityOperator.All)).and(areEqual(MapperS.of(startDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(28)), CardinalityOperator.All)).or(areEqual(MapperS.of(startDateIsInLeapYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).and(areEqual(MapperS.of(startDate).<Integer>map("Month", Date::getMonth), MapperS.of(Integer.valueOf(2)), CardinalityOperator.All)).and(areEqual(MapperS.of(startDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(29)), CardinalityOperator.All))).or(areEqual(MapperS.of(startDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(31)), CardinalityOperator.All)).getOrDefault(false)) {
						return MapperS.of(Integer.valueOf(30));
					}
					else {
						return MapperS.of(endDate).<Integer>map("Day", Date::getDay);
					}
				});
			}
			
			@Override
			protected Mapper<Integer> endDay(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperUtils.runSingle(() -> {
					if (areEqual(MapperS.of(endDateIsInLeapYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Boolean.valueOf(false)), CardinalityOperator.All).and(areEqual(MapperS.of(endDate).<Integer>map("Month", Date::getMonth), MapperS.of(Integer.valueOf(2)), CardinalityOperator.All)).and(areEqual(MapperS.of(endDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(28)), CardinalityOperator.All)).or(areEqual(MapperS.of(endDateIsInLeapYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).and(areEqual(MapperS.of(endDate).<Integer>map("Month", Date::getMonth), MapperS.of(Integer.valueOf(2)), CardinalityOperator.All)).and(areEqual(MapperS.of(endDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(29)), CardinalityOperator.All))).or(areEqual(MapperS.of(endDate).<Integer>map("Day", Date::getDay), MapperS.of(Integer.valueOf(31)), CardinalityOperator.All)).or(areEqual(MapperS.of(endDate), MapperS.of(terminationDate), CardinalityOperator.All)).getOrDefault(false)) {
						return MapperS.of(Integer.valueOf(30));
					}
					else {
						return MapperS.of(endDate).<Integer>map("Day", Date::getDay);
					}
				});
			}
		}
	}
	@ImplementedBy(YearFractionACT_364.YearFractionACT_364Default.class)
	public static abstract class YearFractionACT_364 implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionACT_364Default extends YearFractionACT_364 {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Integer.valueOf(364))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
		}
	}
	@ImplementedBy(YearFractionACT_365L.YearFractionACT_365LDefault.class)
	public static abstract class YearFractionACT_365L implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Boolean> endDateIsInLeapYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionACT_365LDefault extends YearFractionACT_365L {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(daysInYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get())).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Boolean> endDateIsInLeapYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(new IsLeapYear().execute(MapperS.of(endDate).<Integer>map("Year", Date::getYear).get()));
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
			
			@Override
			protected Mapper<Integer> daysInYear(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperUtils.runSingle(() -> {
					if (areEqual(MapperS.of(endDateIsInLeapYear(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).getOrDefault(false)) {
						return MapperS.of(Integer.valueOf(366));
					}
					else {
						return MapperS.of(Integer.valueOf(365));
					}
				});
			}
		}
	}
	@ImplementedBy(YearFractionCAL_252.YearFractionCAL_252Default.class)
	public static abstract class YearFractionCAL_252 implements RosettaFunction {
		
		// RosettaFunction dependencies
		//
		@Inject protected DateDifference dateDifference;
	
		/**
		* @param dayCountFractionEnum The day count fraction to use
		* @param startDate The start date of the range for which the year fraction is required
		* @param endDate The end date of the range for which the year fraction is required
		* @param terminationDate The termination date of the payout; this is needed for some day count fractions
		* @param periodsInYear The number of periods in a year in the payout; this is needed for some day count fractions
		* @return result The fraction of a year represented by period from the startDate to the endDate
		*/
		public BigDecimal evaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
			BigDecimal result = doEvaluate(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			
			return result;
		}
	
		protected abstract BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		protected abstract Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear);
	
		public static class YearFractionCAL_252Default extends YearFractionCAL_252 {
			@Override
			protected BigDecimal doEvaluate(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				BigDecimal result = null;
				return assignOutput(result, dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear);
			}
			
			protected BigDecimal assignOutput(BigDecimal result, DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				result = MapperMaths.<BigDecimal, Integer, Integer>divide(MapperS.of(daysInPeriod(dayCountFractionEnum, startDate, endDate, terminationDate, periodsInYear).get()), MapperS.of(Integer.valueOf(252))).get();
				
				return result;
			}
			
			@Override
			protected Mapper<Integer> daysInPeriod(DayCountFractionEnum dayCountFractionEnum, Date startDate, Date endDate, Date terminationDate, Integer periodsInYear) {
				return MapperS.of(dateDifference.evaluate(MapperS.of(startDate).get(), MapperS.of(endDate).get()));
			}
		}
	}
}
