/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.file;

import java.io.File;
import java.io.InputStream;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.camel.Exchange;
import org.apache.camel.Expression;
import org.apache.camel.Message;
import org.apache.camel.component.file.GenericFileEndpoint;
import org.apache.camel.component.file.GenericFileExist;
import org.apache.camel.component.file.GenericFileOperationFailedException;
import org.apache.camel.component.file.GenericFileOperations;
import org.apache.camel.support.DefaultExchange;
import org.apache.camel.support.DefaultProducer;
import org.apache.camel.support.LRUCacheFactory;
import org.apache.camel.support.MessageHelper;
import org.apache.camel.support.service.ServiceHelper;
import org.apache.camel.util.FileUtil;
import org.apache.camel.util.ObjectHelper;
import org.apache.camel.util.StringHelper;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GenericFileProducer<T>
extends DefaultProducer {
    private static final Logger LOG = LoggerFactory.getLogger(GenericFileProducer.class);
    protected final GenericFileEndpoint<T> endpoint;
    protected GenericFileOperations<T> operations;
    private final Map<String, Lock> locks = Collections.synchronizedMap(LRUCacheFactory.newLRUCache((int)100));

    protected GenericFileProducer(GenericFileEndpoint<T> endpoint, GenericFileOperations<T> operations) {
        super(endpoint);
        this.endpoint = endpoint;
        this.operations = operations;
    }

    public String getFileSeparator() {
        return File.separator;
    }

    public String normalizePath(String name) {
        return FileUtil.normalizePath((String)name);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void process(Exchange exchange) throws Exception {
        String existing = (String)exchange.getIn().getHeader("CamelFileName", String.class);
        String target = this.createFileName(exchange);
        Lock lock = this.locks.computeIfAbsent(target, f -> new ReentrantLock());
        lock.lock();
        try {
            this.processExchange(exchange, target);
        }
        finally {
            lock.unlock();
            exchange.getIn().removeHeader("CamelOverruleFileName");
            exchange.getIn().setHeader("CamelFileName", (Object)existing);
        }
    }

    public void setOperations(GenericFileOperations<T> operations) {
        this.operations = operations;
    }

    protected void processExchange(Exchange exchange, String target) throws Exception {
        LOG.trace("Processing file: {} for exchange: {}", (Object)target, (Object)exchange);
        try {
            boolean targetExists;
            this.preWriteCheck(exchange);
            boolean writeAsTempAndRename = ObjectHelper.isNotEmpty((Object)this.endpoint.getTempFileName());
            String tempTarget = null;
            if (writeAsTempAndRename) {
                tempTarget = this.createTempFileName(exchange, target);
                LOG.trace("Writing using tempNameFile: {}", (Object)tempTarget);
                if (this.endpoint.getFileExist() != GenericFileExist.TryRename && this.endpoint.isEagerDeleteTargetFile() && (targetExists = this.operations.existsFile(target)) && this.handleExistingTargetEager(target)) {
                    return;
                }
                if (this.endpoint.getFileExist() != GenericFileExist.TryRename && this.operations.existsFile(tempTarget)) {
                    LOG.trace("Deleting existing temp file: {}", (Object)tempTarget);
                    this.tryOverridingFile(tempTarget);
                }
            }
            this.writeFile(exchange, tempTarget != null ? tempTarget : target);
            if (tempTarget != null) {
                if (this.endpoint.getFileExist() != GenericFileExist.TryRename && !this.endpoint.isEagerDeleteTargetFile() && (targetExists = this.operations.existsFile(target)) && this.handleExistingTarget(target)) {
                    return;
                }
                LOG.trace("Renaming file: [{}] to: [{}]", (Object)tempTarget, (Object)target);
                boolean renamed = this.operations.renameFile(tempTarget, target);
                if (!renamed) {
                    throw new GenericFileOperationFailedException("Cannot rename file from: " + tempTarget + " to: " + target);
                }
            }
            if (this.endpoint.getChecksumFileAlgorithm() != null) {
                this.writeChecksumFile(exchange, target);
            }
            if (this.endpoint.getDoneFileName() != null) {
                this.writeDoneFile(exchange, target);
            }
            exchange.getIn().setHeader("CamelFileNameProduced", (Object)target);
        }
        catch (Exception e) {
            this.handleFailedWrite(exchange, e);
        }
        this.postWriteCheck(exchange);
    }

    private static void throwFileAlreadyExistException(String target) {
        throw new GenericFileOperationFailedException("File already exist: " + target + ". Cannot write new file.");
    }

    private static boolean doIgnore(String target) {
        LOG.trace("An existing file already exists: {}. Ignore and do not override it.", (Object)target);
        return true;
    }

    private void tryOverridingFile(String target) {
        if (!this.operations.deleteFile(target)) {
            throw new GenericFileOperationFailedException("Cannot delete file: " + target);
        }
    }

    private boolean handleExistingTargetEager(String target) {
        LOG.trace("EagerDeleteTargetFile, target exists");
        if (this.endpoint.getFileExist() == GenericFileExist.Ignore) {
            return GenericFileProducer.doIgnore(target);
        }
        if (this.endpoint.getFileExist() == GenericFileExist.Fail) {
            GenericFileProducer.throwFileAlreadyExistException(target);
        } else if (this.endpoint.getFileExist() == GenericFileExist.Move) {
            this.endpoint.getMoveExistingFileStrategy().moveExistingFile(this.endpoint, this.operations, target);
        } else if (this.endpoint.isEagerDeleteTargetFile() && this.endpoint.getFileExist() == GenericFileExist.Override) {
            LOG.trace("Eagerly deleting existing file: {}", (Object)target);
            this.tryOverridingFile(target);
        }
        return false;
    }

    private boolean handleExistingTarget(String target) {
        LOG.trace("Not using EagerDeleteTargetFile, target exists");
        if (this.endpoint.getFileExist() == GenericFileExist.Ignore) {
            return GenericFileProducer.doIgnore(target);
        }
        if (this.endpoint.getFileExist() == GenericFileExist.Fail) {
            GenericFileProducer.throwFileAlreadyExistException(target);
        } else if (this.endpoint.getFileExist() == GenericFileExist.Override) {
            LOG.trace("Deleting existing file: {}", (Object)target);
            this.tryOverridingFile(target);
        }
        return false;
    }

    private void writeDoneFile(Exchange exchange, String target) {
        String doneFileName = this.endpoint.createDoneFileName(target);
        StringHelper.notEmpty((String)doneFileName, (String)"doneFileName", this.endpoint);
        DefaultExchange empty = new DefaultExchange(exchange);
        empty.getIn().setBody((Object)"");
        LOG.trace("Writing done file: [{}]", (Object)doneFileName);
        if (this.operations.existsFile(doneFileName) && !this.operations.deleteFile(doneFileName)) {
            throw new GenericFileOperationFailedException("Cannot delete existing done file: " + doneFileName);
        }
        this.writeFile((Exchange)empty, doneFileName);
    }

    protected void writeChecksumFile(Exchange exchange, String target) throws Exception {
        String algorithm = this.endpoint.getChecksumFileAlgorithm();
        String checksumFileName = target + "." + algorithm;
        MessageHelper.resetStreamCache((Message)exchange.getIn());
        InputStream is = (InputStream)exchange.getIn().getMandatoryBody(InputStream.class);
        DefaultExchange checksumExchange = new DefaultExchange(exchange);
        checksumExchange.getIn().setBody((Object)new DigestUtils(algorithm).digestAsHex(is));
        LOG.trace("Writing checksum file: [{}]", (Object)checksumFileName);
        if (this.operations.existsFile(checksumFileName) && !this.operations.deleteFile(checksumFileName)) {
            throw new GenericFileOperationFailedException("Cannot delete existing checksum file: " + checksumFileName);
        }
        this.writeFile((Exchange)checksumExchange, checksumFileName);
    }

    public void handleFailedWrite(Exchange exchange, Exception exception) throws Exception {
        throw exception;
    }

    public void preWriteCheck(Exchange exchange) throws Exception {
    }

    public void postWriteCheck(Exchange exchange) {
    }

    public void writeFile(Exchange exchange, String fileName) throws GenericFileOperationFailedException {
        boolean success;
        if (this.endpoint.isAutoCreate()) {
            String name = FileUtil.normalizePath((String)fileName);
            File file = new File(name);
            String directory = file.getParent();
            boolean absolute = FileUtil.isAbsolute((File)file);
            if (directory != null && !this.operations.buildDirectory(directory, absolute)) {
                LOG.debug("Cannot build directory [{}] (could be because of denied permissions)", (Object)directory);
            }
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("About to write [{}] to [{}] from exchange [{}]", new Object[]{fileName, this.getEndpoint(), exchange});
        }
        if (!(success = this.operations.storeFile(fileName, exchange, -1L))) {
            throw new GenericFileOperationFailedException("Error writing file [" + fileName + "]");
        }
        LOG.debug("Wrote [{}] to [{}]", (Object)fileName, (Object)this.getEndpoint());
    }

    public String createFileName(Exchange exchange) {
        Object overrule = exchange.getIn().getHeader("CamelOverruleFileName");
        Object value = GenericFileProducer.getOverrule(exchange, overrule);
        if (overrule != null) {
            exchange.getIn().setHeader("CamelFileName", value);
        }
        if (value instanceof String && StringHelper.hasStartToken((String)((String)value), (String)"simple")) {
            LOG.warn("Simple expression: {} detected in header: {} of type String. This feature has been removed (see CAMEL-6748).", value, (Object)"CamelFileName");
        }
        Expression expression = this.endpoint.getFileName();
        if (value instanceof Expression) {
            expression = (Expression)value;
        }
        String name = this.evaluateName(exchange, expression, value);
        String endpointPath = this.endpoint.getConfiguration().getDirectory();
        Object baseDir = "";
        if (!endpointPath.isEmpty()) {
            boolean trailingSlash;
            baseDir = endpointPath;
            boolean bl = trailingSlash = endpointPath.endsWith("/") || endpointPath.endsWith("\\");
            if (!trailingSlash) {
                baseDir = (String)baseDir + this.getFileSeparator();
            }
        }
        Object answer = name != null ? (String)baseDir + name : (String)baseDir + this.endpoint.getGeneratedFileName(exchange.getIn());
        if (this.endpoint.isJailStartingDirectory()) {
            GenericFileProducer.jailedCheck((String)answer, (String)baseDir);
        }
        if (this.endpoint.getConfiguration().needToNormalize()) {
            answer = this.normalizePath((String)answer);
        }
        return answer;
    }

    private static Object getOverrule(Exchange exchange, Object overrule) {
        Object value = overrule != null ? (overrule instanceof Expression ? overrule : exchange.getContext().getTypeConverter().convertTo(String.class, exchange, overrule)) : exchange.getIn().getHeader("CamelFileName");
        return value;
    }

    private static void jailedCheck(String answer, String baseDir) {
        String compatchBaseDir;
        String compatchAnswer = FileUtil.compactPath((String)answer);
        if (!compatchAnswer.startsWith(compatchBaseDir = FileUtil.compactPath((String)baseDir))) {
            throw new IllegalArgumentException("Cannot write file with name: " + compatchAnswer + " as the filename is jailed to the starting directory: " + compatchBaseDir);
        }
    }

    private String evaluateName(Exchange exchange, Expression expression, Object value) {
        String name;
        if (expression != null) {
            LOG.trace("Filename evaluated as expression: {}", (Object)expression);
            name = (String)expression.evaluate(exchange, String.class);
        } else {
            name = (String)exchange.getContext().getTypeConverter().convertTo(String.class, exchange, value);
        }
        if (name != null && this.endpoint.isFlatten()) {
            name = GenericFileProducer.flattenName(name);
        }
        return name;
    }

    private static String flattenName(String name) {
        int pos = Math.max(name.lastIndexOf(47), name.lastIndexOf("\\"));
        if (pos != -1) {
            name = name.substring(pos + 1);
        }
        return name;
    }

    public String createTempFileName(Exchange exchange, String fileName) {
        String tempName;
        String answer = fileName;
        if (exchange.getIn().getHeader("CamelFileName") == null) {
            exchange.getIn().setHeader("CamelFileName", (Object)FileUtil.stripPath((String)fileName));
            tempName = (String)this.endpoint.getTempFileName().evaluate(exchange, String.class);
            exchange.getIn().removeHeader("CamelFileName");
        } else {
            tempName = (String)this.endpoint.getTempFileName().evaluate(exchange, String.class);
        }
        int pos = Math.max(answer.lastIndexOf(47), answer.lastIndexOf("\\"));
        if (pos == -1) {
            answer = tempName;
        } else {
            StringBuilder sb = new StringBuilder(answer.substring(0, pos + 1));
            sb.append(tempName);
            answer = sb.toString();
        }
        if (this.endpoint.getConfiguration().needToNormalize()) {
            answer = this.normalizePath(answer);
        }
        answer = FileUtil.compactPath((String)answer, (String)this.getFileSeparator());
        return answer;
    }

    protected void doStart() throws Exception {
        ServiceHelper.startService(this.locks);
        super.doStart();
    }

    protected void doStop() throws Exception {
        super.doStop();
        ServiceHelper.stopService(this.locks);
    }
}

