/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib;

import javax.servlet.jsp.JspException;
import java.lang.ref.SoftReference;
import java.text.DateFormat;
import java.text.Format;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Locale;
import java.util.WeakHashMap;
import java.util.TimeZone;
import java.io.IOException;

/**
 * Access a i18n-ized message. The message must be in a resource bundle with the same name as the action that it is
 * associated with. In practice this means that you should create a properties file in the same package as your Java
 * class with the same name as your class, but with .properties extension.
 * <p/>
 * See examples for further info on how to use.
 * <p/>
 * If the named message is not found, then the body of the tag will be used as default message. If no body is used, then
 * the name of the message will be used.
 *
 * @author Rickard &#214;berg (rickard@dreambean.com)
 * @version $Revision: 1.12 $
 */
public class TextTag
        extends WebWorkBodyTagSupport
        implements ParamTag.UnnamedParametric
{
    // Attributes ----------------------------------------------------
    protected String nameAttr;
    protected String value0Attr;
    protected String value1Attr;
    protected String value2Attr;
    protected String value3Attr;

    private ArrayList values;
    private WeakHashMap fmtCache = new WeakHashMap();

    // Public --------------------------------------------------------
    public void setName(String aName)
    {
        nameAttr = aName;
    }

    public void setValue0(String aName)
    {
        value0Attr = aName;
    }

    public void setValue1(String aName)
    {
        value1Attr = aName;
    }

    public void setValue2(String aName)
    {
        value2Attr = aName;
    }

    public void setValue3(String aName)
    {
        value3Attr = aName;
    }

    public void addParameter(String aName, Object aValue)
    {
        addParameter(aValue);
    }

    public void addParameter(Object aValue)
    {
        if (aValue == null)
        {
            return;
        }
        if (values == null)
        {
            values = new ArrayList();
        }
        values.add(aValue);
    }

    // BodyTag implementation ----------------------------------------
    public int doEndTag() throws JspException
    {
        // Get message
        String msg = getMessage();

        // Add tag attribute values
        // These can be used to parameterize the i18n-ized message
        if (value0Attr != null)
        {
            addParameter(findValue(value0Attr));
        }
        if (value1Attr != null)
        {
            addParameter(findValue(value1Attr));
        }
        if (value2Attr != null)
        {
            addParameter(findValue(value2Attr));
        }
        if (value3Attr != null)
        {
            addParameter(findValue(value3Attr));
        }

        try
        {
            if (values != null)
            {
                // Try to do something interesting
                // Use MessageFormat with parameters
                // Check cache first for previously used MessageFormats
                MessageFormat fmt = null;
                SoftReference ref = (SoftReference) fmtCache.get(msg);
                if (ref != null)
                {
                    fmt = (MessageFormat) ref.get();
                }
                if (fmt == null)
                {
                    //call ActionSupport's getLocale() first in
                    //case there are overrides
                    Locale wwLocale = (Locale) findValue("locale");
                    if (wwLocale == null)
                    {
                        wwLocale = pageContext.getRequest().getLocale();
                    }

                    if (wwLocale.equals(Locale.getDefault()))
                    {
                        fmt = new MessageFormat(msg);
                    }
                    else
                    {
                        // Use chosen locale
                        fmt = new MessageFormat("");
                        fmt.setLocale(wwLocale);
                        fmt.applyPattern(msg);
                    }

                    fmtCache.put(msg, new SoftReference(fmt));
                }

                // This isn't fool-proof, but it's better than nothing
                // The idea is to try and convert strings into the
                // types of objects that the formatters expects
                // i.e. Numbers and Dates
                boolean usesDates = false;
                Format[] formats = fmt.getFormats();
                for (int i = 0; i < formats.length; i++)
                {
                    Format format = formats[i];
                    if (format != null)
                    {
                        if (format instanceof DateFormat)
                        {
                            usesDates = true;
                            if (values.size() > i)
                            {
                                Object value = values.get(i);
                                if (value instanceof String)
                                {
                                    DateFormat dateFmt = (DateFormat) format;
                                    value = dateFmt.parse((String) value);
                                    values.set(i, value);
                                }
                            }
                        }
                        else if (format instanceof NumberFormat)
                        {
                            if (values.size() > i)
                            {
                                Object value = values.get(i);
                                if (value instanceof String)
                                {
                                    NumberFormat numberFmt = (NumberFormat) format;
                                    value = numberFmt.parse((String) value);
                                    values.set(i, value);
                                }
                            }
                        }
                    }
                }

                // set user timezone on the date formatters if necessary
                if (usesDates)
                {
                    TimeZone wwTimezone = (TimeZone) findValue("timezone");
                    if (wwTimezone != null)
                    {
                        for (int i = 0; i < formats.length; i++)
                        {
                            Format format = formats[i];
                            if (format != null && format instanceof DateFormat)
                            {
                                ((DateFormat) format).setTimeZone(wwTimezone);
                            }
                        }
                    }
                }

                String result = fmt.format(values.toArray());
                write(result);

                values = null;
            }
            else
            {
                // Print out string
                write(msg);
            }
        }
        catch (Exception e)
        {
            throw new JspException(e.getMessage(), e);
        }

        return EVAL_PAGE;
    }

    /**
     * Write the result to the screen.
     * <p/>
     * Subclasses may wish to override this method if they wish to decorate the text that is written (eg for
     * debugging).
     */
    protected void write(String result) throws IOException
    {
        pageContext.getOut().write(result);
    }

    /**
     * Look up the value for this textTag.  Currently this looks up the value on the valuestack
     * <p/>
     * Subclasses may wish to override this method if they want to look up the text value manually.
     */
    protected String getMessage()
    {
        String msg;
        try
        {
            msg = findString("text(" + nameAttr + ')');
            if (msg == null)
            {
                if (bodyContent != null && bodyContent.getString().trim().length() > 0)
                {
                    // Show tag body
                    msg = bodyContent.getString().trim();
                }
                else
                {
                    // Show message name
                    msg = findString(nameAttr);
                }
            }
        }
        catch (Exception e)
        {
            if (bodyContent != null && bodyContent.getString().trim().length() > 0)
            {
                // Show tag body
                msg = bodyContent.getString().trim();
            }
            else
            {
                // Show message name
                msg = findString(nameAttr);
            }
        }
        return msg;
    }
}


