/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.util.Encoding;
import webwork.util.FastByteArrayOutputStream;
import webwork.util.URLCodec;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.StringTokenizer;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.PageContext;

/**
 * Include a servlets output (For example, result of servlet, or a JSP page).
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @author <a href="mailto:scott@atlassian.com">Scott Farquhar</a>
 * @version $Revision: 1.24 $
 */
public class IncludeTag
        extends WebWorkBodyTagSupport
        implements ParamTag.Parametric
{
    private static final Log log = LogFactory.getLog(IncludeTag.class);

    // Static --------------------------------------------------------
    public static void include(String aResult, PageContext aContext)
            throws ServletException, IOException
    {
        String resourcePath = getContextRelativePath(aContext.getRequest(), aResult);
        RequestDispatcher rd = aContext.getRequest().getRequestDispatcher(resourcePath);

        if (rd == null)
        {
            String msg = "Not a valid resource path:" + resourcePath + ", make sure this file exists!";
            log.error(msg);
            throw new ServletException(msg);
        }

        PageResponse pageResponse = new PageResponse((HttpServletResponse) aContext.getResponse());
        // Include the resource
        rd.include(aContext.getRequest(), pageResponse);

        //write the response back to the JspWriter, using the correct encoding.
        String encoding = Encoding.getEncoding();
        if (encoding != null)
        {
            //use the encoding specified in the property file
            pageResponse.getContent().writeTo(aContext.getOut(), encoding);
        }
        else
        {
            //use the platform specific encoding
            pageResponse.getContent().writeTo(aContext.getOut(), null);
        }
    }

    public static String getContextRelativePath(ServletRequest request,
            String relativePath)
    {
        String returnValue;

        if (relativePath.startsWith("/"))
        {
            returnValue = relativePath;
        }
        else if (!(request instanceof HttpServletRequest))
        {
            returnValue = relativePath;
        }
        else
        {
            HttpServletRequest hrequest = (HttpServletRequest) request;
            String uri = (String)
                    request.getAttribute("javax.servlet.include.servlet_path");
            if (uri == null)
            {
                uri = hrequest.getServletPath();
            }
            returnValue = uri.substring(0, uri.lastIndexOf('/')) + '/' + relativePath;
        }

        // .. is illegal in an absolute path according to the Servlet Spec and will cause
        // known problems on Orion application servers.
        if (returnValue.indexOf("..") != -1)
        {
            Stack stack = new Stack();
            StringTokenizer pathParts = new StringTokenizer(returnValue.replace('\\', '/'), "/");
            while (pathParts.hasMoreTokens())
            {
                String part = pathParts.nextToken();
                if (!".".equals(part))
                {
                    if ("..".equals(part))
                    {
                        stack.pop();
                    }
                    else
                    {
                        stack.push(part);
                    }
                }
            }

            StringBuilder flatPathBuffer = new StringBuilder();

            for (int i = 0; i < stack.size(); i++)
            {
                flatPathBuffer.append('/').append(stack.elementAt(i));
            }

            returnValue = flatPathBuffer.toString();
        }

        return returnValue;
    }

    // Attributes ----------------------------------------------------
    protected String pageAttr;
    protected String valueAttr;
    protected Map params;

    // Public --------------------------------------------------------

    /**
     * Name of page/servlet to include.
     *
     * @param aPage
     *
     * @deprecated use value attribute instead
     */
    public void setPage(String aPage)
    {
        pageAttr = aPage;
    }

    /**
     * Name of property whose value is the name of the page/servlet to include.
     *
     * @param aName
     */
    public void setValue(String aName)
    {
        valueAttr = aName;
    }

    // ParamTag.Parametric implementation ----------------------------
    /**
     * Add a parameter to the URL of the included page/servlet.
     *
     * @param name
     * @param value
     */
    public void addParameter(String name, Object value)
    {
        if (value != null)
        {
            List currentValues = (List) params.get(name);
            if (currentValues == null)
            {
                currentValues = new ArrayList();
                params.put(name, currentValues);
            }
            currentValues.add(value);
        }
    }

    // BodyTag implementation ----------------------------------------
    public int doStartTag() throws JspException
    {
        // Init parameter map
        params = new HashMap();

        return super.doStartTag();
    }

    public int doEndTag() throws JspException
    {
        String page;
        // If value is set, we resolve it to get the page name
        if (valueAttr != null)
        {
            page = (String) findValue(valueAttr);
        }
        else
        {
            page = pageAttr;
        }

        StringBuilder urlBuf = new StringBuilder();

        // Add URL
        urlBuf.append(page);

        // Add request parameters
        if (params.size() > 0)
        {

            urlBuf.append('?');
            String concat = "";

            // Set parameters
            Iterator iter = params.entrySet().iterator();
            while (iter.hasNext())
            {
                Map.Entry entry = (Map.Entry) iter.next();
                Object name = entry.getKey();
                List values = (List) entry.getValue();
                for (int i = 0; i < values.size(); i++)
                {
                    urlBuf.append(concat);
                    urlBuf.append(name);
                    urlBuf.append('=');
                    urlBuf.append(URLCodec.encode(values.get(i).toString()));
                    concat = "&";
                }
            }
        }
        params = null;

        String result = urlBuf.toString();

        // Include
        try
        {
            include(result, pageContext);
        }
        catch (Exception e)
        {
            throw new JspException("Exception thrown during include of " + result, e);
        }

        super.doEndTag();
        return EVAL_PAGE;
    }


}


/**
 * Implementation of ServletOutputStream that stores all data written to it in a temporary buffer accessible from {@link
 * #getBuffer()} .
 *
 * @author <a href="joe@truemesh.com">Joe Walnes</a>
 * @author <a href="mailto:scott@atlassian.com">Scott Farquhar</a>
 */
final class PageOutputStream extends ServletOutputStream
{
    private FastByteArrayOutputStream buffer;

    public PageOutputStream()
    {
        buffer = new FastByteArrayOutputStream();
    }

    public void write(byte[] b, int o, int l) throws IOException
    {
        buffer.write(b, o, l);
    }

    public void write(int i) throws IOException
    {
        buffer.write(i);
    }

    public void flush() throws IOException
    {
        buffer.flush();
    }

    public void close() throws IOException
    {
        buffer.close();
    }

    public void write(byte b[]) throws IOException
    {
        buffer.write(b);
    }

    /**
     * Return all data that has been written to this OutputStream.
     */
    public FastByteArrayOutputStream getBuffer() throws IOException
    {
        flush();
        return buffer;
    }

}


/**
 * Simple wrapper to HTTPServletResponse that will allow getWriter() and getResponse() to be called as many times as
 * needed without causing conflicts.
 * <p/>
 * The underlying outputStream is a wrapper around {@link webwork.view.taglib.PageOutputStream} which will store the
 * written content to a buffer.
 * <p/>
 * This buffer can later be retrieved by calling {@link #getContent}.
 *
 * @author <a href="mailto:joe@truemesh.com">Joe Walnes</a>
 * @author <a href="mailto:scott@atlassian.com">Scott Farquhar</a>
 */
final class PageResponse extends HttpServletResponseWrapper
{
    protected PrintWriter pagePrintWriter;
    protected ServletOutputStream outputStream;
    private PageOutputStream pageOutputStream = null;

    /**
     * Create PageResponse wrapped around an existing HttpServletResponse.
     */
    public PageResponse(HttpServletResponse response)
    {
        super(response);
    }

    /**
     * Return PrintWriter wrapper around PageOutputStream.
     */
    public PrintWriter getWriter() throws IOException
    {
        if (pagePrintWriter == null)
        {
            pagePrintWriter = new PrintWriter(new OutputStreamWriter(getOutputStream(), getCharacterEncoding()));
        }
        return pagePrintWriter;
    }

    /**
     * Return instance of {@link webwork.view.taglib.PageOutputStream} allowing all data written to stream to be stored
     * in temporary buffer.
     */
    public ServletOutputStream getOutputStream() throws IOException
    {
        if (pageOutputStream == null)
        {
            pageOutputStream = new PageOutputStream();
        }
        return pageOutputStream;
    }

    /**
     * Return the content buffered inside the {@link webwork.view.taglib.PageOutputStream}.
     *
     * @return
     *
     * @throws IOException
     */
    public FastByteArrayOutputStream getContent() throws IOException
    {
        //if we are using a writer, we need to flush the
        //data to the underlying outputstream.
        //most containers do this - but it seems Jetty 4.0.5 doesn't
        if (pagePrintWriter != null)
        {
            pagePrintWriter.flush();
        }

        return ((PageOutputStream) getOutputStream()).getBuffer();
    }
}