/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.action.IllegalArgumentAware;
import webwork.action.ValidationEditorSupport;
import webwork.util.editor.BigDecimalEditor;
import webwork.util.editor.BooleanEditor;
import webwork.util.editor.BooleanObjectEditor;
import webwork.util.editor.ByteEditor;
import webwork.util.editor.ByteObjectEditor;
import webwork.util.editor.DateEditor;
import webwork.util.editor.DateFormatEditor;
import webwork.util.editor.DoubleEditor;
import webwork.util.editor.DoubleObjectEditor;
import webwork.util.editor.FastPropertyEditor;
import webwork.util.editor.FloatEditor;
import webwork.util.editor.FloatObjectEditor;
import webwork.util.editor.IntegerEditor;
import webwork.util.editor.IntegerObjectEditor;
import webwork.util.editor.LongEditor;
import webwork.util.editor.LongObjectEditor;
import webwork.util.editor.PropertyEditorException;
import webwork.util.editor.PropertyMessage;
import webwork.util.editor.ShortEditor;
import webwork.util.editor.ShortObjectEditor;
import webwork.util.editor.StringEditor;
import webwork.util.editor.TimestampEditor;
import webwork.util.injection.ObjectFactory;

import java.beans.BeanInfo;
import java.beans.IndexedPropertyDescriptor;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.WeakHashMap;

/**
 * JavaBean utility methods
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @author Matt Baldree (matt@smallleap.com)
 * @version $Revision: 1.43 $
 */
public final class BeanUtil
{
    // Static  -------------------------------------------------------
    // Class -> PropertyDescriptor[] map
    static Map propertyDescriptors = new WeakHashMap();

    // The map is cloned when it is updated so we do not have to use Hashtable
    static Map propertyEditors = new HashMap();

    static Map objectMap = new WeakHashMap();
    static final Log log = LogFactory.getLog(BeanUtil.class);

    static
    {
        PropertyEditorManager.registerEditor(Integer.class, IntegerObjectEditor.class);
        PropertyEditorManager.registerEditor(Integer[].class, IntegerObjectEditor.class);
        PropertyEditorManager.registerEditor(int.class, IntegerEditor.class);
        PropertyEditorManager.registerEditor(int[].class, IntegerEditor.class);

        PropertyEditorManager.registerEditor(Double.class, DoubleObjectEditor.class);
        PropertyEditorManager.registerEditor(Double[].class, DoubleObjectEditor.class);
        PropertyEditorManager.registerEditor(double.class, DoubleEditor.class);
        PropertyEditorManager.registerEditor(double[].class, DoubleEditor.class);

        PropertyEditorManager.registerEditor(Byte.class, ByteObjectEditor.class);
        PropertyEditorManager.registerEditor(Byte[].class, ByteObjectEditor.class);
        PropertyEditorManager.registerEditor(byte.class, ByteEditor.class);
        PropertyEditorManager.registerEditor(byte[].class, ByteEditor.class);

        PropertyEditorManager.registerEditor(Short.class, ShortObjectEditor.class);
        PropertyEditorManager.registerEditor(Short[].class, ShortObjectEditor.class);
        PropertyEditorManager.registerEditor(short.class, ShortEditor.class);
        PropertyEditorManager.registerEditor(short[].class, ShortEditor.class);

        PropertyEditorManager.registerEditor(Long.class, LongObjectEditor.class);
        PropertyEditorManager.registerEditor(Long[].class, LongObjectEditor.class);
        PropertyEditorManager.registerEditor(long.class, LongEditor.class);
        PropertyEditorManager.registerEditor(long[].class, LongEditor.class);

        PropertyEditorManager.registerEditor(Float.class, FloatObjectEditor.class);
        PropertyEditorManager.registerEditor(Float[].class, FloatObjectEditor.class);
        PropertyEditorManager.registerEditor(float.class, FloatEditor.class);
        PropertyEditorManager.registerEditor(float[].class, FloatEditor.class);

        PropertyEditorManager.registerEditor(Boolean.class, BooleanObjectEditor.class);
        PropertyEditorManager.registerEditor(Boolean[].class, BooleanObjectEditor.class);
        PropertyEditorManager.registerEditor(boolean.class, BooleanEditor.class);
        PropertyEditorManager.registerEditor(boolean[].class, BooleanEditor.class);

        PropertyEditorManager.registerEditor(String.class, StringEditor.class);
        PropertyEditorManager.registerEditor(String[].class, StringEditor.class);

        PropertyEditorManager.registerEditor(BigDecimal.class, BigDecimalEditor.class);
        PropertyEditorManager.registerEditor(BigDecimal[].class, BigDecimalEditor.class);

        PropertyEditorManager.registerEditor(Date.class, DateEditor.class);
        PropertyEditorManager.registerEditor(Date[].class, DateEditor.class);

        PropertyEditorManager.registerEditor(Timestamp.class, TimestampEditor.class);
        PropertyEditorManager.registerEditor(Timestamp[].class, TimestampEditor.class);

        PropertyEditorManager.registerEditor(DateFormat.class, DateFormatEditor.class);
    }

    /**
     * Copy JavaBean property values from one object to another.
     *
     * @param from the source object
     * @param to   the destination object
     *
     * @throws IllegalArgumentException thrown if the copy fails for some reason
     */
    public static void copy(Object from, Object to)
            throws IllegalArgumentException
    {
        copy(from, to, true);
    }

    /**
     * Copy JavaBean property values from one object to another.
     *
     * @param from        the source object
     * @param to          the destination object
     * @param includeNull whether null values are copied or not
     *
     * @throws IllegalArgumentException thrown if the copy fails for some reason
     */
    public static void copy(Object from, Object to, boolean includeNull)
            throws IllegalArgumentException
    {
        try
        {
            Object[] readParameters = new Object[0];
            Object[] writeParameters = new Object[1];
            PropertyDescriptor[] propertiesFrom = getPropertyDescriptors(from.getClass());
            // Get the field map of the destination object once
            Map fieldMap = getFieldMapForClass(to.getClass());
            for (int i = 0; i < propertiesFrom.length; i++)
            {
                PropertyDescriptor pdFrom = propertiesFrom[i];
                // Look up the destination descriptor in the fieldMap
                PropertyDescriptor pdTo = (PropertyDescriptor) fieldMap.get(pdFrom.getName());
                if (pdTo != null)
                {
                    Method readMethod = pdFrom.getReadMethod();
                    Method writeMethod = pdTo.getWriteMethod();
                    if (writeMethod != null && readMethod != null)
                    {
                        writeParameters[0] = InjectionUtils.invoke(pdFrom.getReadMethod(), from, readParameters);
                        if (!(!includeNull && writeParameters[0] == null))
                        {
                            InjectionUtils.invoke(pdTo.getWriteMethod(), to, writeParameters);
                        }
                    }
                }
            }
        }
        catch (Exception e)
        {
            log.warn("Bean copy failed:" + e, e);
            throw new IllegalArgumentException("Bean copy failed:" + e);
        }
    }


    /**
     * Set JavaBean property values on an object with values taken from a map.
     * <p/>
     * If the values in the map are of type String or String[] they may be converted to the actual property type by the
     * use of property editors.
     *
     * @param from map of values
     * @param to   the destination object
     *
     * @throws IllegalArgumentException
     */
    public static void setProperties(Map from, Object to) throws IllegalArgumentException
    {
        String key = null;
        Object value = null;

        try
        {
            Iterator keys = from.keySet().iterator();
            Map fieldMap = null;
            // Get the fieldMap of the destination object once
            if (keys.hasNext())
            {
                fieldMap = getFieldMapForClass(to.getClass());
            }
            while (keys.hasNext())
            {
                key = (String) keys.next();
                if (key != null && key.length() > 0)
                {
                    try
                    {
                        value = from.get(key);
                        setProperty(key, value, to, fieldMap);
                    }
                    catch (IllegalArgumentException e)
                    {
                        // Handle IEA's on actions specially
                        if (to instanceof IllegalArgumentAware)
                        {
                            ((IllegalArgumentAware) to).addIllegalArgumentException(key, e);
                        }
                        else
                        {
                            throw e;
                        }
                    }
                }
            }
        }
        catch (Throwable e)
        {
            log.warn("Could not set parameter \"" + key + "\" with value \"" + value + "\" on class " + to.getClass() + ':' + e, e);
            throw new IllegalArgumentException("Could not set parameter \"" + key + "\":" + e);
        }
    }


    /**
     * Set a single JavaBean property value on an object.
     * <p/>
     * If the values in the map are of type String or String[] they may be converted to the actual property type by the
     * use of property editors.
     *
     * @param propertyName String representing the property or subproperty
     * @param val          the value to set
     * @param obj          the object to set the property on.
     *
     * @throws IllegalArgumentException
     */
    static public void setProperty(String propertyName, Object val, Object obj) throws IllegalArgumentException
    {
        if (obj == null)
        {
            throw new IllegalArgumentException("The target object for property '" + propertyName + "'. The target object needs to be initialized to a non-null value in order to set this property.");
        }
        Map fieldMap = getFieldMapForClass(obj.getClass());
        setProperty(propertyName, val, obj, fieldMap);
    }

    /**
     * Set a single JavaBean property value on an object.
     * <p/>
     * If the values in the map are of type String or String[] they may be converted to the actual property type by the
     * use of property editors.
     *
     * @param propertyName String representing the property or subproperty
     * @param val          the value to set
     * @param obj          the object to set the property on.
     * @param fieldMap     Map with the PropertyDescriptors of the obj
     *
     * @throws IllegalArgumentException
     */
    private static void setProperty(String propertyName, Object val, Object obj, Map fieldMap)
            throws IllegalArgumentException
    {
        Query query = Query.getQuery(propertyName);
        QuerySegment[] segments = query.getSegments();

        Object curObject = obj;
        Map curFieldMap;

        int totalNumberOfSegments = findTheActualNumberOfSegments(segments);

        for (int i = 0; i < totalNumberOfSegments; ++i)
        {
            if (curObject == null)
            {
                throw new IllegalArgumentException("The target object for property '" + propertyName + "'. The target object needs to be initialized to a non-null value in order to set this property.");
            }

            QuerySegment curSegment = segments[i];
            PropertyDescriptor descriptor;
            // Is the current object the same as the original one?
            // If not, get the field map for the new object
            if (curObject != obj)
            {
                curFieldMap = getFieldMapForClass(curObject.getClass());
            }
            else
            {
                curFieldMap = fieldMap;
            }

            // Lookup the descriptor in the curFieldMap
            descriptor = (PropertyDescriptor) curFieldMap.get(curSegment.getId());

            if (descriptor == null && curSegment.getType() != QuerySegment.COLLECTION)
            {
                return;
            }
            else if (curSegment.getType() == QuerySegment.COLLECTION)
            {
                String key = curSegment.getId();

                // Map
                if (Map.class.isAssignableFrom(curObject.getClass()))
                {
                    curObject = ((Map) curObject).get(key);
                }
                // Resource Bundle
                else if (ResourceBundle.class.isAssignableFrom(curObject.getClass()))
                {
                    curObject = ((ResourceBundle) curObject).getObject(key);
                }
                // Array
                else if (curObject.getClass().isArray())
                {
                    curObject = ((Object[]) curObject)[Integer.parseInt(key)];
                }
                // List
                else if (List.class.isAssignableFrom(curObject.getClass()))
                {
                    curObject = ((List) curObject).get(Integer.parseInt(key));
                }
                // Collection
                else if (Collection.class.isAssignableFrom(curObject.getClass()))
                {
                    // Not very efficient, but at least it works
                    curObject = ((Collection) curObject).toArray()[Integer.parseInt(key)];
                }
            }
            else if (curSegment.getType() == QuerySegment.PROPERTY)  //regular properties
            {
                try
                {
                    if (descriptor instanceof IndexedPropertyDescriptor)
                    {
                        throw new IllegalArgumentException("Attempting to set a indexed field " + curSegment.getId() + " as an non-indexed field");
                    }

                    //if here then we are setting the property directly
                    if ((i + 1) == totalNumberOfSegments)
                    {
                        setValue(curObject, descriptor, val);
                        return;
                    }
                    //if here then read a property to continue on...
                    else
                    {
                        Method m = descriptor.getReadMethod();
                        if (m == null)
                        {
                            throw new IllegalArgumentException("Read Method " + descriptor.getName() + " for " + curObject.getClass().getName() + " was not found");
                        }
                        curObject = InjectionUtils.invoke(m, curObject, new Object[0]);
                    }
                }
                catch (IllegalArgumentException iae)
                {
                    throw iae;
                }
                catch (Exception e)
                {
                    log.warn("Error executing read method for " + descriptor.getName() + " on " + curObject.getClass().getName(), e);
                    throw new IllegalArgumentException("Error executing read method for " + descriptor.getName() + " on " + curObject.getClass().getName());
                }
            }
            //indexed properties
            else if (curSegment.getType() == QuerySegment.METHOD)
            {
                Integer index = getIndexedPropertyIndex(curSegment);

                try
                {
                    //if here then we are setting an indexed property directly
                    if ((i + 1) == totalNumberOfSegments)
                    {
                        if (descriptor instanceof IndexedPropertyDescriptor)
                        {
                            setIndexedValue(curObject, (IndexedPropertyDescriptor) descriptor, val, index);
                            return;
                        }
                        else
                        {
                            throw new IllegalArgumentException("Attempting to set non-indexed field \"" + curSegment.getId() + "\" as an indexed field");
                        }
                    }
                    //if here then get next object in the chain
                    else
                    {
                        if (descriptor instanceof IndexedPropertyDescriptor)
                        {
                            Method m = ((IndexedPropertyDescriptor) descriptor).getIndexedReadMethod();
                            if (m == null)
                            {
                                throw new IllegalArgumentException("Indexed Read Method " + descriptor.getName() + " for " + curObject.getClass().getName() + " was not found");
                            }
                            curObject = InjectionUtils.invoke(m, curObject, new Object[] { index });
                        }
                        else
                        {
                            throw new IllegalArgumentException("Attempting to access non-indexed field \"" + curSegment.getId() + "\" as an indexed field");
                        }
                    }
                }
                catch (IllegalArgumentException iae)
                {
                    throw iae;
                }
                catch (Exception e)
                {
                    throw new IllegalArgumentException("Error executing index read method for " + ((descriptor != null) ? descriptor.getName() : "<null>") + " on " + ((curObject != null) ? curObject.getClass().getName() : "<null>"));
                }
            }
        }
    }

    /**
     * Helper method to retrieve a property editor out of cache.
     */
    public static PropertyEditor getPropertyEditor(Class clazz)
    {
        // Try to retrieve editor from cache
        Object editor = propertyEditors.get(clazz);

        // If editor is null we must try to find it using the PropertyEditorManager
        if (editor == null)
        {
            editor = PropertyEditorManager.findEditor(clazz);
            if (editor == null)
            {
                // Flag that editor is null
                // Clone map and add the editor to the map
                // This way we can use the unsynchronized HashMap rather than Hashtable
                Map newEditorMap = (Map) ((HashMap) propertyEditors).clone();
                newEditorMap.put(clazz, new Byte("0"));
                propertyEditors = newEditorMap;
                log.debug("PropertyEditorManager.findEditor returned null for class: " + clazz.getName());
            }
            else if (editor instanceof FastPropertyEditor)
            {
                // Clone map and add the editor to the map
                // This way we can use the unsynchronized HashMap rather than Hashtable
                Map newEditorMap = (Map) ((HashMap) propertyEditors).clone();
                newEditorMap.put(clazz, editor);
                propertyEditors = newEditorMap;
            }
        }
        else if (editor instanceof Byte)
        {
            return null;
        }
        return (PropertyEditor) editor;
    }

    /**
     * Helper method that gets the value using a PropertyEditor If the editor is a FastPropertyEditor then getAsValue is
     * called otherwise normal setAsText, getValue calls are made
     */
    public static Object getAsValue(PropertyEditor pe, String txt)
    {
        if (pe instanceof FastPropertyEditor)
        {
            return ((FastPropertyEditor) pe).getAsValue(txt);
        }
        else
        {
            pe.setAsText(txt);
            return pe.getValue();
        }
    }

    /**
     * Call toString on object. If object has an associated ValidationEditorSupport property editor then its getAsText
     * will be called. This allows you to associate a common toString algorithm to call types. Objects of class String
     * are however only returned
     */
    public static String toStringValue(Object obj)
    {
        String result = "";
        if (obj != null)
        {
            // If the object is a String we just return it to get best performance
            if (obj instanceof String)
            {
                return (String) obj;
            }
            PropertyEditor pe = getPropertyEditor(obj.getClass());
            if (pe != null)
            {
                // FastPropertyEditors are cached so we MUST use the stateless
                // getAsText method to get the value
                if (pe instanceof FastPropertyEditor)
                {
                    result = ((FastPropertyEditor) pe).getAsText(obj);
                }
                else if (pe instanceof ValidationEditorSupport)
                {
                    pe.setValue(obj);
                    result = pe.getAsText();
                }
                else
                {
                    result = obj.toString();
                }
            }
            else
            {
                result = obj.toString();
            }
        }
        return result;
    }

    /**
     * Get the value of the Index for indexed Properties.
     *
     * @param curSegment QuerySegment of type METHOD that we need the index from
     *
     * @throws IllegalArgumentException
     */
    private static Integer getIndexedPropertyIndex(QuerySegment curSegment)
    {
        Integer index;
        try
        {
            List values = curSegment.getValues();
            if (values.size() != 1)
            {
                throw new IllegalArgumentException("Only Indexed properties allowed!");
            }
            Object value = values.get(0);
            if (value instanceof Query)
            {
                index = new Integer(((Query) value).getSegments()[0].getValues().get(0).toString());
            }
            else
            {
                index = new Integer(values.get(0).toString());
            }
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("Only Indexed properties allowed! - Parameter must be an integer ");
        }
        return index;
    }


    /**
     * Get the actual number of QuerySegments in the array
     *
     * @param segments QuerySegment array for the propertyName that we are setting
     */
    private static int findTheActualNumberOfSegments(QuerySegment[] segments)
    {
        for (int i = 0; i < segments.length; ++i)
        {
            if (segments[i] == null)
            {
                return i;
            }
        }
        return segments.length;  //should never  be here
    }


    /**
     * Set a non-indexed field.  Will delegate the actuall setting of the field to one of three methods.  Sections were
     * taken from BeanUtil.setProprty (rev.1.4)
     *
     * @param obj        the object to set the property on.
     * @param descriptor the PropertyDescriptor representing the property to set
     * @param val        the value to set
     *
     * @throws IllegalArgumentException
     */
    private static boolean setValue(Object obj, PropertyDescriptor descriptor, Object val)
            throws IllegalArgumentException
    {
        if (descriptor == null || descriptor.getWriteMethod() == null)
        {
            log.debug("No descriptor found for. Unable to set value. ");
            return false;
        }
        //no value, so don't set it.
        if (val == null)
        {
            return false;
        }

        Method m = descriptor.getWriteMethod();
        Class valueClass = val.getClass();
        // The getParameterTypes method is inefficient so we only call it once and save the result
        Class[] parameterClasses = m.getParameterTypes();
        Class parameterClass = parameterClasses[0];
        if (!valueClass.equals(String.class) && !valueClass.equals(String[].class) && !parameterClass.equals(String.class))
        {
            return setObjectDirectly(parameterClass, valueClass, val, m, obj);
        }

        String[] value = convertObjectToStringArray(val);

        if (value != null)
        {
            if (descriptor.getPropertyEditorClass() == null) // No property editor given -> try default conversions
            {
                if (setStringValueDirectly(obj, descriptor, value))
                {
                    return true;
                }
            }
            // Set the value using a property editor. Pass in the parameterClass
            setStringValueWithPropertyEditor(obj, descriptor, value, parameterClass);
            return true;
        }
        return false;
    }


    /**
     * Will convert an object to a String array
     *
     * @param val the object to convert to an array
     */
    private static String[] convertObjectToStringArray(Object val)
    {
        Class valueClass = val.getClass();

        if (valueClass.equals(String.class))
        {
            return new String[] { (String) val };
        }

        if (valueClass.equals(String[].class))
        {
            return (String[]) val;
        }

        return new String[] { val.toString() };
    }

    /**
     * Set an indexed field.  Will delegate the actual setting of the field to one of three methods.  Sections were
     * taken from BeanUtil.setProprty (rev.1.4)
     *
     * @param obj        the object to set the property on.
     * @param descriptor the PropertyDescriptor representing the property to set
     * @param val        the value to set
     * @param index      the index to set it at
     *
     * @throws IllegalArgumentException
     */
    private static boolean setIndexedValue(Object obj, IndexedPropertyDescriptor descriptor, Object val, Integer index)
    {
        if (descriptor == null)
        {
            log.info("no descriptor found");
            return false;
        }

        Method m = descriptor.getIndexedWriteMethod();
        Class valueClass = val.getClass();
        Class parameterClass = m.getParameterTypes()[1];
        if (!valueClass.equals(String.class) && !valueClass.equals(String[].class) && !parameterClass.equals(String.class))
        {
            return setIndexedObjectDirectly(parameterClass, valueClass, val, m, obj, index);
        }

        String[] value = convertObjectToStringArray(val);

        if (value != null)
        {
            if (descriptor.getPropertyEditorClass() == null) // No property editor given -> try default conversions
            {
                if (setIndexedStringValueDirectly(obj, descriptor, value, index))
                {
                    return true;
                }
            }
            setIndexedStringValueWithPropertyEditor(obj, descriptor, value, index);
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Set a non-indexed field with an Object directly.  This was taken from BeanUtil.setProprty (rev.1.4)
     *
     * @param parameterClass the class for the method
     * @param valueClass     the class for what we are actually trying to set it to.
     * @param val            the value we want to set
     * @param m              the method we are using to set the value
     * @param obj            the object we are trying to set it on
     * @param index          the index to set it at
     *
     * @throws IllegalArgumentException
     */
    private static boolean setIndexedObjectDirectly(Class parameterClass, Class valueClass, Object val, Method m, Object obj, Integer index)
    {
        // Try setting it directly
        if (!parameterClass.isAssignableFrom(valueClass))
        {
            // Convert numbers
            //LogFactory.getLog(this.getClass()).debug("Assignable:"+Number.class.isAssignableFrom(valueClass)+" "+Number.class.isAssignableFrom(parameterClass)+" "+parameterClass);
            if (parameterClass.equals(int.class))
            {
                parameterClass = Integer.class;
            }
            else if (parameterClass.equals(long.class))
            {
                parameterClass = Long.class;
            }
            if (Number.class.isAssignableFrom(valueClass) && Number.class.isAssignableFrom(parameterClass))
            {
                if (Long.class.isAssignableFrom(parameterClass))
                {
                    val = new Long(((Number) val).longValue());
                }
                else if (Integer.class.isAssignableFrom(parameterClass))
                {
                    val = new Integer(((Number) val).intValue());
                }
            }
            //                  throw new IllegalArgumentException("Parameter type of "+descriptor.getName()+"("+aMethod.getParameterTypes()[0].getName()+") does not match property type "+aValue.getClass().getName());
        }

        try
        {
            Object[] writeParameter = new Object[] { index, val };
            InjectionUtils.invoke(m, obj, writeParameter);
            return true;
        }
        catch (Exception e)
        {
            // Ignore and continue
            return false;
        }
    }

    /**
     * Set a non-indexed field with an Object directly.  This was taken from BeanUtil.setProprty (rev.1.4)
     *
     * @param parameterClass the class for the method
     * @param valueClass     the class for what we are actually trying to set it to.
     * @param val            the value we want to set
     * @param m              the method we are using to set the value
     * @param obj            the object we are trying to set it on
     *
     * @throws IllegalArgumentException
     */
    private static boolean setObjectDirectly(Class parameterClass, Class valueClass, Object val, Method m, Object obj)
    {
        // Try setting it directly
        if (!parameterClass.isAssignableFrom(valueClass))
        {
            // Convert numbers
            //Category.getInstance(this.getClass()).debug("Assignable:"+Number.class.isAssignableFrom(valueClass)+" "+Number.class.isAssignableFrom(parameterClass)+" "+parameterClass);
            if (parameterClass.equals(int.class))
            {
                parameterClass = Integer.class;
            }
            else if (parameterClass.equals(long.class))
            {
                parameterClass = Long.class;
            }
            if (Number.class.isAssignableFrom(valueClass) && Number.class.isAssignableFrom(parameterClass))
            {
                if (Long.class.isAssignableFrom(parameterClass))
                {
                    val = new Long(((Number) val).longValue());
                }
                else if (Integer.class.isAssignableFrom(parameterClass))
                {
                    val = new Integer(((Number) val).intValue());
                }
            }
            //                  throw new IllegalArgumentException("Parameter type of "+descriptor.getName()+"("+aMethod.getParameterTypes()[0].getName()+") does not match property type "+aValue.getClass().getName());
        }

        try
        {
            Object[] writeParameter = new Object[] { val };
            InjectionUtils.invoke(m, obj, writeParameter);
            return true;
        }
        catch (Exception e)
        {
            if (e instanceof InvocationTargetException)
            {
                Throwable t = ((InvocationTargetException) e).getTargetException();
                if (t instanceof IllegalArgumentException)
                {
                    throw (IllegalArgumentException) t;
                }
            }
            // Ignore and continue
            return false;
        }
    }


    /**
     * Set a non-indexed field with a String value directly.
     *
     * @param obj        the object we are trying to set it on
     * @param descriptor PropertyDescriptor representing the firled we want to set
     * @param values     the String array we are setting
     *
     * @throws IllegalArgumentException
     */
    private static boolean setStringValueDirectly(Object obj, PropertyDescriptor descriptor, String[] values)
            throws IllegalArgumentException
    {
        try
        {
            Method m = descriptor.getWriteMethod();
            if (descriptor.getPropertyType().equals(String.class) ||
                    descriptor.getPropertyType().equals(Object.class))
            {
                InjectionUtils.invoke(m, obj, new Object[] { values[0] });
                return true;
            }
            else if (descriptor.getPropertyType().equals(String[].class))
            {
                InjectionUtils.invoke(m, obj, new Object[] { values });
                return true;
            }
        }
        catch (Exception e)
        {
            if (e instanceof InvocationTargetException)
            {
                Throwable t = ((InvocationTargetException) e).getTargetException();
                if (t instanceof IllegalArgumentException)
                {
                    throw (IllegalArgumentException) t;
                }
            }
            throw new IllegalArgumentException(e.toString());
        }
        return false;
    }

    /**
     * Set a non-indexed field with a String value directly using the Property Editor.
     *
     * @param obj        the object we are trying to set it on
     * @param descriptor PropertyDescriptor representing the firled we want to set
     * @param values     the String array we are setting
     * @param paramClass The parameter class used in the write method. Passed in for efficiency.
     *
     * @throws IllegalArgumentException
     */
    private static void setStringValueWithPropertyEditor(Object obj, PropertyDescriptor descriptor, String[] values, Class paramClass)
            throws IllegalArgumentException
    {
        try
        {
            PropertyEditor pe = getPropertyEditor(descriptor);
            Method m = descriptor.getWriteMethod();
            if (pe == null)
            {
                throw new IllegalArgumentException("No property editor registered for this type"); // No editor for this parameter type
            }
            try
            {
                try
                {
                    // The paramClass was passed to this method, now get the component type
                    Class compType = paramClass.getComponentType();
                    if (compType != null)
                    {
                        Object a = Array.newInstance(compType, values.length);
                        for (int i = 0; i < values.length; i++)
                        {
                            Array.set(a, i, getAsValue(pe, values[i]));
                        }
                        InjectionUtils.invoke(m, obj, new Object[] { a });
                    }
                    else
                    {
                        Object realValue = getAsValue(pe, values[0]);
                        InjectionUtils.invoke(m, obj, new Object[] { realValue });
                    }
                }
                catch (NumberFormatException nfe)
                {   //need this for Sun's Editor's
                    throw new PropertyEditorException(PropertyMessage.BAD_VALUE, values[0]);
                }
            }
            catch (IllegalArgumentException ex)
            {
                if (obj instanceof IllegalArgumentAware)
                {
                    ((IllegalArgumentAware) obj).addIllegalArgumentException(descriptor.getName(), ex);
                }
                else
                {
                    throw ex;
                }
            }
        }
        catch (IllegalArgumentException iae)
        {
            throw iae;
        }
        catch (Exception e)
        {
            if (e instanceof InvocationTargetException)
            {
                Throwable t = ((InvocationTargetException) e).getTargetException();
                if (t instanceof IllegalArgumentException)
                {
                    throw (IllegalArgumentException) t;
                }
            }
            throw new IllegalArgumentException(e.toString());
        }
    }

    /**
     * Set a indexed field with a String value directly using the Property Editor.
     *
     * @param obj        the object we are trying to set it on
     * @param descriptor PropertyDescriptor representing the firled we want to set
     * @param values     the String array we are setting
     * @param index      the index that we want to set
     *
     * @throws IllegalArgumentException
     */
    private static void setIndexedStringValueWithPropertyEditor(Object obj, IndexedPropertyDescriptor descriptor, String[] values, Integer index)
            throws IllegalArgumentException
    {
        try
        {
            PropertyEditor pe = getPropertyEditor(descriptor);
            Method m = descriptor.getIndexedWriteMethod();
            if (pe == null)
            {
                throw new IllegalArgumentException("No property editor registered for this type"); // No editor for this parameter type
            }
            Object realValue = getAsValue(pe, values[0]);
            InjectionUtils.invoke(m, obj, new Object[] { index, realValue });
        }
        catch (IllegalArgumentException iae)
        {
            throw iae;
        }
        catch (Exception e)
        {
            if (e instanceof InvocationTargetException)
            {
                Throwable t = ((InvocationTargetException) e).getTargetException();
                if (t instanceof IllegalArgumentException)
                {
                    throw (IllegalArgumentException) t;
                }
            }
            throw new IllegalArgumentException(e.toString());
        }
    }

    /**
     * Set a indexed field with a String value directly.
     *
     * @param obj        the object we are trying to set it on
     * @param descriptor PropertyDescriptor representing the firled we want to set
     * @param values     the String array we are setting
     * @param index      the index that we want to set
     *
     * @throws IllegalArgumentException
     */
    private static boolean setIndexedStringValueDirectly(Object obj, IndexedPropertyDescriptor descriptor, String[] values, Integer index)
            throws IllegalArgumentException
    {
        try
        {
            Method m = descriptor.getIndexedWriteMethod();
            if (descriptor.getIndexedPropertyType().equals(String.class) || descriptor.getIndexedPropertyType().equals(Object.class))
            {
                InjectionUtils.invoke(m, obj, new Object[] { index, values[0] });
                return true;
            }
            else if (descriptor.getIndexedPropertyType().equals(String[].class))
            {
                InjectionUtils.invoke(m, obj, new Object[] { index, values });
                return true;
            }
        }
        catch (Exception e)
        {
            if (e instanceof InvocationTargetException)
            {
                Throwable t = ((InvocationTargetException) e).getTargetException();
                if (t instanceof IllegalArgumentException)
                {
                    throw (IllegalArgumentException) t;
                }
                else
                {
                    //this really will say nothing helpful, so let's log it
                    //more clutter, but at least debugging becomes possible
                    log.error("Error invoking indexed property method " +
                            descriptor.getIndexedWriteMethod().getName() + " index=" + index + " values=" + Arrays.asList(values), t);
                    throw new IllegalArgumentException(t.toString());
                }
            }
            throw new IllegalArgumentException(e.toString());
        }
        return false;
    }


    /**
     * Get the field map for a class.  The map is used by getPropertyDescriptor. If it does not exist for the class it
     * will call buildFieldMap
     *
     * @param objClass the object we want to build the field map for
     */
    private static Map getFieldMapForClass(Class objClass)
    {
        Map fieldMap = (Map) objectMap.get(objClass);
        if (fieldMap == null)
        {
            fieldMap = buildFieldMap(objClass);
            if (fieldMap != null)
            {
                objectMap.put(objClass, fieldMap);
            }
        }
        return fieldMap;
    }

    /**
     * Build field map for a class.
     *
     * @param objClass the object we want to build the field map for
     *
     * @return a Map with the keys being property names and values being PropertyDescriptors
     */
    private static Map buildFieldMap(Class objClass)
    {
        Map fieldMap = new WeakHashMap();
        PropertyDescriptor[] descriptors = getPropertyDescriptors(objClass);
        if (descriptors != null)
        {
            for (int i = 0; i < descriptors.length; ++i)
            {
                fieldMap.put(descriptors[i].getName(), descriptors[i]);
            }
            return fieldMap;
        }
        return null;
    }

    /**
     * get The property Editor for the propertyDescriptor.  Handles both Indexed and non_indexed properties
     *
     * @param descriptor PropertyDescriptor
     *
     * @return the PropertyEditor
     */
    private static PropertyEditor getPropertyEditor(PropertyDescriptor descriptor)
    {
        Class peClass = descriptor.getPropertyEditorClass();
        try
        {
            if (peClass == null)
            {
                if (descriptor instanceof IndexedPropertyDescriptor)
                {
                    return getPropertyEditor(((IndexedPropertyDescriptor) descriptor).getIndexedPropertyType());
                }
                else
                {
                    return getPropertyEditor(descriptor.getPropertyType());
                }
            }
            else
            {
                return (PropertyEditor) ObjectFactory.instantiate(peClass);
            }
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException(e.toString());
        }
    }

    /**
     * Get property descriptors for a class. Only return descriptors for properties that are not a property of
     * java.lang.Object based on webwork (0.95) Dispatcher Code
     *
     * @param objClass
     *
     * @return array of property descriptors
     */
    private static synchronized PropertyDescriptor[] getPropertyDescriptors(Class objClass)
    {
        PropertyDescriptor[] descriptors = (PropertyDescriptor[]) propertyDescriptors.get(objClass);

        // Check if available
        if (descriptors == null)
        {
            BeanInfo beanInfo;
            try
            {
                beanInfo = Introspector.getBeanInfo(objClass, Object.class);
            }
            catch (IntrospectionException e)
            {
                // We're in trouble
                // Let it try again on next request, but return "nothing" for now
                return new PropertyDescriptor[0];
            }

            // Get list of descriptors that are ok
            descriptors = beanInfo.getPropertyDescriptors();
            List list = new ArrayList(descriptors.length);
            for (int i = 0; i < descriptors.length; i++)
            {
                PropertyDescriptor descriptor = descriptors[i];
                if (descriptor instanceof IndexedPropertyDescriptor)
                {
                    Method readMethod = ((IndexedPropertyDescriptor) descriptor).getIndexedReadMethod();
                    Method writeMethod = ((IndexedPropertyDescriptor) descriptor).getIndexedWriteMethod();

                    //make sure it is not from Object
                    if (verifyNotObjectProperty(readMethod, writeMethod))
                    {
                        list.add(descriptor);
                    }
                }
                else
                {
                    Method readMethod = descriptor.getReadMethod();
                    Method writeMethod = descriptor.getWriteMethod();

                    if (verifyNotObjectProperty(readMethod, writeMethod))
                    {
                        list.add(descriptor);
                    }
                }
            }
            descriptors = new PropertyDescriptor[list.size()];
            list.toArray(descriptors);

            // Put list into cache
            propertyDescriptors.put(objClass, descriptors);
        }
        return descriptors;
    }

    /**
     * Verify That the property is not from Object
     *
     * @param readMethod  The ReadMethod From the PropertyDescriptor
     * @param writeMethod The WriteMethod From the PropertyDescriptor
     *
     * @return true if not Object property
     */
    private static boolean verifyNotObjectProperty(Method readMethod, Method writeMethod)
    {
        return (readMethod != null && !readMethod.getDeclaringClass().equals(Object.class)) || (writeMethod != null && !writeMethod.getDeclaringClass().equals(Object.class));
    }
}
