/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.config;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableList;

/**
 * Delegating implementation of configuration. Delegates to a list of other configurations.
 *
 * @author Rickard Őberg (rickard@middleware-company.com)
 * @version $Revision: 1.8 $
 */
public class DelegatingConfiguration extends AbstractConfiguration
{
    // Attributes ----------------------------------------------------
    private final List<ConfigurationInterface> configList;

    // Constructors --------------------------------------------------
    public DelegatingConfiguration(final List<ConfigurationInterface> configList)
    {
        this.configList = unmodifiableList(new ArrayList<ConfigurationInterface>(configList));
    }

    public DelegatingConfiguration(final ConfigurationInterface... configs)
    {
        this(asList(configs));
    }

    /**
     * Get a named setting.
     */
    @Override
    public Object getImpl(final String aName) throws IllegalArgumentException
    {
        // Delegate to the other configurations
        IllegalArgumentException e = null;
        for (final ConfigurationInterface config : configList)
        {
            try
            {
                return config.getImpl(aName);
            }
            catch (final IllegalArgumentException ex)
            {
                e = ex;
                // Try next config
            }
        }
        throw e;
    }

    /**
     * Set a named setting
     */
    @Override
    public void setImpl(final String aName, final Object aValue)
            throws IllegalArgumentException, UnsupportedOperationException
    {
        // Determine which config to use by using get
        // Delegate to the other configurations
        IllegalArgumentException e = null;
        for (final ConfigurationInterface config : configList)
        {
            try
            {
                config.getImpl(aName);

                // Found it, now try setting
                config.setImpl(aName, aValue);

                // Worked, now return
                return;
            }
            catch (final IllegalArgumentException ex)
            {
                e = ex;
                // Try next config
            }
        }
        throw e;
    }

    /**
     * List setting names
     */
    @Override
    public Iterator listImpl()
    {
        boolean workedAtAll = false;

        final ArrayList settingList = new ArrayList();
        UnsupportedOperationException e = null;
        for (final ConfigurationInterface config : configList)
        {
            try
            {
                final Iterator list = config.listImpl();
                while (list.hasNext())
                {
                    settingList.add(list.next());
                }
                workedAtAll = true;
            }
            catch (final UnsupportedOperationException ex)
            {
                e = ex;
                // Try next config
            }
        }

        if (!workedAtAll)
        {
            throw e == null ? new UnsupportedOperationException() : e;
        }
        else
        {
            return settingList.iterator();
        }
    }
}
