package io.intercom.android.sdk.push;

import android.app.Application;
import android.content.Context;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;

import com.intercom.twig.Twig;

import java.util.Map;

import io.intercom.android.sdk.Injector;
import io.intercom.android.sdk.Provider;
import io.intercom.android.sdk.identity.AppConfig;
import io.intercom.android.sdk.logger.LumberMill;
import io.intercom.android.sdk.utilities.ContextLocaliser;

/**
 * <p>The IntercomPushClient is responsible for communication between the host app push integration and Intercom.
 * This class is automatically called unless your app has it's own push integration or
 * uses other third party push integrations.</p>
 * <p>If you have a class that extends:
 * com.google.firebase.iid.FirebaseInstanceIdService or
 * com.google.firebase.messaging.FirebaseMessagingService
 * you will need to create an instance of IntercomPushClient:</p>
 * <pre>private final IntercomPushClient intercomPushClient = new IntercomPushClient();</pre>
 */
public class IntercomPushClient {
    private final Twig twig;
    private final PushHandler pushHandler;

    public IntercomPushClient() {
        this(LumberMill.getLogger(), new PushHandler());
    }

    @VisibleForTesting IntercomPushClient(Twig twig, PushHandler pushHandler) {
        this.twig = twig;
        this.pushHandler = pushHandler;
    }

    /**
     * <p>Handle the Intercom push message.</p>
     *
     * <p>This takes a push registration token to send to Intercom to enable this device to receive push.</p>
     * <p>In a FCM integration in the class that extends com.google.firebase.messaging.FirebaseMessagingService</p>
     * <p>You will need to pass the token to the intercomPushClient</p>
     *
     * <pre>
     * public void onNewToken(String refreshedToken) {
     *     pushClient.sendTokenToIntercom(getApplication(), refreshedToken);
     * }
     * </pre>
     *
     * @param application A reference to the Application
     * @param token A device push registration token.
     * @since 3.0.7
     */
    public void sendTokenToIntercom(@NonNull Application application, @NonNull String token) {
        if (TextUtils.isEmpty(token)) {
            twig.e("sendTokenToIntercom() was called with a null or empty token. "
                    + "This user will not receive push notifications until a valid device token is sent.");
            return;
        }

        Injector.initIfCachedCredentials(application);
        if (Injector.isNotInitialised()) {
            twig.w("Token not sent because Intercom is not initialised");
            return;
        }

        Injector injector = Injector.get();
        if (pushHandler.shouldSendDeviceToken(application, token)) {
            pushHandler.sendTokenToIntercom(application, token, injector.getApi(),
                    injector.getUserIdentity(), injector.getAppConfigProvider());
        }
    }

    /**
     * <p>Handle the Intercom push message.</p>
     *
     * <p>This will take the cloud message bundle and create a notification.</p>
     * <p>In the method onMessageReceived you will need to pass the message onto our push client:</p>
     *
     * <pre>
     * public void onMessageReceived(String from, Bundle message) {
     *     if (intercomPushClient.isIntercomPush(message)) {
     *         intercomPushClient.handlePush(getApplication(), message);
     *     } else {
     *         //DO HOST LOGIC HERE
     *     }
     * }
     * </pre>
     *
     * @param application A reference to the Application
     * @param message The bundle message which was sent to your client
     * @since 3.0.5
     */
    public void handlePush(@NonNull Application application, @NonNull Bundle message) {
        if (message == null) {
            twig.i("The message passed to handlePush() was null.");
            return;
        }
        if (!PushPayload.create(message).isIntercomPush()) {
            twig.i("The message passed to handlePush() was not an Intercom push message.");
            return;
        }

        Injector.initIfCachedCredentials(application);
        if (Injector.isNotInitialised()) {
            twig.w("Push not handled because Intercom is not initialised");
            return;
        }

        Injector injector = Injector.get();
        Provider<AppConfig> appConfigProvider = injector.getAppConfigProvider();
        Context localisedContext = new ContextLocaliser(appConfigProvider).createLocalisedContext(application);
        boolean appBackgrounded = injector.getStore().state().hostAppState().isBackgrounded();
        SystemNotificationManager systemNotificationManager = injector.getSystemNotificationManager();
        pushHandler.handlePush(message, injector.getUserIdentity(), systemNotificationManager, appBackgrounded,
                localisedContext, injector.getMetricTracker(), injector.getAppConfigProvider().get());
    }

    /**
     * <p>Handle the Intercom push message.</p>
     *
     * <p>This will take the cloud message map and create a notification.</p>
     *
     * <pre>
     * public void onMessageReceived(RemoteMessage remoteMessage) {
     *     Map&lt;String, String&gt; message = remoteMessage.getData();
     *     if (intercomPushClient.isIntercomPush(message)) {
     *         intercomPushClient.handlePush(getApplication(), message);
     *     } else {
     *         //DO HOST LOGIC HERE
     *     }
     * }
     * </pre>
     *
     * @param application A reference to the Application
     * @param message Pass in the remoteMessage.getData() from your onMessageReceived method.
     * @since 3.0.5
     */
    public void handlePush(@NonNull Application application, @NonNull Map<String, String> message) {
        if (message == null) {
            twig.i("The message passed to handlePush() was null.");
            return;
        }
        handlePush(application, convertMessageMapToBundle(message));
    }

    /**
     * <p>Check if the cloud message is from Intercom.</p>
     *
     * <p>This will take the cloud message bundle and determine if it is from Intercom.</p>
     *
     * @param message A push message bundle
     * @return a boolean allowing you to check if this is an Intercom cloud message or not.
     * @since 3.0.5
     */
    public boolean isIntercomPush(@NonNull Bundle message) {
        return message != null && PushPayload.create(message).isIntercomPush();
    }

    /**
     * <p>Check if the cloud message is from Intercom.</p>
     *
     * <p>This will take the cloud message map and determine if it is from Intercom.</p>
     *
     * @param message Pass in the remoteMessage.getData() from your onMessageReceived method.
     * @return a boolean allowing you to check if this is an Intercom cloud message or not.
     * @since 3.0.5
     */
    public boolean isIntercomPush(@NonNull Map<String, String> message) {
        return message != null && isIntercomPush(convertMessageMapToBundle(message));
    }

    //helper method to convert FCM RemoteMessage data map to a Bundle
    private Bundle convertMessageMapToBundle(Map<String, String> message) {
        Bundle bundle = new Bundle(message.size());
        for (Map.Entry<String, String> entry : message.entrySet()) {
            bundle.putString(entry.getKey(), entry.getValue());
        }
        return bundle;
    }

}
