/*
 * Copyright 2017 xyzxqs (xyzxqs@gmail.com)
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.xyzxqs.libs.xtask;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import io.github.xyzxqs.libs.xtask.func.Action;
import io.github.xyzxqs.libs.xtask.func.BiConsumer;
import io.github.xyzxqs.libs.xtask.func.BiFunction;
import io.github.xyzxqs.libs.xtask.func.Consumer;
import io.github.xyzxqs.libs.xtask.func.Function;
import io.github.xyzxqs.libs.xtask.func.Supplier;

/**
 * @author xyzxqs (xyzxqs@gmail.com)
 */

public class Xtask<T> implements XtaskStage<T>, Cancellable {

    static final Executor ASYNC_POOL = Executors.newCachedThreadPool();
    static final WorkerFactory WORKER_FACTORY = new WorkerFactory(ASYNC_POOL);

    Worker worker;
    private Result<T> taskResult;
    private boolean canceled;
    private final ArrayList<Consumer<Result<T>>> resultConsumers;
    private final ArrayList<Action> cancelActions;

    private XtaskStage<T> delegate;

    static Executor screenExecutor(Executor e) {
        if (e == null) throw new NullPointerException();
        return e;
    }

    /**
     * Returns a new Xtask that is asynchronously completed
     * by a task running in the {@link Executors#newCachedThreadPool()} with
     * the value obtained by calling the given Supplier.
     * <p>
     * Note: the supplier function will be invoke immediately when this Xtask created
     *
     * @param supplier a function returning the value to be used
     *                 to complete the returned Xtask
     * @param <T>      the function's return type
     * @return the new Xtask
     * @see #onResult(Consumer)
     * @see #ifCanceled(Action)
     */
    public static <T> Xtask<T> supplyAsync(final @NonNull Supplier<T> supplier) {
        return supplyAsync(supplier, ASYNC_POOL);
    }

    /**
     * Returns a new Xtask that is asynchronously completed
     * by a task running in the given executor with the value obtained
     * by calling the given Supplier.
     * <p>
     * Note: the supplier function will be invoke immediately when this Xtask created
     *
     * @param supplier a function returning the value to be used
     *                 to complete the returned CompletableFuture
     * @param exec     the executor to use for asynchronous execution
     * @param <T>      the function's return type
     * @return the new Xtask
     * @see #onResult(Consumer)
     * @see #ifCanceled(Action)
     */
    public static <T> Xtask<T> supplyAsync(final @NonNull Supplier<T> supplier,
                                           @NonNull Executor exec) {
        final Xtask<T> task = new Xtask<>();
        task.worker = WORKER_FACTORY.<T>createWorker()
                .supplyAsync(new Supplier<T>() {
                    @Override
                    public T get() throws Exception {
                        return supplier.get();
                    }
                })
                .onResult(new Consumer<Result<T>>() {
                    @Override
                    public void accept(Result<T> result) throws Exception {
                        task.result(result);
                    }
                });/*no ifCanceled(...), Xtask<T> handle this itself*/
        task.execute(screenExecutor(exec));
        return task;
    }


    Xtask() {
        resultConsumers = new ArrayList<>();
        cancelActions = new ArrayList<>();
        canceled = false;
        delegate = new XtaskStageImpl<>(this);
    }

    void execute() {
        execute(ASYNC_POOL);
    }

    void execute(Executor exec) {
        canceled = false;
        worker.execute(exec);
    }

    void result(@NonNull Result<T> result) {
        if (!canceled) {
            taskResult = result;
            for (Consumer<Result<T>> consumer : resultConsumers) {
                try {
                    consumer.accept(result);
                } catch (Exception e) {
                    //ignore
                }
            }
        }
    }

    /**
     * Cancel this Xtask
     */
    @Override
    public void cancel() {
        canceled = true;
        if (worker != null) {
            worker.cancel();
        }
        for (Action action : cancelActions) {
            try {
                action.run();
            } catch (Exception e) {
                //ignore
            }
        }
    }

    /**
     * Register the {@link Result<T>} consumer what is called in android main thread
     * when this Xtask finish it's task
     * <p>
     * Note: if this Xtask has been invoked, the consumer function will be called immediately
     *
     * @param consumer result consumer
     * @return this Xtask
     */
    public Xtask<T> onResult(@NonNull Consumer<Result<T>> consumer) {
        Objects.checkNotNull(consumer, "consumer == null");
        resultConsumers.add(consumer);
        if (taskResult != null) {
            try {
                consumer.accept(taskResult);
            } catch (Exception e) {
                //ignore
            }
        }
        return this;
    }

    /**
     * Register a cancel Action what is called in android main thread
     * when this Xtask was canceled
     * <p>
     * Note: if this Xtask has been canceled, the action function will be called immediately
     *
     * @param action action when this Xtask be canceled
     * @return this Xtask
     */
    public Xtask<T> ifCanceled(@NonNull Action action) {
        Objects.checkNotNull(action, "action == null");
        cancelActions.add(action);
        if (canceled) {
            try {
                action.run();
            } catch (Exception e) {
                //ignore
            }
        }
        return this;
    }

    @Override
    public <U> Xtask<U> thenApplyAsync(@NonNull Function<Result<T>, U> fn) {
        return delegate.thenApplyAsync(fn);
    }

    @Override
    public <U> Xtask<U> thenApplyAsync(@NonNull Function<Result<T>, U> fn, @NonNull Executor exec) {
        return delegate.thenApplyAsync(fn, exec);
    }

    @Override
    public Xtask<Void> thenAcceptAsync(@NonNull Consumer<Result<T>> action) {
        return delegate.thenAcceptAsync(action);
    }

    @Override
    public Xtask<Void> thenAcceptAsync(@NonNull Consumer<Result<T>> action, @NonNull Executor exec) {
        return delegate.thenAcceptAsync(action, exec);
    }

    @Override
    public Xtask<Void> thenRunAsync(@NonNull Action action) {
        return delegate.thenRunAsync(action);
    }

    @Override
    public Xtask<Void> thenRunAsync(@NonNull Action action, @NonNull Executor exec) {
        return delegate.thenRunAsync(action, exec);
    }

    @Override
    public <U, V> Xtask<V> thenCombineAsync(@NonNull Xtask<U> other, @NonNull BiFunction<Result<T>, Result<U>, V> fn) {
        return delegate.thenCombineAsync(other, fn);
    }

    @Override
    public <U, V> Xtask<V> thenCombineAsync(@NonNull Xtask<U> other, @NonNull BiFunction<Result<T>, Result<U>, V> fn, @NonNull Executor exec) {
        return delegate.thenCombineAsync(other, fn, exec);
    }

    @Override
    public <U> Xtask<Void> thenAcceptBothAsync(@NonNull Xtask<U> other, @NonNull BiConsumer<Result<T>, Result<U>> action) {
        return delegate.thenAcceptBothAsync(other, action);
    }

    @Override
    public <U> Xtask<Void> thenAcceptBothAsync(@NonNull Xtask<U> other, @NonNull BiConsumer<Result<T>, Result<U>> action, @NonNull Executor exec) {
        return delegate.thenAcceptBothAsync(other, action, exec);
    }

    @Override
    public <U> Xtask<Void> runAfterBothAsync(@NonNull Xtask<U> other, @NonNull Action action) {
        return delegate.runAfterBothAsync(other, action);
    }

    @Override
    public <U> Xtask<Void> runAfterBothAsync(@NonNull Xtask<U> other, @NonNull Action action, @NonNull Executor exec) {
        return delegate.runAfterBothAsync(other, action, exec);
    }

    @Override
    public <U> Xtask<U> applyToEitherAsync(@NonNull Xtask<T> other, @NonNull Function<Result<T>, U> fn) {
        return delegate.applyToEitherAsync(other, fn);
    }

    @Override
    public <U> Xtask<U> applyToEitherAsync(@NonNull Xtask<T> other, @NonNull Function<Result<T>, U> fn, @NonNull Executor exec) {
        return delegate.applyToEitherAsync(other, fn, exec);
    }

    @Override
    public Xtask<Void> acceptEitherAsync(@NonNull Xtask<T> other, @NonNull Consumer<Result<T>> action) {
        return delegate.acceptEitherAsync(other, action);
    }

    @Override
    public Xtask<Void> acceptEitherAsync(@NonNull Xtask<T> other, @NonNull Consumer<Result<T>> action, @NonNull Executor exec) {
        return delegate.acceptEitherAsync(other, action, exec);
    }

    @Override
    public <U> Xtask<Void> runAfterEitherAsync(@NonNull Xtask<U> other, @NonNull Action action) {
        return delegate.runAfterEitherAsync(other, action);
    }

    @Override
    public <U> Xtask<Void> runAfterEitherAsync(@NonNull Xtask<U> other, @NonNull Action action, @NonNull Executor exec) {
        return delegate.runAfterEitherAsync(other, action, exec);
    }
}
