/*
 * Copyright 2016 xyzxqs (xyzxqs@gmail.com)
 * <i>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <i>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <i>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.xyzxqs.libs.xtask;


import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.Log;

import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;

import io.github.xyzxqs.libs.xtask.func.Action;
import io.github.xyzxqs.libs.xtask.func.Consumer;
import io.github.xyzxqs.libs.xtask.func.Supplier;


/**
 * @author xyzxqs (xyzxqs@gmail.com)
 */

class WorkerImpl<T> implements Worker<T> {
    private static final String TAG = "WorkerImpl";

    private static final int MSG_ON_SUCCESS = 63;
    private static final int MSG_ON_FAILURE = 354;
    private static final int MSG_ON_CANCELED = 621;

    private final Executor executor;
    private final Handler handler;

    private WorkerTask workerTask;

    private Supplier<T> tSupplier;
    private Action cancelAction;

    private Consumer<Result<T>> resultConsumer;

    WorkerImpl(Executor executor) {
        this.executor = executor;
        this.handler = new WorkerHandler<>(this);
    }


    @Override
    public Worker<T> supplyAsync(Supplier<T> supplier) {
        this.tSupplier = Objects.checkNotNull(supplier, "tSupplier == null");
        return this;
    }

    @Override
    public Worker<T> onResult(Consumer<Result<T>> consumer) {
        resultConsumer = Objects.checkNotNull(consumer, "consumer == null");
        return this;
    }


    @Override
    public Worker<T> ifCanceled(Action action) {
        cancelAction = Objects.checkNotNull(action, "action == null");
        return this;
    }

    @Override
    public void execute() {
        execute(executor);
    }

    @Override
    public void execute(Executor exec) {
        exec.execute(workerTask = new WorkerTask<>(handler, tSupplier));
    }

    @Override
    public void cancel() {
        if (workerTask != null) {
            workerTask.cancel();
        }
    }

    private void canceled() {
        if (cancelAction != null) {
            try {
                cancelAction.run();
            } catch (Exception e) {
                loge(e);
            }
        }
    }

    private void failure(Throwable throwable) {
        if (resultConsumer != null) {
            try {
                resultConsumer.accept(Result.<T>error(throwable));
            } catch (Exception e) {
                loge(e);
            }
        }
    }

    private void success(T t) {
        if (resultConsumer != null) {
            try {
                resultConsumer.accept(Result.result(t));
            } catch (Exception e) {
                loge(e);
            }
        }
    }

    private void loge(Exception e) {
        Log.e(TAG, "An error occurred: ", e);
    }

    private static class WorkerHandler<U> extends Handler {

        private final WorkerImpl<U> worker;

        private WorkerHandler(WorkerImpl<U> worker) {
            super(Looper.getMainLooper());
            this.worker = worker;
        }

        @SuppressWarnings("unchecked")
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            switch (msg.what) {
                case MSG_ON_CANCELED:
                    worker.canceled();
                    break;
                case MSG_ON_FAILURE:
                    worker.failure((Throwable) msg.obj);
                    break;
                case MSG_ON_SUCCESS:
                    worker.success((U) msg.obj);
                    break;
            }
        }
    }

    private static class WorkerTask<U> implements Runnable, Cancellable {
        private final Handler handler;
        private final Supplier<U> function;
        private final AtomicBoolean canceled;

        private WorkerTask(Handler handler, Supplier<U> function) {
            this.handler = handler;
            this.function = function;
            canceled = new AtomicBoolean(false);
        }

        @Override
        public void cancel() {
            canceled.set(true);
        }

        @Override
        public void run() {

            if (canceled.get()) {
                handler.sendEmptyMessage(MSG_ON_CANCELED);
                return;
            }

            U u;
            try {
                u = function.get();
            } catch (Exception e) {
                handler.obtainMessage(MSG_ON_FAILURE, e).sendToTarget();
                return;
            }

            if (u != null) {
                if (canceled.get()) {
                    handler.sendEmptyMessage(MSG_ON_CANCELED);
                } else {
                    handler.obtainMessage(MSG_ON_SUCCESS, u).sendToTarget();
                }
            }
        }
    }
}
