/*
 * Copyright 2017 xyzxqs (xyzxqs@gmail.com)
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.xyzxqs.libs.xtask;

import android.support.annotation.NonNull;

import java.util.concurrent.Executor;

import io.github.xyzxqs.libs.xtask.func.Action;
import io.github.xyzxqs.libs.xtask.func.BiConsumer;
import io.github.xyzxqs.libs.xtask.func.BiFunction;
import io.github.xyzxqs.libs.xtask.func.Consumer;
import io.github.xyzxqs.libs.xtask.func.Function;

/**
 * More like {@link java.util.concurrent.CompletionStage} but this is for android
 * and XtaskStage only keep the *Async(...) methods
 *
 * @author xyzxqs (xyzxqs@gmail.com)
 */

public interface XtaskStage<T> {

    /**
     * Returns a new XtaskStage that, when this stage completes
     * normally, is executed using this stage's default asynchronous
     * execution facility, with this stage's result as the argument to
     * the supplied function.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param fn  the function to use to compute the value of the
     *            returned XtaskStage
     * @param <U> the function's return type
     * @return the new XtaskStage
     */
    <U> Xtask<U> thenApplyAsync(@NonNull Function<Result<T>, U> fn);

    /**
     * Returns a new XtaskStage that, when this stage completes
     * normally, is executed using the supplied Executor, with this
     * stage's result as the argument to the supplied function.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param fn       the function to use to compute the value of the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @param <U>      the function's return type
     * @return the new XtaskStage
     */
    <U> Xtask<U> thenApplyAsync(@NonNull Function<Result<T>, U> fn,
                                @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when this stage completes
     * normally, is executed using this stage's default asynchronous
     * execution facility, with this stage's result as the argument to
     * the supplied action.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param action the action to perform before completing the
     *               returned XtaskStage
     * @return the new XtaskStage
     */
    Xtask<Void> thenAcceptAsync(@NonNull Consumer<Result<T>> action);

    /**
     * Returns a new XtaskStage that, when this stage completes
     * normally, is executed using the supplied Executor, with this
     * stage's result as the argument to the supplied action.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param action   the action to perform before completing the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @return the new XtaskStage
     */
    Xtask<Void> thenAcceptAsync(@NonNull Consumer<Result<T>> action,
                                @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when this stage completes
     * normally, executes the given action using this stage's default
     * asynchronous execution facility.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param action the action to perform before completing the
     *               returned XtaskStage
     * @return the new XtaskStage
     */
    Xtask<Void> thenRunAsync(@NonNull Action action);

    /**
     * Returns a new XtaskStage that, when this stage completes
     * normally, executes the given action using the supplied Executor.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param action   the action to perform before completing the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @return the new XtaskStage
     */
    Xtask<Void> thenRunAsync(@NonNull Action action, @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when this and the other
     * given stage both complete normally, is executed using this
     * stage's default asynchronous execution facility, with the two
     * results as arguments to the supplied function.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other the other XtaskStage
     * @param fn    the function to use to compute the value of the
     *              returned XtaskStage
     * @param <U>   the type of the other XtaskStage's result
     * @param <V>   the function's return type
     * @return the new XtaskStage
     */
    <U, V> Xtask<V> thenCombineAsync(@NonNull Xtask<U> other,
                                     @NonNull BiFunction<Result<T>, Result<U>, ? extends V> fn);

    /**
     * Returns a new XtaskStage that, when this and the other
     * given stage both complete normally, is executed using the
     * supplied executor, with the two results as arguments to the
     * supplied function.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other    the other XtaskStage
     * @param fn       the function to use to compute the value of the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @param <U>      the type of the other XtaskStage's result
     * @param <V>      the function's return type
     * @return the new XtaskStage
     */
    <U, V> Xtask<V> thenCombineAsync(@NonNull Xtask<U> other,
                                     @NonNull BiFunction<Result<T>, Result<U>, ? extends V> fn,
                                     @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when this and the other
     * given stage both complete normally, is executed using this
     * stage's default asynchronous execution facility, with the two
     * results as arguments to the supplied action.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other  the other XtaskStage
     * @param action the action to perform before completing the
     *               returned XtaskStage
     * @param <U>    the type of the other XtaskStage's result
     * @return the new XtaskStage
     */
    <U> Xtask<Void> thenAcceptBothAsync(@NonNull Xtask<U> other,
                                        @NonNull BiConsumer<Result<T>, Result<U>> action);

    /**
     * Returns a new XtaskStage that, when this and the other
     * given stage both complete normally, is executed using the
     * supplied executor, with the two results as arguments to the
     * supplied action.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other    the other XtaskStage
     * @param action   the action to perform before completing the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @param <U>      the type of the other XtaskStage's result
     * @return the new XtaskStage
     */
    <U> Xtask<Void> thenAcceptBothAsync(@NonNull Xtask<U> other,
                                        @NonNull BiConsumer<Result<T>, Result<U>> action,
                                        @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when this and the other
     * given stage both complete normally, executes the given action
     * using this stage's default asynchronous execution facility.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other  the other XtaskStage
     * @param action the action to perform before completing the
     *               returned XtaskStage
     * @return the new XtaskStage
     */
    <U> Xtask<Void> runAfterBothAsync(@NonNull Xtask<U> other, @NonNull Action action);

    /**
     * Returns a new XtaskStage that, when this and the other
     * given stage both complete normally, executes the given action
     * using the supplied executor.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other    the other XtaskStage
     * @param action   the action to perform before completing the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @return the new XtaskStage
     */
    <U> Xtask<Void> runAfterBothAsync(@NonNull Xtask<U> other,
                                      @NonNull Action action,
                                      @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when either this or the
     * other given stage complete normally, is executed using this
     * stage's default asynchronous execution facility, with the
     * corresponding result as argument to the supplied function.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other the other XtaskStage
     * @param fn    the function to use to compute the value of the
     *              returned XtaskStage
     * @param <U>   the function's return type
     * @return the new XtaskStage
     */
    <U> Xtask<U> applyToEitherAsync(@NonNull Xtask<T> other,
                                    @NonNull Function<Result<T>, U> fn);

    /**
     * Returns a new XtaskStage that, when either this or the
     * other given stage complete normally, is executed using the
     * supplied executor, with the corresponding result as argument to
     * the supplied function.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other    the other XtaskStage
     * @param fn       the function to use to compute the value of the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @param <U>      the function's return type
     * @return the new XtaskStage
     */
    <U> Xtask<U> applyToEitherAsync(@NonNull Xtask<T> other,
                                    @NonNull Function<Result<T>, U> fn,
                                    @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when either this or the
     * other given stage complete normally, is executed using this
     * stage's default asynchronous execution facility, with the
     * corresponding result as argument to the supplied action.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other  the other XtaskStage
     * @param action the action to perform before completing the
     *               returned XtaskStage
     * @return the new XtaskStage
     */
    Xtask<Void> acceptEitherAsync(@NonNull Xtask<T> other, @NonNull Consumer<Result<T>> action);

    /**
     * Returns a new XtaskStage that, when either this or the
     * other given stage complete normally, is executed using the
     * supplied executor, with the corresponding result as argument to
     * the supplied action.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other    the other XtaskStage
     * @param action   the action to perform before completing the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @return the new XtaskStage
     */
    Xtask<Void> acceptEitherAsync(@NonNull Xtask<T> other,
                                  @NonNull Consumer<Result<T>> action,
                                  @NonNull Executor executor);

    /**
     * Returns a new XtaskStage that, when either this or the
     * other given stage complete normally, executes the given action
     * using this stage's default asynchronous execution facility.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other  the other XtaskStage
     * @param action the action to perform before completing the
     *               returned XtaskStage
     * @return the new XtaskStage
     */
    <U> Xtask<Void> runAfterEitherAsync(@NonNull Xtask<U> other, @NonNull Action action);

    /**
     * Returns a new XtaskStage that, when either this or the
     * other given stage complete normally, executes the given action
     * using the supplied executor.
     * <p>
     * See the {@link XtaskStage} documentation for rules
     * covering exceptional completion.
     *
     * @param other    the other XtaskStage
     * @param action   the action to perform before completing the
     *                 returned XtaskStage
     * @param executor the executor to use for asynchronous execution
     * @return the new XtaskStage
     */
    <U> Xtask<Void> runAfterEitherAsync(@NonNull Xtask<U> other,
                                        @NonNull Action action,
                                        @NonNull Executor executor);
}
