/*
 * Copyright 2020 Wultra s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.getlime.security.powerauth.keychain;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.security.keystore.KeyGenParameterSpec;
import android.security.keystore.KeyInfo;
import android.security.keystore.KeyProperties;
import android.security.keystore.StrongBoxUnavailableException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.text.TextUtils;

import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.List;

import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;

import io.getlime.security.powerauth.system.PA2Log;

/**
 * The {@code SymmetricKeyProvider} class manages symmetric encryption key stored in Android KeyStore.
 */
@RequiresApi(api = Build.VERSION_CODES.M)
public class SymmetricKeyProvider {

    private static final String ANDROID_KEY_STORE = "AndroidKeyStore";

    private final @NonNull KeyStore keyStore;
    private final @NonNull String keyAlias;
    private final @NonNull OnGenerateKey onGenerateKey;

    /**
     * The {@code SymmetricKeyProvider.OnGenerateKey} interface allows additional key configuration
     * before the secret key is generated by Android KeyStore.
     */
    public interface OnGenerateKey {
        /**
         * Method called before the key is generated by Android KeyStore.
         * @param builder Builder object that can be configured before the key is generated.
         */
        void configureBuilder(@NonNull KeyGenParameterSpec.Builder builder);
    }

    /**
     * Create a new instance of this class.
     *
     * @param keyAlias Key alias that identifies the key in Android KeyStore.
     * @param onGenerateKey Required key configuration.
     * @return Instance of this class or {@code null} in case that Android KeyStore is not available.
     */
    @Nullable
    public static SymmetricKeyProvider getSymmetricKeyProvider(
            @NonNull final String keyAlias,
            @NonNull final OnGenerateKey onGenerateKey) {
        try {
            final KeyStore keyStore = KeyStore.getInstance(ANDROID_KEY_STORE);
            keyStore.load(null);
            return new SymmetricKeyProvider(keyStore, keyAlias, onGenerateKey);

        } catch (KeyStoreException | CertificateException | IOException | NoSuchAlgorithmException e) {
            PA2Log.e("SymmetricKeyProvider: " + keyAlias + ": Unable to initialize Android KeyStore. Exception: " + e.getMessage());
            return null;
        }
    }

    /**
     * Create a new instance of this class, configured to generate keys for AES-GCM cipher.
     *
     * @param keyAlias Key alias that identifies the key in Android KeyStore.
     * @param keySize Number of bits for generated key.
     * @param randomizedEncryption Whether encryption using this key must be sufficiently randomized
     *                             to produce different ciphertexts for the same plaintext every time.
     * @param onGenerateKey Additional key configuration.
     * @return Instance of this class or {@code null} in case that Android KeyStore is not available.
     */
    @Nullable
    public static SymmetricKeyProvider getAesGcmKeyProvider(
            @NonNull final String keyAlias,
            final int keySize,
            final boolean randomizedEncryption,
            @Nullable final OnGenerateKey onGenerateKey) {
        return getSymmetricKeyProvider(keyAlias, new OnGenerateKey() {
            @Override
            public void configureBuilder(@NonNull KeyGenParameterSpec.Builder builder) {
                builder
                        .setBlockModes(KeyProperties.BLOCK_MODE_GCM)
                        .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
                        .setRandomizedEncryptionRequired(randomizedEncryption);
                if (keySize != 0) {
                    builder.setKeySize(keySize);
                }
                if (onGenerateKey != null) {
                    onGenerateKey.configureBuilder(builder);
                }
            }
        });
    }

    /**
     * Create a new instance of this class, configured to generate keys for AES-CBC cipher.
     * @param keyAlias Key alias that identifies the key in Android KeyStore.
     * @param keySize Number of bits for generated key.
     * @param randomizedEncryption Whether encryption using this key must be sufficiently randomized
     *                             to produce different ciphertexts for the same plaintext every time.
     * @param onGenerateKey Additional key configuration.
     * @return Instance of this class or {@code null} in case that Android KeyStore is not available.
     */
    @Nullable
    public static SymmetricKeyProvider getAesCbcKeyProvider(
            @NonNull final String keyAlias,
            final int keySize,
            final boolean randomizedEncryption,
            @Nullable final OnGenerateKey onGenerateKey) {
        return getSymmetricKeyProvider(keyAlias, new OnGenerateKey() {
            @Override
            public void configureBuilder(@NonNull KeyGenParameterSpec.Builder builder) {
                builder
                        .setBlockModes(KeyProperties.BLOCK_MODE_CBC)
                        .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_PKCS7)
                        .setRandomizedEncryptionRequired(randomizedEncryption);
                if (keySize != 0) {
                    builder.setKeySize(keySize);
                }
                if (onGenerateKey != null) {
                    onGenerateKey.configureBuilder(builder);
                }
            }
        });
    }

    /**
     * Get existing secret key or create a new one if such key is not exist in Android KeyStore.
     * @param context Android context object.
     * @param forceCreateOnFailure If {@code true} then key is re-generated in case that key exists
     *                             in KeyStore but cannot be obtained.
     * @return Instance of {@link SecretKey} obtained from Android KeyStore or {@code null} in case of failure.
     */
    @Nullable
    public SecretKey getOrCreateSecretKey(@NonNull Context context, boolean forceCreateOnFailure) {
        synchronized (SymmetricKeyProvider.class) {
            if (hasSecretKey()) {
                try {
                    return (SecretKey) keyStore.getKey(keyAlias, null);
                } catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
                    PA2Log.e("SymmetricKeyProvider: " + keyAlias + "Failed to get key. Exception: " + e.getMessage());
                    if (!forceCreateOnFailure) {
                        return null;
                    }
                    removeSecretKey();
                }
            }
            final SecretKey newSecretKey;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                if (isStrongBoxSupported(context)) {
                    newSecretKey = generateStrongBoxSecretKey(context);
                } else {
                    newSecretKey = generateSecretKey();
                }
            } else {
                newSecretKey = generateSecretKey();
            }
            if (newSecretKey != null) {
                final KeyInfo newKeyInfo = getKeyInfoForSecretKey(newSecretKey);
                PA2Log.d("SymmetricKeyProvider: " + keyAlias + ": Created key with attributes: " + getSecretKeyAttributes(newKeyInfo));
            }
            return newSecretKey;
        }
    }

    /**
     * Delete key from Android KeyStore.
     */
    public void deleteSecretKey() {
        synchronized (SymmetricKeyProvider.class) {
            if (hasSecretKey()) {
                removeSecretKey();
            }
        }
    }

    /**
     * @return {@code true} if key exists in Android KeyStore.
     */
    public boolean containsSecretKey() {
        synchronized (SymmetricKeyProvider.class) {
            return hasSecretKey();
        }
    }

    /**
     * @return Key alias that identifies the key in Android KeyStore.
     */
    @NonNull
    public String getKeyAlias() {
        return keyAlias;
    }

    /**
     * Get {@link KeyInfo} object from the key provided by this key provider.
     *
     * @param context Android context.
     * @return {@link KeyInfo} or {@code null} in case that key or information about key cannot be created.
     */
    @Nullable
    public KeyInfo getSecretKeyInfo(@NonNull Context context) {
        synchronized (SymmetricKeyProvider.class) {
            return getKeyInfoForSecretKey(getOrCreateSecretKey(context, false));
        }
    }

    /**
     * Dump information about key into {@link PA2Log} debug log.
     *
     * @param context Android context.
     * @return {@link KeyInfo} acquired for key, or {@code null} if information is not available.
     */
    @Nullable
    public KeyInfo dumpSecretKeyInfo(@NonNull Context context) {
        synchronized (SymmetricKeyProvider.class) {
            if (hasSecretKey()) {
                final SecretKey secretKey = getOrCreateSecretKey(context, false);
                final KeyInfo secretKeyInfo = getKeyInfoForSecretKey(secretKey);
                if (secretKey != null) {
                    PA2Log.d("SymmetricKeyProvider.dumpSecretKeyInfo: " + keyAlias + ": Key is available and has attributes: " + getSecretKeyAttributes(secretKeyInfo));
                } else {
                    PA2Log.d("SymmetricKeyProvider.dumpSecretKeyInfo: " + keyAlias + ": Failed to get key.");
                }
                return secretKeyInfo;
            } else {
                PA2Log.d("SymmetricKeyProvider.dumpSecretKeyInfo: " + keyAlias + ": Key is not created yet.");
                return null;
            }
        }
    }

    // Private methods

    /**
     * Private class constructor.
     *
     * @param keyStore Instance of {@link KeyStore} class.
     * @param keyAlias Key alias that identifies the key in Android KeyStore.
     * @param onGenerateKey Required key configuration.
     */
    private SymmetricKeyProvider(@NonNull KeyStore keyStore, @NonNull String keyAlias, @NonNull OnGenerateKey onGenerateKey) {
        this.keyStore = keyStore;
        this.keyAlias = keyAlias;
        this.onGenerateKey = onGenerateKey;
    }

    /**
     * Acquire information about key from Android KeyStore.
     *
     * @param secretKey {@link SecretKey} to evaluate.
     * @return {@link KeyInfo} or {@code null} in case of failure.
     */
    @Nullable
    private KeyInfo getKeyInfoForSecretKey(@Nullable SecretKey secretKey) {
        try {
            if (secretKey == null) {
                return null;
            }
            final SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(secretKey.getAlgorithm(), ANDROID_KEY_STORE);
            return (KeyInfo) keyFactory.getKeySpec(secretKey, KeyInfo.class);
        } catch (NoSuchAlgorithmException | NoSuchProviderException | InvalidKeySpecException e) {
            PA2Log.e("SymmetricKeyProvider: " + keyAlias + ": Failed to acquire KeyInfo: Exception: " + e.getMessage());
            return null;
        }
    }

    /**
     * Acquire list of comma separated attributes associated with the key. The method is useful only for debugging purposes.
     * @param keyInfo {@link KeyInfo} to evaluate.
     * @return Comma separated list of attributes.
     */
    @NonNull
    private String getSecretKeyAttributes(@Nullable KeyInfo keyInfo) {
        if (keyInfo == null) {
            return "N/A";
        }
        List<String> attrs = new ArrayList<>();
        if (keyInfo.isInsideSecureHardware()) {
            attrs.add("secure-hw");
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            if (keyInfo.isInvalidatedByBiometricEnrollment()) {
                attrs.add("inv-by-bio-enrollment");
            }
        }
        if (keyInfo.isUserAuthenticationRequired()) {
            attrs.add("user-auth-required");
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            if (keyInfo.isUserConfirmationRequired()) {
                attrs.add("user-conf-required");
            }
        }
        if (attrs.isEmpty()) {
            return "none";
        }
        return TextUtils.join(", ", attrs);
    }

    /**
     * Generate new StrongBox backed key.
     *
     * @param context Android context object.
     * @return {@link SecretKey} or {@code null} in case of failure.
     */
    @Nullable
    @RequiresApi(api = Build.VERSION_CODES.P)
    private SecretKey generateStrongBoxSecretKey(@NonNull Context context) {
        try {
            final KeyGenerator keyGenerator = KeyGenerator.getInstance(KeyProperties.KEY_ALGORITHM_AES, ANDROID_KEY_STORE);
            keyGenerator.init(getSecretKeySpecBuilder()
                    .setIsStrongBoxBacked(true)
                    .build());
            return keyGenerator.generateKey();

        } catch (StrongBoxUnavailableException | NoSuchAlgorithmException | NoSuchProviderException | InvalidAlgorithmParameterException e) {
            PA2Log.d("SymmetricKeyProvider: " + keyAlias + ": Failed to generate new StrongBox backed key. Exception: " + e.getMessage());
            return null;
        }
    }

    /**
     * Determine whether StrongBox hardware is available on the device.
     * @param context Android context
     * @return {@code true} if StrongBox is supported on the device.
     */
    @RequiresApi(api = Build.VERSION_CODES.P)
    private boolean isStrongBoxSupported(@NonNull Context context) {
        return context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_STRONGBOX_KEYSTORE);
    }

    /**
     * Generate a new secret key.
     * @return {@link SecretKey} instance or {@code null} in case of failure.
     */
    @Nullable
    private SecretKey generateSecretKey() {
        try {
            // Secret key is not created yet, or has been just removed. Try to create a new one.
            final KeyGenerator keyGenerator = KeyGenerator.getInstance(KeyProperties.KEY_ALGORITHM_AES, ANDROID_KEY_STORE);
            keyGenerator.init(getSecretKeySpecBuilder()
                     .build());
            return keyGenerator.generateKey();

        } catch (NoSuchAlgorithmException | NoSuchProviderException | InvalidAlgorithmParameterException e) {
            PA2Log.e("SymmetricKeyProvider: " + keyAlias + ": Failed to generate new key. Exception: " + e.getMessage());
            return null;
        }
    }

    /**
     * @return New instance of {@link KeyGenParameterSpec.Builder} prepared for the new key generation.
     */
    @NonNull
    private KeyGenParameterSpec.Builder getSecretKeySpecBuilder() {
        final KeyGenParameterSpec.Builder builder = new KeyGenParameterSpec.Builder(keyAlias,KeyProperties.PURPOSE_ENCRYPT | KeyProperties.PURPOSE_DECRYPT);
        onGenerateKey.configureBuilder(builder);
        return builder;
    }

    /**
     * Test whether the key is already in the keystore. The private implementation hides
     * all possible exceptions and returns {@code false} in case of keystore failure.
     *
     * @return {@code true} if keystore contains secret key with key alias.
     */
    private boolean hasSecretKey() {
        try {
            return keyStore.containsAlias(keyAlias);
        } catch (KeyStoreException e) {
            return false;
        }
    }

    /**
     * Remove secret key from keystore. The private implementation hides all keystore exceptions.
     */
    private void removeSecretKey() {
        try {
            keyStore.deleteEntry(keyAlias);
            PA2Log.d("SymmetricKeyProvider: " + keyAlias + ": Key has been deleted.");
        } catch (KeyStoreException e) {
            PA2Log.e("SymmetricKeyProvider: " + keyAlias + ": Failed to remove secret key. Exception: " + e.getMessage());
        }
    }
}
