/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. See the NOTICE file distributed
 * with this work for additional information regarding copyright ownership.
 * Licensed under the Camunda License 1.0. You may not use this file
 * except in compliance with the Camunda License 1.0.
 */
package io.camunda.zeebe.engine.processing.identity;

import io.camunda.zeebe.engine.processing.distribution.CommandDistributionBehavior;
import io.camunda.zeebe.engine.processing.streamprocessor.DistributedTypedRecordProcessor;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.StateWriter;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.TypedRejectionWriter;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.TypedResponseWriter;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.Writers;
import io.camunda.zeebe.engine.state.distribution.DistributionQueue;
import io.camunda.zeebe.engine.state.immutable.AuthorizationState;
import io.camunda.zeebe.engine.state.immutable.ProcessingState;
import io.camunda.zeebe.protocol.impl.record.value.authorization.AuthorizationRecord;
import io.camunda.zeebe.protocol.record.RejectionType;
import io.camunda.zeebe.protocol.record.intent.AuthorizationIntent;
import io.camunda.zeebe.stream.api.records.TypedRecord;
import io.camunda.zeebe.stream.api.state.KeyGenerator;

public final class AuthorizationAddPermissionProcessor
    implements DistributedTypedRecordProcessor<AuthorizationRecord> {

  private final KeyGenerator keyGenerator;
  private final AuthorizationState authorizationState;
  private final CommandDistributionBehavior distributionBehavior;
  private final StateWriter stateWriter;
  private final TypedResponseWriter responseWriter;
  private final TypedRejectionWriter rejectionWriter;

  public AuthorizationAddPermissionProcessor(
      final Writers writers,
      final KeyGenerator keyGenerator,
      final ProcessingState processingState,
      final CommandDistributionBehavior distributionBehavior) {
    this.keyGenerator = keyGenerator;
    authorizationState = processingState.getAuthorizationState();
    this.distributionBehavior = distributionBehavior;
    stateWriter = writers.state();
    responseWriter = writers.response();
    rejectionWriter = writers.rejection();
  }

  @Override
  public void processNewCommand(final TypedRecord<AuthorizationRecord> command) {
    final var authorizationRecord = command.getValue();

    final var ownerType =
        authorizationState
            .getOwnerType(authorizationRecord.getOwnerKey())
            .orElseThrow(() -> new OwnerNotFoundException(authorizationRecord.getOwnerKey()));
    authorizationRecord.setOwnerType(ownerType);

    final long key = keyGenerator.nextKey();
    stateWriter.appendFollowUpEvent(key, AuthorizationIntent.PERMISSION_ADDED, authorizationRecord);
    distributionBehavior
        .withKey(key)
        .inQueue(DistributionQueue.IDENTITY.getQueueId())
        .distribute(command);
    responseWriter.writeEventOnCommand(
        key, AuthorizationIntent.PERMISSION_ADDED, authorizationRecord, command);
  }

  @Override
  public void processDistributedCommand(final TypedRecord<AuthorizationRecord> command) {
    stateWriter.appendFollowUpEvent(
        command.getKey(), AuthorizationIntent.PERMISSION_ADDED, command.getValue());
    distributionBehavior.acknowledgeCommand(command);
  }

  @Override
  public ProcessingError tryHandleError(
      final TypedRecord<AuthorizationRecord> command, final Throwable error) {
    if (error instanceof final OwnerNotFoundException exception) {
      rejectionWriter.appendRejection(command, RejectionType.NOT_FOUND, exception.getMessage());
      responseWriter.writeRejectionOnCommand(
          command, RejectionType.NOT_FOUND, exception.getMessage());
      return ProcessingError.EXPECTED_ERROR;
    }

    return ProcessingError.UNEXPECTED_ERROR;
  }

  private static final class OwnerNotFoundException extends RuntimeException {

    public static final String OWNER_NOT_FOUND_MESSAGE =
        "Expected to find owner with key: '%d', but none was found";

    public OwnerNotFoundException(final long ownerKey) {
      super(OWNER_NOT_FOUND_MESSAGE.formatted(ownerKey));
    }
  }
}
