/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend;

import de.bwaldvogel.mongo.MongoBackend;
import de.bwaldvogel.mongo.MongoCollection;
import de.bwaldvogel.mongo.MongoDatabase;
import de.bwaldvogel.mongo.backend.Index;
import de.bwaldvogel.mongo.backend.LimitedList;
import de.bwaldvogel.mongo.backend.Utils;
import de.bwaldvogel.mongo.bson.Document;
import de.bwaldvogel.mongo.exception.MongoServerError;
import de.bwaldvogel.mongo.exception.MongoServerException;
import de.bwaldvogel.mongo.exception.MongoSilentServerException;
import de.bwaldvogel.mongo.exception.NoSuchCollectionException;
import de.bwaldvogel.mongo.exception.NoSuchCommandException;
import de.bwaldvogel.mongo.wire.message.MongoDelete;
import de.bwaldvogel.mongo.wire.message.MongoInsert;
import de.bwaldvogel.mongo.wire.message.MongoQuery;
import de.bwaldvogel.mongo.wire.message.MongoUpdate;
import io.netty.channel.Channel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractMongoDatabase<P>
implements MongoDatabase {
    private static final String NAMESPACES_COLLECTION_NAME = "system.namespaces";
    private static final String INDEXES_COLLECTION_NAME = "system.indexes";
    private static final Logger log = LoggerFactory.getLogger(AbstractMongoDatabase.class);
    protected final String databaseName;
    private final MongoBackend backend;
    private final Map<String, MongoCollection<P>> collections = new ConcurrentHashMap<String, MongoCollection<P>>();
    private final AtomicReference<MongoCollection<P>> indexes = new AtomicReference();
    private final Map<Channel, List<Document>> lastResults = new ConcurrentHashMap<Channel, List<Document>>();
    private MongoCollection<P> namespaces;

    protected AbstractMongoDatabase(String databaseName, MongoBackend backend) {
        this.databaseName = databaseName;
        this.backend = backend;
    }

    protected void initializeNamespacesAndIndexes() throws MongoServerException {
        this.namespaces = this.openOrCreateCollection(NAMESPACES_COLLECTION_NAME, "name");
        this.collections.put(this.namespaces.getCollectionName(), this.namespaces);
        if (this.namespaces.count() > 0) {
            for (Document namespace : this.namespaces.handleQuery(new Document(), 0, 0, null)) {
                String name = namespace.get("name").toString();
                log.debug("opening {}", (Object)name);
                String collectionName = this.extractCollectionNameFromNamespace(name);
                MongoCollection<P> collection = this.openOrCreateCollection(collectionName, "_id");
                this.collections.put(collectionName, collection);
                log.debug("opened collection '{}'", (Object)collectionName);
            }
            MongoCollection<P> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, null);
            this.indexes.set(indexCollection);
            for (Document indexDescription : indexCollection.handleQuery(new Document(), 0, 0, null)) {
                this.openOrCreateIndex(indexDescription);
            }
        }
    }

    @Override
    public final String getDatabaseName() {
        return this.databaseName;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.getDatabaseName() + ")";
    }

    private Document commandDropDatabase() throws MongoServerException {
        this.backend.dropDatabase(this.getDatabaseName());
        Document response = new Document("dropped", this.getDatabaseName());
        Utils.markOkay(response);
        return response;
    }

    @Override
    public Document handleCommand(Channel channel, String command, Document query) throws MongoServerException {
        if (command.equalsIgnoreCase("getlasterror")) {
            return this.commandGetLastError(channel, command, query);
        }
        if (command.equalsIgnoreCase("getpreverror")) {
            return this.commandGetPrevError(channel, command, query);
        }
        if (command.equalsIgnoreCase("reseterror")) {
            return this.commandResetError(channel, command, query);
        }
        this.clearLastStatus(channel);
        if (command.equalsIgnoreCase("insert")) {
            return this.commandInsert(channel, command, query);
        }
        if (command.equalsIgnoreCase("update")) {
            return this.commandUpdate(channel, command, query);
        }
        if (command.equalsIgnoreCase("delete")) {
            return this.commandDelete(channel, command, query);
        }
        if (command.equalsIgnoreCase("create")) {
            return this.commandCreate(command, query);
        }
        if (command.equalsIgnoreCase("createIndexes")) {
            return this.commandCreateIndexes(query);
        }
        if (command.equalsIgnoreCase("count")) {
            return this.commandCount(command, query);
        }
        if (command.equalsIgnoreCase("distinct")) {
            String collectionName = query.get(command).toString();
            MongoCollection<P> collection = this.resolveCollection(collectionName, true);
            return collection.handleDistinct(query);
        }
        if (command.equalsIgnoreCase("drop")) {
            return this.commandDrop(query);
        }
        if (command.equalsIgnoreCase("dropDatabase")) {
            return this.commandDropDatabase();
        }
        if (command.equalsIgnoreCase("dbstats")) {
            return this.commandDatabaseStats();
        }
        if (command.equalsIgnoreCase("collstats")) {
            String collectionName = query.get(command).toString();
            MongoCollection<P> collection = this.resolveCollection(collectionName, true);
            return collection.getStats();
        }
        if (command.equalsIgnoreCase("validate")) {
            String collectionName = query.get(command).toString();
            MongoCollection<P> collection = this.resolveCollection(collectionName, true);
            return collection.validate();
        }
        if (command.equalsIgnoreCase("findAndModify")) {
            String collectionName = query.get(command).toString();
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            return collection.findAndModify(query);
        }
        if (command.equalsIgnoreCase("listCollections")) {
            return this.listCollections();
        }
        if (command.equalsIgnoreCase("listIndexes")) {
            return this.listIndexes();
        }
        log.error("unknown query: {}", (Object)query);
        throw new NoSuchCommandException(command);
    }

    private Document listCollections() throws MongoServerException {
        Document cursor = new Document();
        cursor.put("id", (Object)0L);
        cursor.put("ns", (Object)(this.getDatabaseName() + ".$cmd.listCollections"));
        ArrayList<Document> firstBatch = new ArrayList<Document>();
        for (Document collection : this.namespaces.handleQuery(new Document(), 0, 0, null)) {
            Document collectionDescription = new Document();
            Document collectionOptions = new Document();
            String namespace = (String)collection.get("name");
            String collectionName = this.extractCollectionNameFromNamespace(namespace);
            collectionDescription.put("name", (Object)collectionName);
            collectionDescription.put("options", (Object)collectionOptions);
            firstBatch.add(collectionDescription);
        }
        cursor.put("firstBatch", (Object)firstBatch);
        Document response = new Document();
        response.put("cursor", (Object)cursor);
        Utils.markOkay(response);
        return response;
    }

    private Document listIndexes() throws MongoServerException {
        MongoCollection<P> indexes = this.resolveCollection(INDEXES_COLLECTION_NAME, true);
        Document cursor = new Document();
        cursor.put("id", (Object)0L);
        cursor.put("ns", (Object)(this.getDatabaseName() + ".$cmd.listIndexes"));
        ArrayList<Document> firstBatch = new ArrayList<Document>();
        for (Document description : indexes.handleQuery(new Document(), 0, 0, null)) {
            firstBatch.add(description);
        }
        cursor.put("firstBatch", (Object)firstBatch);
        Document response = new Document();
        response.put("cursor", (Object)cursor);
        Utils.markOkay(response);
        return response;
    }

    private synchronized MongoCollection<P> resolveOrCreateCollection(String collectionName) throws MongoServerException {
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            return collection;
        }
        return this.createCollection(collectionName);
    }

    private Document commandInsert(Channel channel, String command, Document query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List documents = (List)query.get("documents");
        ArrayList<Document> writeErrors = new ArrayList<Document>();
        int n = 0;
        for (Document document : documents) {
            try {
                this.insertDocuments(channel, collectionName, Collections.singletonList(document));
                ++n;
            }
            catch (MongoServerError e) {
                Document error = new Document();
                error.put("index", (Object)n);
                error.put("errmsg", (Object)e.getMessage());
                error.put("code", (Object)e.getCode());
                error.putIfNotNull("codeName", e.getCodeName());
                writeErrors.add(error);
            }
        }
        Document result = new Document();
        result.put("n", (Object)n);
        if (!writeErrors.isEmpty()) {
            result.put("writeErrors", (Object)writeErrors);
        }
        Utils.markOkay(result);
        return result;
    }

    private Document commandUpdate(Channel channel, String command, Document query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List updates = (List)query.get("updates");
        int nMatched = 0;
        int nModified = 0;
        ArrayList<Document> upserts = new ArrayList<Document>();
        for (Document updateObj : updates) {
            boolean upsert;
            boolean multi;
            Document update;
            Document selector = (Document)updateObj.get("q");
            Document result = this.updateDocuments(channel, collectionName, selector, update = (Document)updateObj.get("u"), multi = Utils.isTrue(updateObj.get("multi")), upsert = Utils.isTrue(updateObj.get("upsert")));
            if (result.containsKey("upserted")) {
                Object id = result.get("upserted");
                Document upserted = new Document("index", upserts.size());
                upserted.put("_id", id);
                upserts.add(upserted);
            }
            nMatched += ((Integer)result.get("n")).intValue();
            nModified += ((Integer)result.get("nModified")).intValue();
        }
        Document response = new Document();
        response.put("n", (Object)nMatched);
        response.put("nModified", (Object)nModified);
        if (!upserts.isEmpty()) {
            response.put("upserted", (Object)upserts);
        }
        Utils.markOkay(response);
        this.putLastResult(channel, response);
        return response;
    }

    private Document commandDelete(Channel channel, String command, Document query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List deletes = (List)query.get("deletes");
        int n = 0;
        for (Document delete : deletes) {
            Document selector = (Document)delete.get("q");
            int limit = ((Number)delete.get("limit")).intValue();
            Document result = this.deleteDocuments(channel, collectionName, selector, limit);
            Integer resultNumber = (Integer)result.get("n");
            n += resultNumber.intValue();
        }
        Document response = new Document("n", n);
        Utils.markOkay(response);
        return response;
    }

    private Document commandCreate(String command, Document query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isCapped = Utils.isTrue(query.get("capped"));
        if (isCapped) {
            throw new MongoServerException("Creating capped collections is not yet implemented");
        }
        Object autoIndexId = query.get("autoIndexId");
        if (autoIndexId != null && !Utils.isTrue(autoIndexId)) {
            throw new MongoServerException("Disabling autoIndexId is not yet implemented");
        }
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            throw new MongoServerError(48, "collection already exists");
        }
        this.createCollection(collectionName);
        Document response = new Document();
        Utils.markOkay(response);
        return response;
    }

    private Document commandCreateIndexes(Document query) throws MongoServerException {
        int indexesBefore = this.countIndexes();
        Collection indexDescriptions = (Collection)query.get("indexes");
        for (Document indexDescription : indexDescriptions) {
            this.addIndex(indexDescription);
        }
        int indexesAfter = this.countIndexes();
        Document response = new Document();
        response.put("numIndexesBefore", (Object)indexesBefore);
        response.put("numIndexesAfter", (Object)indexesAfter);
        Utils.markOkay(response);
        return response;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int countIndexes() throws MongoServerException {
        MongoCollection<P> indexesCollection;
        AtomicReference<MongoCollection<P>> atomicReference = this.indexes;
        synchronized (atomicReference) {
            indexesCollection = this.indexes.get();
        }
        if (indexesCollection == null) {
            return 0;
        }
        return indexesCollection.count();
    }

    private Document commandDatabaseStats() throws MongoServerException {
        Document response = new Document("db", this.getDatabaseName());
        response.put("collections", (Object)this.namespaces.count());
        long storageSize = this.getStorageSize();
        long fileSize = this.getFileSize();
        long indexSize = 0L;
        long objects = 0L;
        long dataSize = 0L;
        double averageObjectSize = 0.0;
        for (MongoCollection<P> collection : this.collections.values()) {
            Document stats = collection.getStats();
            objects += ((Number)stats.get("count")).longValue();
            dataSize += ((Number)stats.get("size")).longValue();
            Document indexSizes = (Document)stats.get("indexSize");
            for (String indexName : indexSizes.keySet()) {
                indexSize += ((Number)indexSizes.get(indexName)).longValue();
            }
        }
        if (objects > 0L) {
            averageObjectSize = (double)dataSize / (double)objects;
        }
        response.put("objects", (Object)objects);
        response.put("avgObjSize", (Object)averageObjectSize);
        response.put("dataSize", (Object)dataSize);
        response.put("storageSize", (Object)storageSize);
        response.put("numExtents", (Object)0);
        response.put("indexes", (Object)this.countIndexes());
        response.put("indexSize", (Object)indexSize);
        response.put("fileSize", (Object)fileSize);
        response.put("nsSizeMB", (Object)0);
        Utils.markOkay(response);
        return response;
    }

    protected abstract long getFileSize();

    protected abstract long getStorageSize();

    private Document commandDrop(Document query) throws MongoServerException {
        String collectionName = query.get("drop").toString();
        MongoCollection<P> collection = this.collections.remove(collectionName);
        if (collection == null) {
            throw new MongoSilentServerException("ns not found");
        }
        Document response = new Document();
        this.namespaces.removeDocument(new Document("name", collection.getFullName()));
        response.put("nIndexesWas", (Object)collection.getNumIndexes());
        response.put("ns", (Object)collection.getFullName());
        Utils.markOkay(response);
        return response;
    }

    private Document commandGetLastError(Channel channel, String command, Document query) throws MongoServerException {
        Iterator<String> it = query.keySet().iterator();
        String cmd = it.next();
        if (!cmd.equals(command)) {
            throw new IllegalStateException();
        }
        if (it.hasNext()) {
            String subCommand;
            switch (subCommand = it.next()) {
                case "w": {
                    break;
                }
                case "fsync": {
                    break;
                }
                default: {
                    throw new MongoServerException("unknown subcommand: " + subCommand);
                }
            }
        }
        List<Document> results = this.lastResults.get(channel);
        Document result = null;
        if (results != null && !results.isEmpty()) {
            result = results.get(results.size() - 1);
            if (result == null) {
                result = new Document();
            }
        } else {
            result = new Document();
            result.put("err", (Object)null);
        }
        Utils.markOkay(result);
        return result;
    }

    private Document commandGetPrevError(Channel channel, String command, Document query) {
        List<Document> results = this.lastResults.get(channel);
        if (results != null) {
            for (int i = 1; i < results.size(); ++i) {
                Document result = results.get(results.size() - i);
                if (result == null) continue;
                boolean isRelevant = false;
                if (result.get("err") != null) {
                    isRelevant = true;
                } else if (((Number)result.get("n")).intValue() > 0) {
                    isRelevant = true;
                }
                if (!isRelevant) continue;
                result.put("nPrev", (Object)i);
                return result;
            }
        }
        Document result = new Document();
        result.put("nPrev", (Object)-1);
        Utils.markOkay(result);
        return result;
    }

    private Document commandResetError(Channel channel, String command, Document query) {
        List<Document> results = this.lastResults.get(channel);
        if (results != null) {
            results.clear();
        }
        Document result = new Document();
        Utils.markOkay(result);
        return result;
    }

    private Document commandCount(String command, Document query) throws MongoServerException {
        String collection = query.get(command).toString();
        Document response = new Document();
        MongoCollection<P> coll = this.collections.get(collection);
        if (coll == null) {
            response.put("missing", (Object)Boolean.TRUE);
            response.put("n", (Object)0);
        } else {
            Document queryObject = (Document)query.get("query");
            int limit = this.getOptionalNumber(query, "limit", -1);
            int skip = this.getOptionalNumber(query, "skip", 0);
            response.put("n", (Object)coll.count(queryObject, skip, limit));
        }
        Utils.markOkay(response);
        return response;
    }

    private int getOptionalNumber(Document query, String fieldName, int defaultValue) {
        Number limitNumber = (Number)query.get(fieldName);
        return limitNumber != null ? limitNumber.intValue() : defaultValue;
    }

    @Override
    public Iterable<Document> handleQuery(MongoQuery query) throws MongoServerException {
        this.clearLastStatus(query.getChannel());
        String collectionName = query.getCollectionName();
        MongoCollection<P> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return Collections.emptyList();
        }
        int numSkip = query.getNumberToSkip();
        int numReturn = query.getNumberToReturn();
        Document fieldSelector = query.getReturnFieldSelector();
        return collection.handleQuery(query.getQuery(), numSkip, numReturn, fieldSelector);
    }

    @Override
    public void handleClose(Channel channel) {
        this.lastResults.remove(channel);
    }

    private synchronized void clearLastStatus(Channel channel) {
        List<Document> results = this.lastResults.get(channel);
        if (results == null) {
            results = new LimitedList<Document>(10);
            this.lastResults.put(channel, results);
        }
        results.add(null);
    }

    @Override
    public void handleInsert(MongoInsert insert) throws MongoServerException {
        Channel channel = insert.getChannel();
        String collectionName = insert.getCollectionName();
        List<Document> documents = insert.getDocuments();
        if (collectionName.equals(INDEXES_COLLECTION_NAME)) {
            for (Document indexDescription : documents) {
                this.addIndex(indexDescription);
            }
        } else {
            try {
                this.insertDocuments(channel, collectionName, documents);
            }
            catch (MongoServerException e) {
                log.error("failed to insert {}", (Object)insert, (Object)e);
            }
        }
    }

    public synchronized MongoCollection<P> resolveCollection(String collectionName, boolean throwIfNotFound) throws MongoServerException {
        this.checkCollectionName(collectionName);
        MongoCollection<P> collection = this.collections.get(collectionName);
        if (collection == null && throwIfNotFound) {
            throw new NoSuchCollectionException(collectionName);
        }
        return collection;
    }

    private void checkCollectionName(String collectionName) throws MongoServerException {
        if (collectionName.length() > 128) {
            throw new MongoServerError(10080, "ns name too long, max size is 128");
        }
        if (collectionName.isEmpty()) {
            throw new MongoServerError(16256, "Invalid ns [" + collectionName + "]");
        }
    }

    @Override
    public boolean isEmpty() {
        return this.collections.isEmpty();
    }

    private void addNamespace(MongoCollection<P> collection) throws MongoServerException {
        this.collections.put(collection.getCollectionName(), collection);
        this.namespaces.addDocument(new Document("name", collection.getFullName()));
    }

    @Override
    public void handleDelete(MongoDelete delete) throws MongoServerException {
        Channel channel = delete.getChannel();
        String collectionName = delete.getCollectionName();
        Document selector = delete.getSelector();
        int limit = delete.isSingleRemove() ? 1 : Integer.MAX_VALUE;
        try {
            this.deleteDocuments(channel, collectionName, selector, limit);
        }
        catch (MongoServerException e) {
            log.error("failed to delete {}", (Object)delete, (Object)e);
        }
    }

    @Override
    public void handleUpdate(MongoUpdate updateCommand) throws MongoServerException {
        Channel channel = updateCommand.getChannel();
        String collectionName = updateCommand.getCollectionName();
        Document selector = updateCommand.getSelector();
        Document update = updateCommand.getUpdate();
        boolean multi = updateCommand.isMulti();
        boolean upsert = updateCommand.isUpsert();
        try {
            Document result = this.updateDocuments(channel, collectionName, selector, update, multi, upsert);
            this.putLastResult(channel, result);
        }
        catch (MongoServerException e) {
            log.error("failed to update {}", (Object)updateCommand, (Object)e);
        }
    }

    private void addIndex(Document indexDescription) throws MongoServerException {
        this.openOrCreateIndex(indexDescription);
        this.getOrCreateIndexesCollection().addDocument(indexDescription);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private MongoCollection<P> getOrCreateIndexesCollection() throws MongoServerException {
        AtomicReference<MongoCollection<P>> atomicReference = this.indexes;
        synchronized (atomicReference) {
            if (this.indexes.get() == null) {
                MongoCollection<P> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, null);
                this.addNamespace(indexCollection);
                this.indexes.set(indexCollection);
            }
            return this.indexes.get();
        }
    }

    private String extractCollectionNameFromNamespace(String namespace) {
        if (!namespace.startsWith(this.databaseName)) {
            throw new IllegalArgumentException();
        }
        return namespace.substring(this.databaseName.length() + 1);
    }

    private void openOrCreateIndex(Document indexDescription) throws MongoServerException {
        String ns = indexDescription.get("ns").toString();
        String collectionName = this.extractCollectionNameFromNamespace(ns);
        MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
        Document key = (Document)indexDescription.get("key");
        if (key.keySet().equals(Collections.singleton("_id"))) {
            boolean ascending = Utils.normalizeValue(key.get("_id")).equals(1.0);
            collection.addIndex(this.openOrCreateUniqueIndex(collectionName, "_id", ascending));
            log.info("adding unique _id index for collection {}", (Object)collectionName);
        } else if (Utils.isTrue(indexDescription.get("unique"))) {
            if (key.keySet().size() != 1) {
                throw new MongoServerException("Compound unique indices are not yet implemented");
            }
            log.info("adding unique index {} for collection {}", key.keySet(), (Object)collectionName);
            String field = key.keySet().iterator().next();
            boolean ascending = Utils.normalizeValue(key.get(field)).equals(1.0);
            collection.addIndex(this.openOrCreateUniqueIndex(collectionName, field, ascending));
        } else {
            log.warn("adding non-unique non-id index with key {} is not yet implemented", (Object)key);
        }
    }

    protected abstract Index<P> openOrCreateUniqueIndex(String var1, String var2, boolean var3) throws MongoServerException;

    private void insertDocuments(Channel channel, String collectionName, List<Document> documents) throws MongoServerException {
        this.clearLastStatus(channel);
        try {
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(16459, "attempt to insert in system namespace");
            }
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            int n = collection.insertDocuments(documents);
            assert (n == documents.size());
            Document result = new Document("n", n);
            this.putLastResult(channel, result);
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    private Document deleteDocuments(Channel channel, String collectionName, Document selector, int limit) throws MongoServerException {
        this.clearLastStatus(channel);
        try {
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(12050, "cannot delete from system namespace");
            }
            MongoCollection<P> collection = this.resolveCollection(collectionName, false);
            int n = collection == null ? 0 : collection.deleteDocuments(selector, limit);
            Document result = new Document("n", n);
            this.putLastResult(channel, result);
            return result;
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    private Document updateDocuments(Channel channel, String collectionName, Document selector, Document update, boolean multi, boolean upsert) throws MongoServerException {
        this.clearLastStatus(channel);
        try {
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(10156, "cannot update system collection");
            }
            MongoCollection<P> collection = this.resolveOrCreateCollection(collectionName);
            return collection.updateDocuments(selector, update, multi, upsert);
        }
        catch (MongoServerException e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    private void putLastError(Channel channel, MongoServerException ex) {
        Document error = new Document();
        if (ex instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)ex;
            error.put("err", (Object)err.getMessage());
            error.put("code", (Object)err.getCode());
            error.putIfNotNull("codeName", err.getCodeName());
        } else {
            error.put("err", (Object)ex.getMessage());
        }
        error.put("connectionId", (Object)channel.id().asShortText());
        this.putLastResult(channel, error);
    }

    private synchronized void putLastResult(Channel channel, Document result) {
        List<Document> results = this.lastResults.get(channel);
        Document last = results.get(results.size() - 1);
        if (last != null) {
            throw new IllegalStateException("last result already set: " + last);
        }
        results.set(results.size() - 1, result);
    }

    private MongoCollection<P> createCollection(String collectionName) throws MongoServerException {
        this.checkCollectionName(collectionName);
        if (collectionName.contains("$")) {
            throw new MongoServerError(10093, "cannot insert into reserved $ collection");
        }
        MongoCollection<P> collection = this.openOrCreateCollection(collectionName, "_id");
        this.addNamespace(collection);
        Document indexDescription = new Document();
        indexDescription.put("name", (Object)"_id_");
        indexDescription.put("ns", (Object)collection.getFullName());
        indexDescription.put("key", (Object)new Document("_id", 1));
        this.addIndex(indexDescription);
        log.info("created collection {}", (Object)collection.getFullName());
        return collection;
    }

    protected abstract MongoCollection<P> openOrCreateCollection(String var1, String var2) throws MongoServerException;

    @Override
    public void drop() throws MongoServerException {
        log.debug("dropping {}", (Object)this);
        for (String collectionName : this.collections.keySet()) {
            this.dropCollection(collectionName);
        }
    }

    @Override
    public void dropCollection(String collectionName) throws MongoServerException {
        this.unregisterCollection(collectionName);
    }

    public MongoCollection<P> unregisterCollection(String collectionName) throws MongoServerException {
        MongoCollection<P> removedCollection = this.collections.remove(collectionName);
        this.namespaces.deleteDocuments(new Document("name", removedCollection.getFullName()), 1);
        return removedCollection;
    }

    @Override
    public void moveCollection(MongoDatabase oldDatabase, MongoCollection<?> collection, String newCollectionName) throws MongoServerException {
        oldDatabase.unregisterCollection(collection.getCollectionName());
        collection.renameTo(this.getDatabaseName(), newCollectionName);
        MongoCollection<?> newCollection = collection;
        this.collections.put(newCollectionName, newCollection);
        ArrayList<Document> newDocuments = new ArrayList<Document>();
        newDocuments.add(new Document("name", collection.getFullName()));
        this.namespaces.insertDocuments(newDocuments);
    }
}

