/*
    This file is part of the iText (R) project.
    Copyright (c) 1998-2025 Apryse Group NV
    Authors: Apryse Software.

    This program is offered under a commercial and under the AGPL license.
    For commercial licensing, contact us at https://itextpdf.com/sales.  For AGPL licensing, see below.

    AGPL licensing:
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package com.itextpdf.io.font;

import com.itextpdf.commons.datastructures.Tuple2;
import com.itextpdf.io.exceptions.IOException;
import com.itextpdf.io.exceptions.IoExceptionMessageConstant;
import com.itextpdf.io.font.constants.TrueTypeCodePages;
import com.itextpdf.io.font.otf.Glyph;
import com.itextpdf.io.font.otf.GlyphPositioningTableReader;
import com.itextpdf.io.font.otf.GlyphSubstitutionTableReader;
import com.itextpdf.io.font.otf.OpenTypeGdefTableReader;
import com.itextpdf.io.util.IntHashtable;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;

public class TrueTypeFont extends FontProgram {


    private OpenTypeParser fontParser;

    protected int[][] bBoxes;

    protected boolean isVertical;

    private GlyphSubstitutionTableReader gsubTable;
    private GlyphPositioningTableReader gposTable;
    private OpenTypeGdefTableReader gdefTable;

    /**
     * The map containing the kerning information. It represents the content of
     * table 'kern'. The key is an <CODE>Integer</CODE> where the top 16 bits
     * are the glyph number for the first character and the lower 16 bits are the
     * glyph number for the second character. The value is the amount of kerning in
     * normalized 1000 units as an <CODE>Integer</CODE>. This value is usually negative.
     */
    protected IntHashtable kerning = new IntHashtable();

    private byte[] fontStreamBytes;

    private TrueTypeFont(OpenTypeParser fontParser) throws java.io.IOException {
        this.fontParser = fontParser;
        this.fontParser.loadTables(true);
        initializeFontProperties();
    }

    protected TrueTypeFont() {
        fontNames = new FontNames();
    }

    public TrueTypeFont(String path) throws java.io.IOException {
        this(new OpenTypeParser(path));
    }

    public TrueTypeFont(byte[] ttf) throws java.io.IOException {
        this(new OpenTypeParser(ttf));
    }

    public TrueTypeFont(byte[] ttf, boolean isLenientMode) throws java.io.IOException {
        this(new OpenTypeParser(ttf, isLenientMode));
    }

    TrueTypeFont(String ttcPath, int ttcIndex) throws java.io.IOException {
        this(new OpenTypeParser(ttcPath, ttcIndex));
    }

    TrueTypeFont(byte[] ttc, int ttcIndex) throws java.io.IOException {
        this(new OpenTypeParser(ttc, ttcIndex));
    }

    @Override
    public boolean hasKernPairs() {
        return kerning.size() > 0;
    }

    /**
     * Gets the kerning between two glyphs.
     *
     * @param first the first glyph
     * @param second the second glyph
     *
     * @return the kerning to be applied
     */
    @Override
    public int getKerning(Glyph first, Glyph second) {
        if (first == null || second == null) {
            return 0;
        }
        return kerning.get((first.getCode() << 16) + second.getCode());
    }

    public boolean isCff() {
        return fontParser.isCff();
    }

    public Map<Integer, int[]> getActiveCmap() {
        OpenTypeParser.CmapTable cmaps = fontParser.getCmapTable();
        if (cmaps.cmap310 != null) {
            return cmaps.cmap310;
        } else if (!cmaps.fontSpecific && cmaps.cmap31 != null) {
            return cmaps.cmap31;
        } else if (cmaps.fontSpecific && cmaps.cmap10 != null) {
            return cmaps.cmap10;
        } else if (cmaps.cmap31 != null) {
            return cmaps.cmap31;
        } else {
            return cmaps.cmap10;
        }
    }

    public byte[] getFontStreamBytes() {
        if (fontStreamBytes != null)
            return fontStreamBytes;
        try {
            if (fontParser.isCff()) {
                fontStreamBytes = fontParser.readCffFont();
            } else {
                fontStreamBytes = fontParser.getFullFont();
            }
        } catch (java.io.IOException e) {
            fontStreamBytes = null;
            throw new IOException(IoExceptionMessageConstant.IO_EXCEPTION, e);
        }
        return fontStreamBytes;
    }

    @Override
    public int getPdfFontFlags() {
        int flags = 0;
        if (fontMetrics.isFixedPitch()) {
            flags |= 1;
        }
        flags |= isFontSpecific() ? 4 : 32;
        if (fontNames.isItalic()) {
            flags |= 64;
        }
        if (fontNames.isBold() || fontNames.getFontWeight() > 500) {
            flags |= 262144;
        }
        return flags;
    }

    /**
     * The offset from the start of the file to the table directory.
     * It is 0 for TTF and may vary for TTC depending on the chosen font.
     *
     * @return directory Offset
     */
    public int getDirectoryOffset() {
        return fontParser.directoryOffset;
    }

    public GlyphSubstitutionTableReader getGsubTable() {
        return gsubTable;
    }

    public GlyphPositioningTableReader getGposTable() {
        return gposTable;
    }

    public OpenTypeGdefTableReader getGdefTable() {
        return gdefTable;
    }

    /**
     * Gets subset based on the passed glyphs.
     *
     * @param glyphs the glyphs to subset the font
     * @param subsetTables whether subset tables (remove `name` and `post` tables) or not. It's used in case of ttc
     *                       (true type collection) font where single "full" font is needed. Despite the value of that
     *                       flag, only used glyphs will be left in the font
     *
     * @return the subset font
     */
    public byte[] getSubset(Set<Integer> glyphs, boolean subsetTables) {
        return subset(glyphs, subsetTables).getSecond();
    }

    /**
     * Gets subset and a number of glyphs in it based on the passed glyphs.
     *
     * <p>
     * The number of glyphs in a subset is not just glyphs.size() here. It's the biggest glyph id + 1 (for glyph 0).
     * It also may include possible composite glyphs.
     *
     * @param glyphs the glyphs to subset the font
     * @param subsetTables whether subset tables (remove `name` and `post` tables) or not. It's used in case of ttc
     *                       (true type collection) font where single "full" font is needed. Despite the value of that
     *                       flag, only used glyphs will be left in the font
     *
     * @return the subset of the font and the number of glyphs in it
     */
    public Tuple2<Integer, byte[]> subset(Set<Integer> glyphs, boolean subsetTables) {
        try {
            return fontParser.getSubset(glyphs, subsetTables);
        } catch (java.io.IOException e) {
            throw new IOException(IoExceptionMessageConstant.IO_EXCEPTION, e);
        }
    }

    /**
     * Merges the passed font into one. Used glyphs per each font are applied to subset the merged font.
     *
     * @param toMerge the fonts to merge with used glyphs per each font
     * @param fontName the name of fonts to merge
     *
     * @return the raw data of merged font
     *
     * @deprecated in favour of {@link #merge(Map, String, boolean)}
     */
    @Deprecated
    public static byte[] merge(Map<TrueTypeFont, Set<Integer>> toMerge, String fontName) {
        return merge(toMerge, fontName, true);
    }

    /**
     * Merges the passed font into one. Used glyphs per each font are applied to subset the merged font.
     *
     * @param toMerge the fonts to merge with used glyphs per each font
     * @param fontName the name of fonts to merge
     * @param isCmapCheckRequired the flag which specifies whether 'cmap' table should be checked while merging or not
     *
     * @return the raw data of merged font
     */
    public static byte[] merge(Map<TrueTypeFont, Set<Integer>> toMerge, String fontName, boolean isCmapCheckRequired) {
        try {
            Map<OpenTypeParser, Set<Integer>> toMergeWithParsers = new LinkedHashMap<>();
            for (Map.Entry<TrueTypeFont, Set<Integer>> entry : toMerge.entrySet()) {
                toMergeWithParsers.put(entry.getKey().fontParser, entry.getValue());
            }
            TrueTypeFontMerger trueTypeFontMerger = new TrueTypeFontMerger(fontName, toMergeWithParsers,
                    isCmapCheckRequired);
            return trueTypeFontMerger.process().getSecond();
        } catch (java.io.IOException e) {
            throw new IOException(IoExceptionMessageConstant.IO_EXCEPTION, e);
        }
    }

    /**
     * Maps a set of glyph CIDs (as used in PDF file) to corresponding GID values
     * (as a glyph primary identifier in the font file).
     * This call is only meaningful for fonts that return true for {@link #isCff()}.
     * For other types of fonts, GID and CID are always the same, so that call would essentially
     * return a set of the same values.
     *
     * @param glyphs a set of glyph CIDs
     *
     * @return a set of glyph ids corresponding to the passed glyph CIDs
     */
    public Set<Integer> mapGlyphsCidsToGids(Set<Integer> glyphs) {
        return glyphs.stream()
                .map((Integer i) -> {
                    Glyph usedGlyph = getGlyphByCode(i);
                    if (usedGlyph instanceof GidAwareGlyph) {
                        return ((GidAwareGlyph) usedGlyph).getGid();
                    }
                    return i;
                })
                .collect(Collectors.toSet());
    }

    /**
     * Checks whether current {@link TrueTypeFont} program contains the “cmap” subtable
     * with provided platform ID and encoding ID.
     *
     * @param platformID platform ID
     * @param encodingID encoding ID
     *
     * @return {@code true} if “cmap” subtable with provided platform ID and encoding ID is present in the font program,
     * {@code false} otherwise
     */
    public boolean isCmapPresent(int platformID, int encodingID) {
        OpenTypeParser.CmapTable cmaps = fontParser.getCmapTable();
        if (cmaps == null) {
            return false;
        }
        return cmaps.cmapEncodings.contains(new Tuple2<>(platformID, encodingID));
    }

    /**
     * Gets the number of the “cmap” subtables for the current {@link TrueTypeFont} program.
     *
     * @return the number of the “cmap” subtables
     */
    public int getNumberOfCmaps() {
        OpenTypeParser.CmapTable cmaps = fontParser.getCmapTable();
        if (cmaps == null) {
            return 0;
        }
        return cmaps.cmapEncodings.size();
    }

    protected void readGdefTable() throws java.io.IOException {
        int[] gdef = fontParser.tables.get("GDEF");
        if (gdef != null) {
            gdefTable = new OpenTypeGdefTableReader(fontParser.raf, gdef[0]);
        } else {
            gdefTable = new OpenTypeGdefTableReader(fontParser.raf, 0);
        }
        gdefTable.readTable();
    }

    protected void readGsubTable() throws java.io.IOException {
        int[] gsub = fontParser.tables.get("GSUB");
        if (gsub != null) {
            gsubTable = new GlyphSubstitutionTableReader(fontParser.raf, gsub[0], gdefTable, codeToGlyph, fontMetrics.getUnitsPerEm());
        }
    }

    protected void readGposTable() throws java.io.IOException {
        int[] gpos = fontParser.tables.get("GPOS");
        if (gpos != null) {
            gposTable = new GlyphPositioningTableReader(fontParser.raf, gpos[0], gdefTable, codeToGlyph,
                    fontMetrics.getUnitsPerEm());
        }
    }

    private void initializeFontProperties() throws java.io.IOException {
        // initialize sfnt tables
        OpenTypeParser.HeaderTable head = fontParser.getHeadTable();
        OpenTypeParser.HorizontalHeader hhea = fontParser.getHheaTable();
        OpenTypeParser.WindowsMetrics os_2 = fontParser.getOs_2Table();
        OpenTypeParser.PostTable post = fontParser.getPostTable();
        isFontSpecific = fontParser.getCmapTable().fontSpecific;
        kerning = fontParser.readKerning(head.unitsPerEm);
        bBoxes = fontParser.readBbox(head.unitsPerEm);

        // font names group
        fontNames = fontParser.getFontNames();

        // font metrics group
        fontMetrics.setUnitsPerEm(head.unitsPerEm);
        fontMetrics.updateBbox(head.xMin, head.yMin, head.xMax, head.yMax);
        fontMetrics.setNumberOfGlyphs(fontParser.readNumGlyphs());
        fontMetrics.setGlyphWidths(fontParser.getGlyphWidthsByIndex());
        fontMetrics.setTypoAscender(os_2.sTypoAscender);
        fontMetrics.setTypoDescender(os_2.sTypoDescender);
        fontMetrics.setCapHeight(os_2.sCapHeight);
        fontMetrics.setXHeight(os_2.sxHeight);
        fontMetrics.setItalicAngle(post.italicAngle);
        fontMetrics.setAscender(hhea.Ascender);
        fontMetrics.setDescender(hhea.Descender);
        fontMetrics.setLineGap(hhea.LineGap);
        fontMetrics.setWinAscender(os_2.usWinAscent);
        fontMetrics.setWinDescender(os_2.usWinDescent);
        fontMetrics.setAdvanceWidthMax(hhea.advanceWidthMax);
        fontMetrics.setUnderlinePosition((post.underlinePosition - post.underlineThickness) / 2);
        fontMetrics.setUnderlineThickness(post.underlineThickness);
        fontMetrics.setStrikeoutPosition(os_2.yStrikeoutPosition);
        fontMetrics.setStrikeoutSize(os_2.yStrikeoutSize);
        fontMetrics.setSubscriptOffset(-os_2.ySubscriptYOffset);
        fontMetrics.setSubscriptSize(os_2.ySubscriptYSize);
        fontMetrics.setSuperscriptOffset(os_2.ySuperscriptYOffset);
        fontMetrics.setSuperscriptSize(os_2.ySuperscriptYSize);
        fontMetrics.setIsFixedPitch(post.isFixedPitch);

        // font identification group
        String[][] ttfVersion = fontNames.getNames(5);
        if (ttfVersion != null) {
            fontIdentification.setTtfVersion(ttfVersion[0][3]);
        }
        String[][] ttfUniqueId = fontNames.getNames(3);
        if (ttfUniqueId != null) {
            fontIdentification.setTtfVersion(ttfUniqueId[0][3]);
        }

        byte[] pdfPanose = new byte[12];
        pdfPanose[1] = (byte) (os_2.sFamilyClass);
        pdfPanose[0] = (byte) (os_2.sFamilyClass >> 8);
        System.arraycopy(os_2.panose, 0, pdfPanose, 2, 10);
        fontIdentification.setPanose(pdfPanose);

        Map<Integer, int[]> cmap = getActiveCmap();
        int[] glyphWidths = fontParser.getGlyphWidthsByIndex();
        int numOfGlyphs = fontMetrics.getNumberOfGlyphs();
        unicodeToGlyph = new LinkedHashMap<>(cmap.size());
        codeToGlyph = new LinkedHashMap<>(numOfGlyphs);
        avgWidth = 0;
        CFFFontSubset cffFontSubset = null;
        if (isCff()) {
            cffFontSubset = new CFFFontSubset(getFontStreamBytes());
        }
        for (int charCode : cmap.keySet()) {
            int index = cmap.get(charCode)[0];
            if (index >= numOfGlyphs) {
                // It seems to be a valid case. If the font is subsetted but cmap table is not, it's a valid case
                continue;
            }
            int cid;
            Glyph glyph;
            int[] glyphBBox = (bBoxes != null && index < bBoxes.length) ? bBoxes[index] : null;
            if (cffFontSubset != null && cffFontSubset.isCID()) {
                cid = cffFontSubset.getCidForGlyphId(index);
                GidAwareGlyph cffGlyph = new GidAwareGlyph(cid, glyphWidths[index], charCode, glyphBBox);
                cffGlyph.setGid(index);
                glyph = cffGlyph;
            } else {
                cid = index;
                glyph = new Glyph(cid, glyphWidths[index], charCode, glyphBBox);
            }

            unicodeToGlyph.put(charCode, glyph);
            // This is done on purpose to keep the mapping to glyphs with smaller unicode values, in contrast with
            // larger values which often represent different forms of other characters.
            if (!codeToGlyph.containsKey(cid)) {
                codeToGlyph.put(cid, glyph);
            }
            avgWidth += glyph.getWidth();
        }
        fixSpaceIssue();
        for (int index = 0; index < glyphWidths.length; index++) {
            if (codeToGlyph.containsKey(index)) {
                continue;
            }
            Glyph glyph = new Glyph(index, glyphWidths[index], -1);
            codeToGlyph.put(index, glyph);
            avgWidth += glyph.getWidth();
        }

        if (!codeToGlyph.isEmpty()) {
            avgWidth /= codeToGlyph.size();
        }

        readGdefTable();
        readGsubTable();
        readGposTable();

        isVertical = false;
    }

    /**
     * Gets the code pages supported by the font.
     *
     * @return the code pages supported by the font
     */
    public String[] getCodePagesSupported() {
        long cp = ((long) fontParser.getOs_2Table().ulCodePageRange2 << 32) + (fontParser.getOs_2Table().ulCodePageRange1 & 0xffffffffL);
        int count = 0;
        long bit = 1;
        for (int k = 0; k < 64; ++k) {
            if ((cp & bit) != 0 && TrueTypeCodePages.get(k) != null)
                ++count;
            bit <<= 1;
        }
        String[] ret = new String[count];
        count = 0;
        bit = 1;
        for (int k = 0; k < 64; ++k) {
            if ((cp & bit) != 0 && TrueTypeCodePages.get(k) != null)
                ret[count++] = TrueTypeCodePages.get(k);
            bit <<= 1;
        }
        return ret;
    }

    @Override
    public boolean isBuiltWith(String fontProgram) {
        return Objects.equals(fontParser.fileName, fontProgram);
    }

    public void close() throws java.io.IOException {
        if (fontParser != null) {
            fontParser.close();
        }
        fontParser = null;
    }

    /**
     * The method will update usedGlyphs with additional range or with all glyphs if there is no subset.
     * This set of used glyphs can be used for building width array and ToUnicode CMAP.
     *
     * @param usedGlyphs a set of integers, which are glyph ids that denote used glyphs.
     *                   This set is updated inside the method if needed.
     * @param subset subset status
     * @param subsetRanges additional subset ranges
     */
    public void updateUsedGlyphs(SortedSet<Integer> usedGlyphs, boolean subset, List<int[]> subsetRanges) {
        int[] compactRange = toCompactRange(subsetRanges, subset);
        Set<Integer> missingGlyphs = getMissingGlyphs(compactRange);
        for(Integer glyphId : missingGlyphs) {
            if (getGlyphByCode(glyphId.intValue()) != null) {
                usedGlyphs.add(glyphId.intValue());
            }
        }
    }

    /**
     * The method will update usedGlyphs with additional range or with all glyphs if there is no subset.
     * This map of used glyphs can be used for building width array and ToUnicode CMAP.
     *
     * @param usedGlyphs a map of glyph ids to glyphs. This map is updated inside the method if needed
     * @param subset subset status
     * @param subsetRanges additional subset ranges
     */
    public void updateUsedGlyphs(Map<Integer, Glyph> usedGlyphs, boolean subset, List<int[]> subsetRanges) {
        int[] compactRange = toCompactRange(subsetRanges, subset);
        Set<Integer> missingGlyphs = getMissingGlyphs(compactRange);
        for(Integer glyphId : missingGlyphs) {
            Glyph glyph = getGlyphByCode(glyphId.intValue());
            if (glyph != null && !usedGlyphs.containsKey(glyphId.intValue())) {
                usedGlyphs.put(glyphId.intValue(), glyph);
            }
        }
    }

    /**
     * Normalizes given ranges by making sure that first values in pairs are lower than second values and merges overlapping
     * ranges in one.
     *
     * @param ranges a {@link List} of integer arrays, which are constituted by pairs of ints that denote
     *               each range limits. Each integer array size shall be a multiple of two
     * @param subset {@code true} if a font subset is required. Used only if {@code ranges} is {@code null}
     *
     * @return single merged array consisting of pairs of integers, each of them denoting a range
     */
    private static int[] toCompactRange(List<int[]> ranges, boolean subset) {
        if (ranges == null) {
            if (subset) {
                return new int[]{};
            } else {
                return new int[]{0, 0xFFFF};
            }
        }

        // Ranges are requested
        List<int[]> simp = new ArrayList<>();
        for (int[] range : ranges) {
            for (int j = 0; j < range.length; j += 2) {
                simp.add(new int[]{Math.max(0, Math.min(range[j], range[j + 1])), Math.min(0xffff, Math.max(range[j], range[j + 1]))});
            }
        }
        for (int k1 = 0; k1 < simp.size() - 1; ++k1) {
            for (int k2 = k1 + 1; k2 < simp.size(); ++k2) {
                int[] r1 = simp.get(k1);
                int[] r2 = simp.get(k2);
                if (r1[0] >= r2[0] && r1[0] <= r2[1] || r1[1] >= r2[0] && r1[0] <= r2[1]) {
                    r1[0] = Math.min(r1[0], r2[0]);
                    r1[1] = Math.max(r1[1], r2[1]);
                    simp.remove(k2);
                    --k2;
                }
            }
        }
        int[] s = new int[simp.size() * 2];
        for (int k = 0; k < simp.size(); ++k) {
            int[] r = simp.get(k);
            s[k * 2] = r[0];
            s[k * 2 + 1] = r[1];
        }
        return s;
    }

    private static Set<Integer> getMissingGlyphs(int[] compactRange) {
        Set<Integer> missingGlyphs = new HashSet<>();
        for (int k = 0; k < compactRange.length; k += 2) {
            int from = compactRange[k];
            int to = compactRange[k + 1];
            for (int glyphId = from; glyphId <= to; glyphId++) {
                missingGlyphs.add(glyphId);
            }
        }

        return missingGlyphs;
    }
}
