/*
    This file is part of the iText (R) project.
    Copyright (c) 1998-2025 Apryse Group NV
    Authors: Apryse Software.

    This program is offered under a commercial and under the AGPL license.
    For commercial licensing, contact us at https://itextpdf.com/sales.  For AGPL licensing, see below.

    AGPL licensing:
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package com.itextpdf.commons.json;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

class JsonValueDeserializer extends JsonDeserializer<JsonValue> {

    JsonValueDeserializer() {
        super();
    }

    @Override
    public JsonValue deserialize(JsonParser p, DeserializationContext ctxt)
            throws IOException {
        return readValue(p);
    }

    private JsonValue readValue(JsonParser p) throws IOException {
        JsonToken t = p.currentToken();
        if (t == null) {
            t = p.nextToken();
        }

        switch (t) {
            case VALUE_STRING:
                return new JsonString(p.getValueAsString());
            case VALUE_NUMBER_INT:
                return new JsonNumber(p.getLongValue());
            case VALUE_NUMBER_FLOAT:
                return new JsonNumber(p.getDoubleValue());
            case VALUE_TRUE:
                return JsonBoolean.of(true);
            case VALUE_FALSE:
                return JsonBoolean.of(false);
            case VALUE_NULL:
                return JsonNull.JSON_NULL;
            case START_ARRAY:
                return readArray(p);
            case START_OBJECT:
                return readObject(p);
            default:
                throw new IOException("Unexpected token: " + t);
        }
    }

    private JsonValue readArray(JsonParser p) throws IOException {
        List<JsonValue> list = new ArrayList<>();

        while (true) {
            JsonToken t = p.nextToken();
            if (t == JsonToken.END_ARRAY) {
                break;
            }
            list.add(readValue(p));
        }

        return new JsonArray(list);
    }

    private JsonValue readObject(JsonParser p) throws IOException {
        Map<String, JsonValue> map = new LinkedHashMap<>();

        while (true) {
            JsonToken t = p.nextToken();
            if (t == JsonToken.END_OBJECT) {
                break;
            }

            if (t != JsonToken.FIELD_NAME) {
                throw new IOException("Expected field name, got: " + t);
            }

            String field = p.currentName();
            p.nextToken();
            map.put(field, readValue(p));
        }

        return new JsonObject(map);
    }
}
