/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.jet.sql.impl.connector.file;

import com.hazelcast.jet.sql.SqlTestSupport;
import com.hazelcast.jet.sql.impl.connector.file.FileUtil;
import com.hazelcast.sql.SqlService;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.Collections;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.parquet.hadoop.util.HadoopOutputFile;
import org.apache.parquet.io.OutputFile;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.Assertions;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

public class SqlHadoopTest
extends SqlTestSupport {
    private static MiniDFSCluster cluster;
    private static SqlService sqlService;

    @BeforeClass
    public static void setUpClass() throws Exception {
        SqlHadoopTest.assumeThatNoWindowsOS();
        SqlHadoopTest.assumeHadoopSupportsIbmPlatform();
        SqlHadoopTest.initialize((int)1, null);
        sqlService = SqlHadoopTest.instance().getSql();
        File directory = Files.createTempDirectory("sql-test-hdfs", new FileAttribute[0]).toFile().getAbsoluteFile();
        directory.deleteOnExit();
        Configuration configuration = new Configuration();
        configuration.set("hdfs.minidfs.basedir", directory.getAbsolutePath());
        cluster = new MiniDFSCluster.Builder(configuration).build();
        cluster.waitClusterUp();
    }

    @AfterClass
    public static void afterClass() {
        if (cluster != null) {
            cluster.shutdown();
        }
    }

    @Test
    public void test_csv() throws IOException {
        SqlHadoopTest.store("/csv/file.csv", "id,name\n1,Alice\n2,Bob");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (id BIGINT, name VARCHAR) TYPE File OPTIONS ('format'='csv', 'path'='" + SqlHadoopTest.path("csv") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT name, id FROM " + name, Arrays.asList(new SqlTestSupport.Row("Alice", 1L), new SqlTestSupport.Row("Bob", 2L)));
    }

    @Test
    public void test_csvSchemaDiscovery() throws IOException {
        SqlHadoopTest.store("/discovered-csv/file.csv", "id,name\n1,Alice\n2,Bob");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " TYPE File OPTIONS ('format'='csv', 'path'='" + SqlHadoopTest.path("discovered-csv") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT name, id FROM " + name, Arrays.asList(new SqlTestSupport.Row("Alice", "1"), new SqlTestSupport.Row("Bob", "2")));
    }

    @Test
    public void test_json() throws IOException {
        SqlHadoopTest.store("/json/file.json", "{\"id\": 1, \"name\": \"Alice\"}\n{\"id\": 2, \"name\": \"Bob\"}");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (id BIGINT, name VARCHAR) TYPE File OPTIONS ('format'='json-flat', 'path'='" + SqlHadoopTest.path("json") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT name, id FROM " + name, Arrays.asList(new SqlTestSupport.Row("Alice", 1L), new SqlTestSupport.Row("Bob", 2L)));
    }

    @Test
    public void test_jsonSchemaDiscovery() throws IOException {
        SqlHadoopTest.store("/discovered-json/file.json", "{\"id\": 1, \"name\": \"Alice\"}\n{\"id\": 2, \"name\": \"Bob\"}");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " TYPE File OPTIONS ('format'='json-flat', 'path'='" + SqlHadoopTest.path("discovered-json") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT name, id FROM " + name, Arrays.asList(new SqlTestSupport.Row("Alice", 1.0), new SqlTestSupport.Row("Bob", 2.0)));
    }

    @Test
    public void test_avro() throws IOException {
        SqlHadoopTest.store("/avro/file.avro", FileUtil.createAvroPayload());
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (id BIGINT EXTERNAL NAME long, name VARCHAR EXTERNAL NAME string) TYPE File OPTIONS ('format'='avro', 'path'='" + SqlHadoopTest.path("avro") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT * FROM " + name, Collections.singletonList(new SqlTestSupport.Row(Long.MAX_VALUE, "string")));
    }

    @Test
    public void test_avroSchemaDiscovery() throws IOException {
        SqlHadoopTest.store("/discovered-avro/file.avro", FileUtil.createAvroPayload());
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " TYPE File OPTIONS ('format'='avro', 'path'='" + SqlHadoopTest.path("discovered-avro") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT byte, string FROM " + name, Collections.singletonList(new SqlTestSupport.Row(127, "string")));
    }

    @Test
    public void test_parquet_nulls() throws IOException {
        SqlHadoopTest.storeParquet("/parquet-nulls/file.parquet");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (nonExistingField VARCHAR) TYPE File OPTIONS ('format'='parquet', 'path'='" + SqlHadoopTest.path("parquet-nulls") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT * FROM " + name, Collections.singletonList(new SqlTestSupport.Row(new Object[]{null})));
    }

    @Test
    public void test_parquet_fieldsMapping() throws IOException {
        SqlHadoopTest.storeParquet("/parquet-fields-mapping/file.parquet");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (id TINYINT EXTERNAL NAME byte, name VARCHAR EXTERNAL NAME string) TYPE File OPTIONS ('format'='parquet', 'path'='" + SqlHadoopTest.path("parquet-fields-mapping") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT id, name FROM " + name, Collections.singletonList(new SqlTestSupport.Row((byte)127, "string")));
    }

    @Test
    public void test_parquet_allTypes() throws IOException {
        SqlHadoopTest.storeParquet("/parquet-all-types/file.parquet");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (string VARCHAR, \"boolean\" BOOLEAN, byte TINYINT, short SMALLINT, \"int\" INT, long BIGINT, \"float\" REAL, \"double\" DOUBLE, \"decimal\" DECIMAL, \"time\" TIME, \"date\" DATE, \"timestamp\" TIMESTAMP, timestampTz TIMESTAMP WITH TIME ZONE) TYPE File OPTIONS ( 'format'='parquet', 'path'='" + SqlHadoopTest.path("parquet-all-types") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT * FROM " + name, Collections.singletonList(new SqlTestSupport.Row("string", true, (byte)127, (short)Short.MAX_VALUE, Integer.MAX_VALUE, Long.MAX_VALUE, Float.valueOf(1.234568E9f), 1.234512345678901E14, new BigDecimal("9223372036854775.123"), LocalTime.of(12, 23, 34), LocalDate.of(2020, 4, 15), LocalDateTime.of(2020, 4, 15, 12, 23, 34, 1000000), OffsetDateTime.of(2020, 4, 15, 12, 23, 34, 200000000, ZoneOffset.UTC))));
    }

    @Test
    public void test_parquet_schemaDiscovery() throws IOException {
        SqlHadoopTest.storeParquet("/parquet-schema-discovery/file.parquet");
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " TYPE File OPTIONS ( 'format'='parquet', 'path'='" + SqlHadoopTest.path("parquet-schema-discovery") + "')", new Object[0]);
        SqlHadoopTest.assertRowsAnyOrder("SELECT string, \"boolean\", byte, short, \"int\", long, \"float\", \"double\", \"decimal\", \"time\", \"date\", \"timestamp\", \"timestampTz\" FROM " + name, Collections.singletonList(new SqlTestSupport.Row("string", true, 127, Short.MAX_VALUE, Integer.MAX_VALUE, Long.MAX_VALUE, Float.valueOf(1.234568E9f), 1.234512345678901E14, "9223372036854775.123", "12:23:34", "2020-04-15", "2020-04-15T12:23:34.001", "2020-04-15T12:23:34.200Z")));
    }

    @Test
    public void test_parquet_tableFunction() throws IOException {
        SqlHadoopTest.storeParquet("/parquet-table-function/file.parquet");
        SqlHadoopTest.assertRowsAnyOrder("SELECT string, \"boolean\", byte, short, \"int\", long, \"float\", \"double\", \"decimal\", \"time\", \"date\", \"timestamp\", \"timestampTz\" FROM TABLE (PARQUET_FILE ('" + SqlHadoopTest.path("parquet-table-function") + "'))", Collections.singletonList(new SqlTestSupport.Row("string", true, 127, Short.MAX_VALUE, Integer.MAX_VALUE, Long.MAX_VALUE, Float.valueOf(1.234568E9f), 1.234512345678901E14, "9223372036854775.123", "12:23:34", "2020-04-15", "2020-04-15T12:23:34.001", "2020-04-15T12:23:34.200Z")));
    }

    @Test
    public void when_fileDoesNotExist_thenThrowException() {
        String name = SqlHadoopTest.randomName();
        Assertions.assertThatThrownBy(() -> sqlService.execute("CREATE MAPPING " + name + " TYPE File OPTIONS ( 'format'='csv', 'path'='" + SqlHadoopTest.path("") + "', 'glob'='foo.csv')", new Object[0])).hasMessageContaining("matches no files");
    }

    @Test
    public void when_fileDoesNotExistAndIgnoreFileNotFound_thenReturnNoResults() throws IOException {
        String name = SqlHadoopTest.randomName();
        sqlService.execute("CREATE MAPPING " + name + " (field INT)  TYPE File OPTIONS ( 'format'='csv', 'path'='" + SqlHadoopTest.path("") + "', 'ignoreFileNotFound'='true', 'glob'='foo.csv')", new Object[0]);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)sqlService.execute("SELECT * FROM " + name, new Object[0]).iterator().hasNext()).describedAs("no results from non existing file", new Object[0])).isFalse();
    }

    @Test
    public void when_fileDoesNotExist_thenTableFunctionThrowsException() {
        Assertions.assertThatThrownBy(() -> sqlService.execute("SELECT *  FROM TABLE (csv_file ( path => '" + SqlHadoopTest.path("") + "' , glob => 'foo.csv' , options => MAP['key', 'value']))", new Object[0])).hasMessageContaining("matches no files");
    }

    private static String path(String suffix) throws IOException {
        return String.valueOf(cluster.getFileSystem().getUri()) + "/" + suffix;
    }

    private static void store(String path, String content) throws IOException {
        try (FSDataOutputStream output = cluster.getFileSystem().create(new Path(path));){
            output.writeBytes(content);
        }
    }

    private static void store(String path, byte[] content) throws IOException {
        try (FSDataOutputStream output = cluster.getFileSystem().create(new Path(path));){
            output.write(content);
        }
    }

    private static void storeParquet(String path) throws IOException {
        HadoopOutputFile file = HadoopOutputFile.fromPath((Path)new Path(path), (Configuration)cluster.getFileSystem().getConf());
        FileUtil.writeParquetPayloadTo((OutputFile)file);
    }
}

