/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner;

import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutureCallback;
import com.google.api.core.ApiFutures;
import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.core.FixedCredentialsProvider;
import com.google.api.gax.core.NoCredentialsProvider;
import com.google.api.gax.rpc.PermissionDeniedException;
import com.google.auth.Credentials;
import com.google.cloud.monitoring.v3.MetricServiceClient;
import com.google.cloud.monitoring.v3.MetricServiceSettings;
import com.google.cloud.spanner.BuiltInMetricsConstant;
import com.google.cloud.spanner.SpannerCloudMonitoringExporterUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.monitoring.v3.CreateTimeSeriesRequest;
import com.google.monitoring.v3.ProjectName;
import com.google.monitoring.v3.TimeSeries;
import com.google.protobuf.Empty;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.metrics.InstrumentType;
import io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.export.MetricExporter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.threeten.bp.Duration;

class SpannerCloudMonitoringExporter
implements MetricExporter {
    private static final Logger logger = Logger.getLogger(SpannerCloudMonitoringExporter.class.getName());
    private static final String MONITORING_ENDPOINT = (String)MoreObjects.firstNonNull((Object)System.getProperty("spanner.test-monitoring-endpoint"), (Object)MetricServiceSettings.getDefaultEndpoint());
    private static final int EXPORT_BATCH_SIZE_LIMIT = 200;
    private final AtomicBoolean spannerExportFailureLogged = new AtomicBoolean(false);
    private CompletableResultCode lastExportCode;
    private final MetricServiceClient client;
    private final String spannerProjectId;

    static SpannerCloudMonitoringExporter create(String projectId, @Nullable Credentials credentials) throws IOException {
        MetricServiceSettings.Builder settingsBuilder = MetricServiceSettings.newBuilder();
        Object credentialsProvider = credentials == null ? NoCredentialsProvider.create() : FixedCredentialsProvider.create((Credentials)credentials);
        settingsBuilder.setCredentialsProvider((CredentialsProvider)credentialsProvider);
        settingsBuilder.setEndpoint(MONITORING_ENDPOINT);
        Duration timeout = Duration.ofMinutes((long)1L);
        settingsBuilder.createServiceTimeSeriesSettings().setSimpleTimeoutNoRetries(timeout);
        return new SpannerCloudMonitoringExporter(projectId, MetricServiceClient.create((MetricServiceSettings)settingsBuilder.build()));
    }

    @VisibleForTesting
    SpannerCloudMonitoringExporter(String projectId, MetricServiceClient client) {
        this.client = client;
        this.spannerProjectId = projectId;
    }

    public CompletableResultCode export(Collection<MetricData> collection) {
        if (this.client.isShutdown()) {
            logger.log(Level.WARNING, "Exporter is shut down");
            return CompletableResultCode.ofFailure();
        }
        this.lastExportCode = this.exportSpannerClientMetrics(collection);
        return this.lastExportCode;
    }

    private CompletableResultCode exportSpannerClientMetrics(Collection<MetricData> collection) {
        List<TimeSeries> spannerTimeSeries;
        List<MetricData> spannerMetricData = collection.stream().filter(md -> BuiltInMetricsConstant.SPANNER_METRICS.contains(md.getName())).collect(Collectors.toList());
        if (spannerMetricData.isEmpty()) {
            return CompletableResultCode.ofSuccess();
        }
        if (!spannerMetricData.stream().flatMap(metricData -> metricData.getData().getPoints().stream()).allMatch(pd -> this.spannerProjectId.equals(SpannerCloudMonitoringExporterUtils.getProjectId(pd)))) {
            logger.log(Level.WARNING, "Metric data has a different projectId. Skipping export.");
            return CompletableResultCode.ofFailure();
        }
        try {
            spannerTimeSeries = SpannerCloudMonitoringExporterUtils.convertToSpannerTimeSeries(spannerMetricData);
        }
        catch (Throwable e) {
            logger.log(Level.WARNING, "Failed to convert spanner metric data to cloud monitoring timeseries.", e);
            return CompletableResultCode.ofFailure();
        }
        final ProjectName projectName = ProjectName.of((String)this.spannerProjectId);
        ApiFuture<List<Empty>> futureList = this.exportTimeSeriesInBatch(projectName, spannerTimeSeries);
        final CompletableResultCode spannerExportCode = new CompletableResultCode();
        ApiFutures.addCallback(futureList, (ApiFutureCallback)new ApiFutureCallback<List<Empty>>(){

            public void onFailure(Throwable throwable) {
                if (SpannerCloudMonitoringExporter.this.spannerExportFailureLogged.compareAndSet(false, true)) {
                    String msg = "createServiceTimeSeries request failed for spanner metrics.";
                    if (throwable instanceof PermissionDeniedException) {
                        msg = msg + String.format(" Need monitoring metric writer permission on project=%s.", projectName.getProject());
                    }
                    logger.log(Level.WARNING, msg, throwable);
                }
                spannerExportCode.fail();
            }

            public void onSuccess(List<Empty> empty) {
                SpannerCloudMonitoringExporter.this.spannerExportFailureLogged.set(false);
                spannerExportCode.succeed();
            }
        }, (Executor)MoreExecutors.directExecutor());
        return spannerExportCode;
    }

    private ApiFuture<List<Empty>> exportTimeSeriesInBatch(ProjectName projectName, List<TimeSeries> timeSeries) {
        ArrayList<ApiFuture> batchResults = new ArrayList<ApiFuture>();
        for (List batch : Iterables.partition(timeSeries, (int)200)) {
            CreateTimeSeriesRequest req = CreateTimeSeriesRequest.newBuilder().setName(projectName.toString()).addAllTimeSeries((Iterable)batch).build();
            batchResults.add(this.client.createServiceTimeSeriesCallable().futureCall((Object)req));
        }
        return ApiFutures.allAsList(batchResults);
    }

    public CompletableResultCode flush() {
        return CompletableResultCode.ofSuccess();
    }

    public CompletableResultCode shutdown() {
        if (this.client.isShutdown()) {
            logger.log(Level.WARNING, "shutdown is called multiple times");
            return CompletableResultCode.ofSuccess();
        }
        CompletableResultCode shutdownResult = new CompletableResultCode();
        try {
            this.client.shutdown();
            shutdownResult.succeed();
        }
        catch (Throwable e) {
            logger.log(Level.WARNING, "failed to shutdown the monitoring client", e);
            shutdownResult.fail();
        }
        return shutdownResult;
    }

    public AggregationTemporality getAggregationTemporality(InstrumentType instrumentType) {
        return AggregationTemporality.CUMULATIVE;
    }
}

