/*
 * Decompiled with CFR 0.152.
 */
package com.arcmutate.spring.licence;

import com.arcmutate.spring.licence.LicenceType;
import com.arcmutate.spring.licence.PathFinder;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.X509EncodedKeySpec;
import java.time.Clock;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.OptionalInt;
import java.util.Properties;
import java.util.function.BiConsumer;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.pitest.util.Log;

public final class Licence {
    private static final Logger LOG = Log.getLogger();
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("dd/MM/yyyy");
    private final LocalDate expires;
    private final LicenceType type;
    private final List<String> packages;
    private final List<String> products;
    private final int users;

    Licence(LocalDate expires, LicenceType type, List<String> packages, List<String> products, int users) {
        this.expires = expires;
        this.type = type;
        this.packages = packages;
        this.products = products;
        this.users = users;
    }

    public static Licence findAndCheckLicence(Clock clock, Path reportDir, Path workingDir, PathFinder repoFinder, String product) {
        try (Stream<Licence> possible = Licence.findBestLicence(reportDir, workingDir, repoFinder, product);){
            Licence found = possible.findFirst().orElseThrow(() -> new IllegalStateException("No licence found. See https://www.arcmutate.com for details."));
            Licence.checkLicence(found, clock, product);
            Licence licence = found;
            return licence;
        }
    }

    private static Stream<Licence> findBestLicence(Path reportDir, Path workingDir, PathFinder repoFinder, String product) {
        Stream<Licence> preferred = Licence.pathsToLicences(Licence.reportDir(reportDir.resolve("arcmutate-licences")), product);
        Stream<Licence> found = Licence.checkDotPitestDir(workingDir, product);
        List<Licence> legacy = Licence.findLicenceAtPath(repoFinder, "arcmutate-licence.txt");
        if (legacy.isEmpty()) {
            legacy = Licence.findLicenceAtPath(repoFinder, "cdg-pitest-licence.txt");
        }
        return Stream.concat(preferred, Stream.concat(found, legacy.stream()));
    }

    private static List<Licence> findLicenceAtPath(PathFinder repoFinder, String file) {
        return repoFinder.findPath().map(p -> p.resolve(file)).filter(x$0 -> Files.exists(x$0, new LinkOption[0])).filter(Files::isReadable).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(Licence::asInputStream).map(i -> Licence.streamToLicence(i)).map(Stream::of).orElseGet(Stream::empty).collect(Collectors.toList());
    }

    private static Stream<Licence> checkDotPitestDir(Path workingDir, String product) {
        if (workingDir == null) {
            return Stream.empty();
        }
        return Licence.pathsToLicences(Licence.pitestDir(workingDir), product);
    }

    private static Stream<Licence> pathsToLicences(Stream<Path> paths, String product) {
        return paths.filter(x$0 -> Files.exists(x$0, new LinkOption[0])).filter(Files::isReadable).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(Licence::asInputStream).map(i -> Licence.streamToLicence(i)).sorted(Licence.compare(product).reversed());
    }

    private static Comparator<Licence> compare(String product) {
        return (a, b) -> {
            if (a.covers(product) && b.covers(product)) {
                return a.expires.compareTo(b.expires);
            }
            if (a.covers(product)) {
                return 1;
            }
            if (b.covers(product)) {
                return -1;
            }
            return a.expires.compareTo(b.expires);
        };
    }

    boolean covers(String product) {
        return this.products.isEmpty() || this.products.contains(product);
    }

    private static Stream<Path> reportDir(Path dir) {
        if (Files.exists(dir, new LinkOption[0])) {
            try {
                return Files.walk(dir.toRealPath(new LinkOption[0]), 1, new FileVisitOption[0]).filter(p -> p.getFileName().toString().contains("licence"));
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return Stream.empty();
    }

    private static Stream<Path> pitestDir(Path workingDir) {
        Path dir = workingDir.resolve(".pitest");
        if (Files.exists(dir, new LinkOption[0])) {
            try {
                return Files.walk(dir.toRealPath(new LinkOption[0]), 3, new FileVisitOption[0]).filter(p -> p.getFileName().toString().contains("licence"));
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return Stream.empty();
    }

    private static InputStream asInputStream(Path p) {
        try {
            return Files.newInputStream(p, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    static void checkLicence(Licence licence, Clock clock, String product) {
        LocalDate now = LocalDate.now(clock);
        LocalDate failDate = licence.expires().plusMonths(1L);
        LOG.info(licence.describe());
        if (now.isAfter(failDate)) {
            throw new RuntimeException("Licence expired on " + licence.expires() + " and grace period ended.");
        }
        if (now.isAfter(licence.expires())) {
            LOG.severe("Licence expired on " + licence.expires() + ". Please renew licence or build will start to fail on " + failDate);
        }
        if (!licence.products.isEmpty() && !licence.products.contains(product)) {
            throw new RuntimeException("Licenced products " + licence.products + " do not include " + product);
        }
    }

    public LocalDate expires() {
        return this.expires;
    }

    public LicenceType type() {
        return this.type;
    }

    public List<String> packages() {
        return Collections.unmodifiableList(this.packages);
    }

    public OptionalInt users() {
        if (this.users <= 0) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(this.users);
    }

    public String describe() {
        String users = this.users().isPresent() ? String.format("Valid for " + this.users().getAsInt() + " users%n", new Object[0]) : "";
        return String.format("%n>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> %n" + this.type().description() + "%nExpires: " + this.expires() + "%nFor use with packages: " + String.join((CharSequence)", ", this.packages()) + "%n" + users + "See https://www.arcmutate.com%n>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> %n", new Object[0]);
    }

    private static Licence streamToLicence(InputStream stream) {
        try {
            Properties props = new Properties();
            try (BufferedInputStream bs = new BufferedInputStream(stream);){
                props.load(bs);
            }
            Licence.checkSignature(Licence.asMap(props));
            return Licence.makeLicence(props);
        }
        catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    private static Licence makeLicence(Properties props) {
        List<String> packages = Licence.fromCommaSeparated(props.getProperty("packages"));
        List<String> products = Licence.fromCommaSeparated(props.getProperty("products"));
        LicenceType type = LicenceType.valueOf(props.getProperty("type").toUpperCase());
        int users = Integer.parseInt(props.getOrDefault((Object)"users", "0").toString());
        return new Licence(LocalDate.parse(props.getProperty("expires"), DATE_FORMATTER), type, packages, products, users);
    }

    private static List<String> fromCommaSeparated(String list) {
        if (list == null) {
            return Collections.emptyList();
        }
        return Arrays.stream(list.split(",")).collect(Collectors.toList());
    }

    private static void checkSignature(Map<String, String> values) {
        String data = Licence.dataWithoutSignature(values);
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(Licence.decode(Licence.encodedPublicKey()));
            PublicKey publicKey = keyFactory.generatePublic(publicKeySpec);
            Signature sig = Signature.getInstance("SHA256withRSA");
            sig.initVerify(publicKey);
            sig.update(data.getBytes(StandardCharsets.UTF_8));
            if (!sig.verify(Licence.getSignatureBytes(values))) {
                throw new RuntimeException("Invalid licence file");
            }
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    private static String encodedPublicKey() {
        return "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCN7xxTpwIhc3m1qj9Uhr04BGeLHbGIK/C2KSJ7TcYK168/2jTUbOzvjeLStOCTvnCwdkUxx3yBuNtCCXHaW4X7tXyyz4973v66L5UEQwqt4OtAaBthIdKjzy8W1qVqt3jRtEAxbwfiWA33wN2ldJbL5bCKLAlgHQ6ZmZAa1b9vGwIDAQAB";
    }

    private static byte[] decode(String key) {
        return Base64.getDecoder().decode(key.getBytes(StandardCharsets.UTF_8));
    }

    private static String dataWithoutSignature(Map<String, String> values) {
        return values.entrySet().stream().filter(pair -> !((String)pair.getKey()).equals("signature")).sorted(Map.Entry.comparingByKey()).map(pair -> ((String)pair.getKey()).trim() + ":" + ((String)pair.getValue()).trim()).collect(Collectors.joining("\n"));
    }

    private static byte[] getSignatureBytes(Map<String, String> values) {
        String encoded = values.get("signature");
        return Base64.getDecoder().decode(encoded);
    }

    private static Map<String, String> asMap(Properties props) {
        HashMap<String, String> map = new HashMap<String, String>();
        props.forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(key, value) -> map.put(key.toString(), value.toString())));
        return map;
    }
}

