/*
 * Copyright 2016-2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *    http://aws.amazon.com/apache2.0
 *
 * This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and
 * limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import java.util.Date;
import java.util.TimeZone;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 * Annotation to format a timestamp object using Java's standard date and time
 * patterns.
 *
 * <pre class="brush: java">
 * &#064;DynamoDBTypeConvertedTimestamp(pattern=&quot;yyyyMMddHHmmssSSS&quot;, timeZone=&quot;UTC&quot;)
 * public Date getCreatedDate()
 * </pre>
 *
 * <p>Supports the standard {@link Date} type-conversions; such as
 * {@link java.util.Calendar}, {@link Long}.</p>
 *
 * <p>Primitives such as {@code long} are not supported since the unset
 * (or null) state can't be detected.</p>
 *
 * <p>Compatible with {@link DynamoDBAutoGeneratedTimestamp}</p>
 *
 * @see com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBAutoGeneratedTimestamp
 * @see com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBTypeConverted
 * @see java.text.SimpleDateFormat
 * @see java.util.TimeZone
 */
@DynamoDBTypeConverted(converter=DynamoDBTypeConvertedTimestamp.Converter.class)
@DynamoDBTyped(DynamoDBMapperFieldModel.DynamoDBAttributeType.S)
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.TYPE, ElementType.FIELD, ElementType.METHOD})
public @interface DynamoDBTypeConvertedTimestamp {

    /**
     * The pattern format; default is ISO8601.
     * @see java.text.SimpleDateFormat
     */
    String pattern() default "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";

    /**
     * The time zone; default is {@code UTC}.
     * @see java.util.TimeZone
     */
    String timeZone() default "UTC";

    /**
     * Timestamp format converter.
     */
    static final class Converter<T> implements DynamoDBTypeConverter<String,T> {
        private final DynamoDBTypeConverter<DateTime,T> converter;
        private final DateTimeFormatter formatter;

        public Converter(Class<T> targetType, DynamoDBTypeConvertedTimestamp annotation) {
            this.formatter = DateTimeFormat.forPattern(annotation.pattern()).withZone(
                DateTimeZone.forTimeZone(StandardTypeConverters.Scalar.TIME_ZONE.<TimeZone>convert(annotation.timeZone()))
            );
            this.converter = StandardTypeConverters.factory().getConverter(DateTime.class, targetType);
        }

        @Override
        public final String convert(final T object) {
            return formatter.print(converter.convert(object));
        }

        @Override
        public final T unconvert(final String object) {
            return converter.unconvert(formatter.parseDateTime(object));
        }
    }

}
