package app.raybritton.elog.ui

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.os.Bundle
import android.view.LayoutInflater
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.DefaultItemAnimator
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import app.raybritton.elog.ELog
import app.raybritton.elog.ELogConfig
import app.raybritton.elog.R
import app.raybritton.elog.arch.LogModule
import app.raybritton.elog.data.LogFile
import kotlinx.android.synthetic.main.elog_activity_log_list.*
import kotlinx.android.synthetic.main.elog_element_log_file.view.*
import java.text.SimpleDateFormat
import java.util.*
import kotlin.concurrent.thread
import kotlin.math.log

class LogListActivity : Activity() {

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.elog_activity_log_list)
        elog_list_toolbar.setBackgroundColor(ELogConfig.toolbarBackgroundColor)
        elog_list_toolbar.setTitleTextColor(ELogConfig.toolbarForegroundColor)
        getDrawable(R.drawable.elog_ic_close)?.setTint(ELogConfig.toolbarForegroundColor)
        getDrawable(R.drawable.elog_ic_send)?.setTint(ELogConfig.toolbarForegroundColor)
        setStatusBarColor(ELogConfig.toolbarBackgroundColor)

        setActionBar(elog_list_toolbar)
        actionBar?.setTitle(R.string.elog_list_title)
        actionBar?.setDisplayHomeAsUpEnabled(true)
        actionBar?.setDisplayShowHomeEnabled(true)
        actionBar?.setHomeAsUpIndicator(R.drawable.elog_ic_close)

        elog_list_content.layoutManager = LinearLayoutManager(this)
        elog_list_content.itemAnimator = DefaultItemAnimator()
        thread {
            val logs = LogModule.logManager.getLogs()
            runOnUiThread {
                elog_list_content.adapter = LogAdapter(logs, ::onViewLog, ::onSendLog)
                elog_list_progress.visibility = View.GONE
            }
        }
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        if (item.itemId == android.R.id.home) {
            finish()
        }
        return super.onOptionsItemSelected(item)
    }

    private fun onViewLog(logFile: LogFile) {
        LogViewerActivity.start(this, logFile)
    }

    private fun onSendLog(logFile: LogFile) {
        if (logFile.code == null) {
            LogSubmitActivity.start(this, logFile)
        } else {
            AlreadySentActivity.start(this, logFile.code)
        }
    }

    private class LogAdapter(
        private val logFiles: List<LogFile>,
        private val onViewLog: (LogFile) -> Unit, private val onSendLog: (LogFile) -> Unit
    ) : RecyclerView.Adapter<LogAdapter.LogFileViewHolder>() {
        val dateFormatter = SimpleDateFormat("yyyy/MM/dd HH:mm:ss", Locale.UK)

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): LogFileViewHolder {
            val view = LayoutInflater.from(parent.context).inflate(R.layout.elog_element_log_file, parent, false)
            return LogFileViewHolder(view)
        }

        override fun getItemCount() = logFiles.size

        override fun onBindViewHolder(holder: LogFileViewHolder, position: Int) {
            val logFile = logFiles[position]
            holder.itemView.elog_log_file_timestamp.text = holder.itemView.elog_log_file_timestamp.context.getString(R.string.elog_session_started, dateFormatter.format(Date(logFile.created)))
            if (logFile.code == null) {
                holder.itemView.elog_log_file_code.setTextColor(Color.BLACK)
                holder.itemView.elog_log_file_code.setText(R.string.elog_not_submitted)
                holder.itemView.elog_log_file_code.paint.isFakeBoldText = false
            } else {
                holder.itemView.elog_log_file_code.setTextColor(holder.getColorById(R.color.elog_code))
                holder.itemView.elog_log_file_code.text = logFile.code
                holder.itemView.elog_log_file_code.paint.isFakeBoldText = true
            }
        }

        private inner class LogFileViewHolder(view: View) : RecyclerView.ViewHolder(view) {
            init {
                view.setOnClickListener {
                    onViewLog(logFiles[adapterPosition])
                }
                view.elog_log_file_send.setOnClickListener {
                    onSendLog(logFiles[adapterPosition])
                }
            }
        }
    }

    companion object {
        internal fun start(ctx: Context) {
            val intent = Intent(ctx, LogListActivity::class.java)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            ctx.startActivity(intent)
        }
    }
}
