package xyz.truenight.databinding.rxjava

import androidx.databinding.ObservableChar
import io.reactivex.Flowable
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import xyz.truenight.utils.optional.Optional
import xyz.truenight.utils.optional.safe
import java.util.concurrent.CopyOnWriteArraySet

/**
 * Copyright (C) 2017 Mikhail Frolov
 */

class RxObservableChar private constructor(val observable: Observable<Char>, default: Char) : ObservableChar(default) {

    private val count = CopyOnWriteArraySet<androidx.databinding.Observable.OnPropertyChangedCallback>()

    private var subscription: Disposable? = null

    override fun addOnPropertyChangedCallback(callback: androidx.databinding.Observable.OnPropertyChangedCallback) {
        super.addOnPropertyChangedCallback(callback)
        if (count.isEmpty()) {
            subscription = observable.subscribe { super.set(it) }
        }
        count.add(callback)
    }

    override fun removeOnPropertyChangedCallback(callback: androidx.databinding.Observable.OnPropertyChangedCallback) {
        super.removeOnPropertyChangedCallback(callback)
        count.remove(callback)
        if (count.isEmpty()) {
            subscription?.dispose()
        }
    }

    companion object {
        private const val DEFAULT = '\u0000'

        private fun safe(value: Optional<Char>?) = value?.value.safe { DEFAULT }

        @JvmStatic
        @JvmOverloads
        fun Observable<Char>.toBinding(default: Char = DEFAULT) = RxObservableChar(this, default)

        @JvmStatic
        @JvmOverloads
        fun Observable<Optional<Char>>.toBindingOptional(default: Char = DEFAULT) = RxObservableChar(this.map { safe(it) }, default)

        @JvmStatic
        @JvmOverloads
        fun Flowable<Char>.toBinding(default: Char = DEFAULT) = RxObservableChar(this.toObservable(), default)

        @JvmStatic
        @JvmOverloads
        fun Flowable<Optional<Char>>.toBindingOptional(default: Char = DEFAULT) = RxObservableChar(this.map { safe(it) }.toObservable(), default)
    }
}
