package wu.poker.library.view;

import android.animation.Animator;
import android.animation.AnimatorSet;
import android.animation.ValueAnimator;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.animation.AccelerateDecelerateInterpolator;
import android.widget.TextView;

import wu.poker.library.R;

/**
 * Created by pokerwu on 16-12-1.
 */

public class RippleSelectView extends TextView {
    public final static int MODE_INNER = 0;
    public final static int MODE_OUT = 1;

    private Drawable bgNormal;
    private Drawable bgSelect;
    //view的矩形区域
    private Rect bounds;
    //持续时间
    private int duration;
    private int mode;

    //按下的位置
    private float downX;
    private float downY;

    private float centerX;
    private float centerY;
    private float rippleX;
    private float rippleY;
    //最大圆形半径
    private int maxRippleRadius;
    private int rippleRadius;

    private boolean isAnimator;

    private boolean isSelect;

    private Paint paint;

    private RippleAnimatorListener listener;
    private int rippleColor = 0xff14c128;


    public RippleSelectView(Context context) {
        super(context);
        init();
    }

    public RippleSelectView(Context context, AttributeSet attrs) {
        super(context, attrs);
        getDrawable(attrs);
        init();
    }

    public RippleSelectView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        getDrawable(attrs);
        init();
    }

    private void getDrawable(AttributeSet attrs) {

        TypedArray array = getContext().obtainStyledAttributes(attrs, R.styleable.RippleSelectView);
        bgNormal = array.getDrawable(R.styleable.RippleSelectView_ripple_bg_normal);
        bgSelect = array.getDrawable(R.styleable.RippleSelectView_ripple_bg_select);
        duration = array.getInt(R.styleable.RippleSelectView_ripple_duration, 500);
        mode = array.getInt(R.styleable.RippleSelectView_ripple_mode, MODE_INNER);
        rippleColor = array.getColor(R.styleable.RippleSelectView_ripple_color, rippleColor);
        array.recycle();
    }

    private void init() {
        if (bgNormal == null) if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            bgNormal = getContext().getDrawable(R.drawable.drawable_normal);
        } else {
            bgNormal = getContext().getResources().getDrawable(R.drawable.drawable_normal);
        }


        if (bgSelect == null) if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            bgSelect = getContext().getDrawable(R.drawable.drawable_select);
        } else {
            bgSelect = getContext().getResources().getDrawable(R.drawable.drawable_select);
        }
        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setColor(rippleColor);
        //paint.setFlags();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        //bgNormal.draw(canvas);

        if (isAnimator) {
            canvas.drawCircle(rippleX, rippleY, rippleRadius, paint);
        }
        super.onDraw(canvas);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (isAnimator) return super.onTouchEvent(event);

        if (event.getAction() == MotionEvent.ACTION_DOWN) {

            downX = event.getX();
            downY = event.getY();
        }
        return super.onTouchEvent(event);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        bounds = new Rect(0, 0, w, h);
        centerX = w / 2;
        centerY = h / 2;

        bgNormal.setBounds(bounds);
        if (mode == MODE_INNER) maxRippleRadius = Math.max(w / 2, h / 2);
        else
            maxRippleRadius = (int) Math.sqrt(w / 2 * w / 2 + h / 2 * h / 2);
    }

    public boolean isSelect() {
        return isSelect;
    }

    public void setSelect(boolean select) {
        if (isAnimator) return;
        isSelect = select;
        if (select) {
            animateToSelect();
        } else
            animateToNormal();
    }

    private void animateToNormal() {
        rippleAnimator(maxRippleRadius,
                maxRippleRadius / 5, centerX, downX, centerY, downY, duration);
    }

    private void animateToSelect() {
        rippleAnimator(maxRippleRadius / 5,
                maxRippleRadius, downX, centerX, downY, centerY, duration);
    }

    private void rippleAnimator(int rFrom, int rTo, float xFrom, float xTo,
                                float yFrom, float yTo, int duration) {
        if (!isSelect) duration = (int) (3.0f / 4 * duration);

        RippleAnimation animation = new RippleAnimation(rFrom,
                rTo, xFrom, xTo, yFrom, yTo, duration);
        animation.setRadiusUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                rippleRadius = (Integer) valueAnimator.getAnimatedValue();
                float alpha = rippleRadius * 1.0f / maxRippleRadius * 255;
                paint.setAlpha((int) alpha);
                if (!isSelect)
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                        setBackground(bgNormal);
                    } else
                        setBackgroundDrawable(bgNormal);
                invalidate();
                // Log.e(TAG,rippleRadius +"");
            }
        }).setCenterXUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                rippleX = (Float) valueAnimator.getAnimatedValue();
            }
        }).setCenterYUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                rippleY = (Float) valueAnimator.getAnimatedValue();

            }
        }).setListener(new Animator.AnimatorListener() {
            @Override
            public void onAnimationStart(Animator animator) {
                isAnimator = true;
                if (listener != null) listener.onStart();
            }

            @Override
            public void onAnimationEnd(Animator animator) {
                isAnimator = false;
                if (isSelect)
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                        setBackground(bgSelect);
                    } else
                        setBackgroundDrawable(bgSelect);
                else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                    setBackground(bgNormal);
                } else
                    setBackgroundDrawable(bgNormal);
                //setBackground(bgNormal);
                rippleX = 0;
                rippleY = 0;
                rippleRadius = 0;

                if (listener != null) listener.onEnd();
            }

            @Override
            public void onAnimationCancel(Animator animator) {
                isAnimator = false;

                if (listener != null) listener.onCancel();
            }

            @Override
            public void onAnimationRepeat(Animator animator) {

            }
        }).start();
    }

    public RippleAnimatorListener getListener() {
        return listener;
    }

    public void setListener(RippleAnimatorListener listener) {
        this.listener = listener;
    }

    private class RippleAnimation {
        private AnimatorSet animatorSet;
        private ValueAnimator radiusAnimator;
        private ValueAnimator centerXAnimator;
        private ValueAnimator centerYAnimator;

        RippleAnimation(int rFrom, int rTo, float xFrom, float xTo, float yFrom, float yTo, int duration) {
            animatorSet = new AnimatorSet();
            animatorSet.setDuration(duration);

            radiusAnimator = ValueAnimator.ofInt(rFrom, rTo);
            centerXAnimator = ValueAnimator.ofFloat(xFrom, xTo);
            centerYAnimator = ValueAnimator.ofFloat(yFrom, yTo);
            animatorSet.playTogether(radiusAnimator, centerXAnimator, centerYAnimator);
            animatorSet.setInterpolator(new AccelerateDecelerateInterpolator());
            //radiusAnimator.setRepeatCount(1);
        }

        void start() {
            animatorSet.start();
        }

        RippleAnimation setRadiusUpdateListener(ValueAnimator.AnimatorUpdateListener listener) {
            radiusAnimator.addUpdateListener(listener);

            return this;
        }

       /* RippleAnimation setRadiusListener(Animator.AnimatorListener listener) {
            radiusAnimator.addListener(listener);

            return this;
        }*/

        RippleAnimation setCenterXUpdateListener(ValueAnimator.AnimatorUpdateListener listener) {
            centerXAnimator.addUpdateListener(listener);

            return this;
        }

        /*RippleAnimation setCenterXListener(Animator.AnimatorListener listener) {
            centerXAnimator.addListener(listener);

            return this;
        }*/
        RippleAnimation setCenterYUpdateListener(ValueAnimator.AnimatorUpdateListener listener) {
            centerYAnimator.addUpdateListener(listener);

            return this;
        }

        RippleAnimation setListener(Animator.AnimatorListener listener) {
            animatorSet.addListener(listener);

            return this;
        }

    }

    public interface RippleAnimatorListener {
        void onStart();

        void onEnd();

        void onCancel();
    }
}
