/*
 * Decompiled with CFR 0.152.
 */
package convex.core;

import convex.core.Block;
import convex.core.BlockResult;
import convex.core.Constants;
import convex.core.ErrorCodes;
import convex.core.Result;
import convex.core.ResultContext;
import convex.core.data.ABlob;
import convex.core.data.ACell;
import convex.core.data.AIndex;
import convex.core.data.AMap;
import convex.core.data.ARecord;
import convex.core.data.ASequence;
import convex.core.data.AVector;
import convex.core.data.AccountKey;
import convex.core.data.AccountStatus;
import convex.core.data.Address;
import convex.core.data.Blob;
import convex.core.data.Cells;
import convex.core.data.Format;
import convex.core.data.Hash;
import convex.core.data.IRefFunction;
import convex.core.data.Index;
import convex.core.data.Keyword;
import convex.core.data.Keywords;
import convex.core.data.LongBlob;
import convex.core.data.MapEntry;
import convex.core.data.PeerStatus;
import convex.core.data.Ref;
import convex.core.data.SignedData;
import convex.core.data.Strings;
import convex.core.data.Symbol;
import convex.core.data.Vectors;
import convex.core.data.prim.CVMLong;
import convex.core.exceptions.BadFormatException;
import convex.core.exceptions.BadSignatureException;
import convex.core.exceptions.InvalidDataException;
import convex.core.lang.AOp;
import convex.core.lang.Context;
import convex.core.lang.Juice;
import convex.core.lang.RT;
import convex.core.lang.Symbols;
import convex.core.lang.impl.RecordFormat;
import convex.core.transactions.ATransaction;
import convex.core.util.Counters;
import convex.core.util.Economics;
import convex.core.util.Utils;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class State
extends ARecord {
    public static final Index<ABlob, AVector<ACell>> EMPTY_SCHEDULE = Index.none();
    public static final Index<AccountKey, PeerStatus> EMPTY_PEERS = Index.none();
    private static final Keyword[] STATE_KEYS = new Keyword[]{Keywords.ACCOUNTS, Keywords.PEERS, Keywords.GLOBALS, Keywords.SCHEDULE};
    private static final RecordFormat FORMAT = RecordFormat.of(STATE_KEYS);
    public static final AVector<Symbol> GLOBAL_SYMBOLS = Vectors.of(Symbols.TIMESTAMP, Symbols.FEES, Symbols.JUICE_PRICE, Symbols.MEMORY, Symbols.MEMORY_VALUE, Symbols.PROTOCOL);
    public static final int GLOBAL_TIMESTAMP = 0;
    public static final int GLOBAL_FEES = 1;
    public static final int GLOBAL_JUICE_PRICE = 2;
    public static final int GLOBAL_MEMORY_MEM = 3;
    public static final int GLOBAL_MEMORY_CVX = 4;
    public static final int GLOBAL_PROTOCOL = 5;
    public static final State EMPTY = State.create(Vectors.empty(), EMPTY_PEERS, Constants.INITIAL_GLOBALS, EMPTY_SCHEDULE);
    private static final Logger log = LoggerFactory.getLogger((String)State.class.getName());
    private final AVector<AccountStatus> accounts;
    private final Index<AccountKey, PeerStatus> peers;
    private final AVector<ACell> globals;
    private final Index<ABlob, AVector<ACell>> schedule;

    private State(AVector<AccountStatus> accounts, Index<AccountKey, PeerStatus> peers, AVector<ACell> globals, Index<ABlob, AVector<ACell>> schedule) {
        super(FORMAT.count());
        this.accounts = accounts;
        this.peers = peers;
        this.globals = globals;
        this.schedule = schedule;
    }

    @Override
    public ACell get(Keyword k) {
        if (Keywords.ACCOUNTS.equals(k)) {
            return this.accounts;
        }
        if (Keywords.PEERS.equals(k)) {
            return this.peers;
        }
        if (Keywords.GLOBALS.equals(k)) {
            return this.globals;
        }
        if (Keywords.SCHEDULE.equals(k)) {
            return this.schedule;
        }
        return null;
    }

    @Override
    public int getRefCount() {
        int rc = this.accounts.getRefCount();
        rc += this.peers.getRefCount();
        rc += this.globals.getRefCount();
        return rc += this.schedule.getRefCount();
    }

    @Override
    public <R extends ACell> Ref<R> getRef(int i) {
        if (i < 0) {
            throw new IndexOutOfBoundsException(i);
        }
        int c = this.accounts.getRefCount();
        if (i < c) {
            return this.accounts.getRef(i);
        }
        if ((i -= c) < (c = this.peers.getRefCount())) {
            return this.peers.getRef(i);
        }
        if ((i -= c) < (c = this.globals.getRefCount())) {
            return this.globals.getRef(i);
        }
        if ((i -= c) < (c = this.schedule.getRefCount())) {
            return this.schedule.getRef(i);
        }
        throw new IndexOutOfBoundsException(i -= c);
    }

    @Override
    public State updateRefs(IRefFunction func) {
        ACell newAccounts = this.accounts.updateRefs(func);
        ACell newPeers = this.peers.updateRefs(func);
        ACell newGlobals = this.globals.updateRefs(func);
        ACell newSchedule = this.schedule.updateRefs(func);
        if (this.accounts == newAccounts && this.peers == newPeers && this.globals == newGlobals && this.schedule == newSchedule) {
            return this;
        }
        return new State((AVector<AccountStatus>)newAccounts, (Index<AccountKey, PeerStatus>)newPeers, (AVector<ACell>)newGlobals, (Index<ABlob, AVector<ACell>>)newSchedule);
    }

    public static State create(AVector<AccountStatus> accounts, Index<AccountKey, PeerStatus> peers, AVector<ACell> globals, Index<ABlob, AVector<ACell>> schedule) {
        return new State(accounts, peers, globals, schedule);
    }

    @Override
    public int encode(byte[] bs, int pos) {
        bs[pos++] = this.getTag();
        return this.encodeRaw(bs, pos);
    }

    @Override
    public int encodeRaw(byte[] bs, int pos) {
        pos = this.accounts.encode(bs, pos);
        pos = this.peers.encode(bs, pos);
        pos = this.globals.encode(bs, pos);
        pos = this.schedule.encode(bs, pos);
        return pos;
    }

    @Override
    public int getEncodingLength() {
        int length = 1;
        length += this.accounts.getEncodingLength();
        length += this.peers.getEncodingLength();
        length += this.globals.getEncodingLength();
        return length += this.schedule.getEncodingLength();
    }

    @Override
    public int estimatedEncodingSize() {
        int est = 1;
        est += this.accounts.estimatedEncodingSize();
        est += this.peers.estimatedEncodingSize();
        est += this.globals.estimatedEncodingSize();
        return est += this.schedule.estimatedEncodingSize();
    }

    public static State read(Blob b, int pos) throws BadFormatException {
        int epos = pos + 1;
        AVector accounts = (AVector)Format.read(b, epos);
        if (accounts == null) {
            throw new BadFormatException("Null accounts!");
        }
        Index peers = (Index)Format.read(b, epos += Format.getEncodingLength(accounts));
        if (peers == null) {
            throw new BadFormatException("Null peers!");
        }
        AVector globals = (AVector)Format.read(b, epos += Format.getEncodingLength(peers));
        if (globals == null) {
            throw new BadFormatException("Null globals!");
        }
        Index schedule = (Index)Format.read(b, epos += Format.getEncodingLength(globals));
        if (schedule == null) {
            throw new BadFormatException("Null schedule!");
        }
        State result = State.create(accounts, peers, globals, schedule);
        result.attachEncoding(b.slice(pos, epos += Format.getEncodingLength(schedule)));
        return result;
    }

    public AVector<AccountStatus> getAccounts() {
        return this.accounts;
    }

    public Index<AccountKey, PeerStatus> getPeers() {
        return this.peers;
    }

    public Long getBalance(Address address) {
        AccountStatus acc = this.getAccount(address);
        if (acc == null) {
            return null;
        }
        return acc.getBalance();
    }

    public State withBalance(Address address, long newBalance) {
        AccountStatus acc = this.getAccount(address);
        if (acc == null) {
            throw new Error("No account for " + String.valueOf(address));
        }
        acc = acc.withBalance(newBalance);
        return this.putAccount(address, acc);
    }

    public BlockResult applyBlock(SignedData<Block> signedBlock) {
        Block block = signedBlock.getValue();
        ++Counters.applyBlock;
        BlockResult maybeFailed = this.checkBlock(signedBlock);
        if (maybeFailed != null) {
            return maybeFailed;
        }
        State state = this.prepareBlock(block);
        return state.applyTransactions(block);
    }

    public BlockResult checkBlock(SignedData<Block> signedBlock) {
        Block block = signedBlock.getValue();
        AccountKey peerKey = signedBlock.getAccountKey();
        PeerStatus ps = this.peers.get(peerKey);
        if (ps == null) {
            return BlockResult.createInvalidBlock(this, block, Strings.MISSING_PEER);
        }
        if (ps.getPeerStake() < 1000000000L) {
            return BlockResult.createInvalidBlock(this, block, Strings.INSUFFICIENT_STAKE);
        }
        if (block.getTimeStamp() < ps.getTimestamp()) {
            return BlockResult.createInvalidBlock(this, block, Strings.MISORDERED_BLOCK);
        }
        if (block.getTimeStamp() < this.getTimestamp().longValue() - 60000L) {
            return BlockResult.createInvalidBlock(this, block, Strings.BACKDATED_BLOCK);
        }
        if (block.getTransactions().count() > 1024L) {
            return BlockResult.createInvalidBlock(this, block, Strings.ILLEGAL_BLOCK_SIZE);
        }
        return null;
    }

    private State prepareBlock(Block b) {
        State state = this;
        state = state.applyTimeUpdate(b.getTimeStamp());
        state = state.applyScheduledTransactions();
        return state;
    }

    public State applyTimeUpdate(long newTimestamp) {
        State state = this;
        ASequence glbs = state.globals;
        long oldTimestamp = ((CVMLong)((AVector)glbs).get(0)).longValue();
        if (newTimestamp <= oldTimestamp) {
            return this;
        }
        glbs = ((AVector)glbs).assoc(0L, (ACell)CVMLong.create(newTimestamp));
        long memAdditions = newTimestamp / 86400000L - oldTimestamp / 86400000L;
        if (memAdditions > 0L) {
            long mem = ((CVMLong)((AVector)glbs).get(3)).longValue();
            long add = memAdditions * 1000000L;
            if (add > 0L) {
                long newMem = mem + add;
                glbs = ((AVector)glbs).assoc(3L, (ACell)CVMLong.create(newMem));
            } else {
                throw new Error("Bad memory additions?");
            }
        }
        state = state.withGlobals((AVector<ACell>)glbs);
        return state;
    }

    public State applyScheduledTransactions() {
        MapEntry<ABlob, AVector<ACell>> me;
        ABlob key;
        long time;
        long tcount = 0L;
        AIndex sched = this.schedule;
        CVMLong timestamp = this.getTimestamp();
        ArrayList al = null;
        while (tcount < 100L && !sched.isEmpty() && (time = (key = (ABlob)(me = sched.entryAt(0L)).getKey()).longValue()) <= timestamp.longValue()) {
            ASequence trans = (AVector)me.getValue();
            long numScheduled = trans.count();
            long take = Math.min(numScheduled, 100L - tcount);
            if (al == null) {
                al = new ArrayList();
            }
            for (long i = 0L; i < take; ++i) {
                al.add(((AVector)trans).get(i));
            }
            if (!(trans = ((AVector)trans).slice(take, numScheduled)).isEmpty()) continue;
            sched = sched.dissoc(key);
        }
        if (al == null) {
            return this;
        }
        State state = this.withSchedule((Index<ABlob, AVector<ACell>>)sched);
        int n = al.size();
        log.debug("Applying {} scheduled transactions", (Object)n);
        for (int i = 0; i < n; ++i) {
            AVector st = (AVector)al.get(i);
            Address origin = (Address)st.get(0);
            AOp op = (AOp)st.get(1);
            Context ctx = Context.createInitial(state, origin, 10000000L);
            if ((ctx = ctx.run(op)).isExceptional()) {
                log.trace("Scheduled transaction error: {}", (Object)ctx.getExceptional());
                continue;
            }
            state = ctx.getState();
            log.trace("Scheduled transaction succeeded");
        }
        return state;
    }

    private State withSchedule(Index<ABlob, AVector<ACell>> newSchedule) {
        if (this.schedule == newSchedule) {
            return this;
        }
        return new State(this.accounts, this.peers, this.globals, newSchedule);
    }

    private State withGlobals(AVector<ACell> newGlobals) {
        if (newGlobals == this.globals) {
            return this;
        }
        return new State(this.accounts, this.peers, newGlobals, this.schedule);
    }

    private BlockResult applyTransactions(Block block) {
        State state = this;
        int blockLength = block.length();
        Result[] results = new Result[blockLength];
        AVector<SignedData<ATransaction>> transactions = block.getTransactions();
        for (int i = 0; i < blockLength; ++i) {
            try {
                SignedData signed = (SignedData)transactions.get(i);
                ResultContext rc = state.applyTransaction(signed);
                results[i] = Result.fromContext(CVMLong.create(i), rc);
                state = rc.context.getState();
                continue;
            }
            catch (Exception t) {
                String msg = "Unexpected fatal exception applying transaction: " + t.toString();
                results[i] = Result.create(CVMLong.create(i), Strings.create(msg), ErrorCodes.UNEXPECTED);
                log.error(msg, (Throwable)t);
            }
        }
        return BlockResult.create(state, results);
    }

    public ResultContext applyTransaction(SignedData<? extends ATransaction> signedTransaction) throws BadSignatureException {
        long expectedSequence;
        ATransaction t = signedTransaction.getValue();
        Address addr = t.getOrigin();
        AccountStatus as = this.getAccount(addr);
        if (as == null) {
            return ResultContext.error(this, ErrorCodes.NOBODY, "Transaction for non-existent Account: " + String.valueOf(addr));
        }
        long sequence = t.getSequence();
        if (sequence != (expectedSequence = as.getSequence() + 1L)) {
            return ResultContext.error(this, ErrorCodes.SEQUENCE, "Sequence = " + sequence + " but expected " + expectedSequence);
        }
        AccountKey key = as.getAccountKey();
        if (key == null) {
            return ResultContext.error(this, ErrorCodes.NOBODY, "Transaction for account that is an Actor: " + String.valueOf(addr));
        }
        boolean sigValid = signedTransaction.checkSignature(key);
        if (!sigValid) {
            return ResultContext.error(this, ErrorCodes.SIGNATURE, Strings.BAD_SIGNATURE);
        }
        ResultContext ctx = this.applyTransaction(t);
        return ctx;
    }

    private ResultContext createResultContext(ATransaction t) {
        long juicePrice = this.getJuicePrice().longValue();
        ResultContext rc = new ResultContext(t, juicePrice);
        return rc;
    }

    public ResultContext applyTransaction(ATransaction t) {
        ResultContext rc = this.createResultContext(t);
        Context ctx = this.prepareTransaction(rc);
        if (!ctx.isExceptional()) {
            State preparedState = ctx.getState();
            rc.context = ctx = t.apply(ctx);
            ctx = ctx.completeTransaction(preparedState, rc);
        }
        return rc.withContext(ctx);
    }

    private Context prepareTransaction(ResultContext rc) {
        long initialJuice;
        ATransaction t = rc.tx;
        long juicePrice = rc.juicePrice;
        Address origin = t.getOrigin();
        AccountStatus account = this.getAccount(origin);
        if (account == null) {
            return Context.createFake(this).withError(ErrorCodes.NOBODY);
        }
        long balance = account.getBalance();
        long juiceLimit = Juice.calcAvailable(balance, juicePrice);
        if ((juiceLimit = Math.min(10000000L, juiceLimit)) <= (initialJuice = 0L)) {
            return Context.createFake(this, origin).withJuiceError();
        }
        Context ctx = Context.createInitial(this, origin, juiceLimit);
        ctx = ctx.withJuice(initialJuice);
        return ctx;
    }

    @Override
    public boolean isCanonical() {
        return true;
    }

    public HashMap<AccountKey, Double> computeStakes() {
        HashMap<AccountKey, Double> hm = new HashMap<AccountKey, Double>(this.peers.size());
        long timeStamp = this.getTimestamp().longValue();
        Double totalStake = this.peers.reduceEntries((? super R acc, MapEntry<K, V> e) -> {
            PeerStatus ps = (PeerStatus)e.getValue();
            double stake = ps.getTotalStake();
            long peerTimestamp = ps.getTimestamp();
            double decay = Economics.stakeDecay(timeStamp, peerTimestamp);
            hm.put(RT.ensureAccountKey((ACell)e.getKey()), stake *= decay);
            return stake + acc;
        }, 0.0);
        hm.put(null, totalStake);
        return hm;
    }

    public State withAccounts(AVector<AccountStatus> newAccounts) {
        if (newAccounts == this.accounts) {
            return this;
        }
        return State.create(newAccounts, this.peers, this.globals, this.schedule);
    }

    public State putAccount(Address address, AccountStatus accountStatus) {
        long n;
        long ix = address.longValue();
        if (ix > (n = this.accounts.count())) {
            throw new IndexOutOfBoundsException("Trying to add an account beyond accounts array at position: " + ix);
        }
        ASequence newAccounts = ix == n ? this.accounts.conj(accountStatus) : this.accounts.assoc(ix, (ACell)accountStatus);
        return this.withAccounts((AVector<AccountStatus>)newAccounts);
    }

    public AccountStatus getAccount(Address target) {
        long ix = target.longValue();
        if (ix < 0L || ix >= this.accounts.count()) {
            return null;
        }
        return this.accounts.get(ix);
    }

    public AMap<Symbol, ACell> getEnvironment(Address addr) {
        AccountStatus as = this.getAccount(addr);
        if (as == null) {
            return null;
        }
        return as.getEnvironment();
    }

    public State withPeers(Index<AccountKey, PeerStatus> newPeers) {
        if (this.peers == newPeers) {
            return this;
        }
        return State.create(this.accounts, newPeers, this.globals, this.schedule);
    }

    @Override
    public byte getTag() {
        return -96;
    }

    public State addActor() {
        AccountStatus as = AccountStatus.createActor();
        ASequence newAccounts = this.accounts.conj(as);
        return this.withAccounts((AVector<AccountStatus>)newAccounts);
    }

    public long computeTotalFunds() {
        long total = this.accounts.reduce((acc, as) -> acc + as.getBalance(), 0L);
        total += this.peers.reduceValues((? super R acc, ? super V ps) -> acc + ps.getBalance(), 0L).longValue();
        total += this.getGlobalFees().longValue();
        return total += this.getGlobalMemoryValue().longValue();
    }

    public long computeTotalMemory() {
        long total = this.accounts.reduce((acc, as) -> acc + as.getMemory(), 0L);
        return total += this.getGlobalMemoryPool().longValue();
    }

    @Override
    public void validate() throws InvalidDataException {
        super.validate();
    }

    @Override
    public void validateCell() throws InvalidDataException {
        this.accounts.validateCell();
        this.peers.validateCell();
        this.globals.validateCell();
        this.schedule.validateCell();
    }

    public CVMLong getTimestamp() {
        return (CVMLong)this.globals.get(0);
    }

    public CVMLong getJuicePrice() {
        return (CVMLong)this.globals.get(2);
    }

    public State scheduleOp(long time, Address address, AOp<?> op) {
        AVector v = Vectors.of(address, op);
        LongBlob key = LongBlob.create(time);
        AVector<ACell> list = this.schedule.get(key);
        list = list == null ? Vectors.of(v) : list.append(v);
        AIndex newSchedule = this.schedule.assoc(key, list);
        return this.withSchedule((Index<ABlob, AVector<ACell>>)newSchedule);
    }

    public Index<ABlob, AVector<ACell>> getSchedule() {
        return this.schedule;
    }

    public CVMLong getGlobalFees() {
        return (CVMLong)this.globals.get(1);
    }

    public State withGlobalFees(CVMLong newFees) {
        return this.withGlobals((AVector<ACell>)this.globals.assoc(1L, (ACell)newFees));
    }

    public PeerStatus getPeer(AccountKey peerAddress) {
        return this.getPeers().get(peerAddress);
    }

    public State withPeer(AccountKey peerKey, PeerStatus updatedPeer) {
        return this.withPeers((Index<AccountKey, PeerStatus>)this.peers.assoc(peerKey, updatedPeer));
    }

    public Address nextAddress() {
        return Address.create(this.accounts.count());
    }

    public Address lookupCNS(String name) {
        Context ctx = Context.createFake(this);
        return (Address)ctx.lookupCNS(name).getResult();
    }

    public AVector<ACell> getGlobals() {
        return this.globals;
    }

    public State withTimestamp(long timestamp) {
        return this.withGlobals((AVector<ACell>)this.globals.assoc(0L, (ACell)CVMLong.create(timestamp)));
    }

    @Override
    public boolean equals(ACell a) {
        if (!(a instanceof State)) {
            return false;
        }
        State as = (State)a;
        return this.equals(as);
    }

    public boolean equals(State a) {
        Hash ha;
        if (this == a) {
            return true;
        }
        if (a == null) {
            return false;
        }
        Hash h = this.cachedHash();
        if (h != null && (ha = a.cachedHash()) != null) {
            return Cells.equals(h, ha);
        }
        if (!Cells.equals(this.accounts, a.accounts)) {
            return false;
        }
        if (!Cells.equals(this.globals, a.globals)) {
            return false;
        }
        if (!Cells.equals(this.peers, a.peers)) {
            return false;
        }
        return Cells.equals(this.schedule, a.schedule);
    }

    @Override
    public RecordFormat getFormat() {
        return FORMAT;
    }

    public CVMLong getGlobalMemoryValue() {
        return (CVMLong)this.globals.get(4);
    }

    public CVMLong getGlobalMemoryPool() {
        return (CVMLong)this.globals.get(3);
    }

    public double getMemoryPrice() {
        long pool = this.getGlobalMemoryPool().longValue();
        long value = this.getGlobalMemoryValue().longValue();
        return (double)value / (double)pool;
    }

    public State updateMemoryPool(long cvx, long mem) {
        ASequence r = this.globals;
        r = ((AVector)r).assoc(4L, (ACell)CVMLong.create(cvx));
        r = ((AVector)r).assoc(3L, (ACell)CVMLong.create(mem));
        return this.withGlobals((AVector<ACell>)r);
    }

    public boolean hasAccount(Address address) {
        long av = address.longValue();
        return av >= 0L && av < this.accounts.count();
    }

    public static AVector<State> statesAsOfRange(AVector<State> states, CVMLong timestamp, long interval, int count) {
        ASequence v = Vectors.empty();
        for (int i = 0; i < count; ++i) {
            v = v.conj(State.stateAsOf(states, timestamp));
            timestamp = CVMLong.create(timestamp.longValue() + interval);
        }
        return v;
    }

    public static State stateAsOf(AVector<State> states, CVMLong timestamp) {
        return Utils.binarySearchLeftmost(states, State::getTimestamp, Comparator.comparingLong(CVMLong::longValue), timestamp);
    }
}

