/*
========================================================================
SchemaCrawler
http://www.schemacrawler.com
Copyright (c) 2000-2017, Sualeh Fatehi <sualeh@hotmail.com>.
All rights reserved.
------------------------------------------------------------------------

SchemaCrawler is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

SchemaCrawler and the accompanying materials are made available under
the terms of the Eclipse Public License v1.0, GNU General Public License
v3 or GNU Lesser General Public License v3.

You may elect to redistribute this code under any of these licenses.

The Eclipse Public License is available at:
http://www.eclipse.org/legal/epl-v10.html

The GNU General Public License v3 and the GNU Lesser General Public
License v3 are available at:
http://www.gnu.org/licenses/

========================================================================
*/

package schemacrawler.crawl;


import static java.util.Objects.requireNonNull;

import org.threeten.bp.LocalDateTime;
import org.threeten.bp.format.DateTimeFormatter;

import schemacrawler.schema.CrawlInfo;
import schemacrawler.schema.DatabaseInfo;
import schemacrawler.schema.JdbcDriverInfo;
import schemacrawler.schema.SchemaCrawlerInfo;

/**
 * SchemaCrawler information.
 *
 * @author Sualeh Fatehi sualeh@hotmail.com
 */
final class ImmutableCrawlInfo
  implements CrawlInfo
{

  private static final long serialVersionUID = 5982990326485881993L;

  private final String schemaCrawlerInfo;
  private final String jdbcDriverInfo;
  private final String databaseInfo;
  private final String title;
  private final LocalDateTime crawlTimestamp;

  ImmutableCrawlInfo(final SchemaCrawlerInfo schemaCrawlerInfo,
                     final JdbcDriverInfo jdbcDriverInfo,
                     final DatabaseInfo databaseInfo,
                     final String title)
  {
    requireNonNull(schemaCrawlerInfo);
    this.schemaCrawlerInfo = String
      .format("%s %s",
              schemaCrawlerInfo.getSchemaCrawlerProductName(),
              schemaCrawlerInfo.getSchemaCrawlerVersion());

    requireNonNull(jdbcDriverInfo);
    this.jdbcDriverInfo = String.format("%s %s",
                                        jdbcDriverInfo.getDriverName(),
                                        jdbcDriverInfo.getDriverVersion());

    requireNonNull(databaseInfo);
    this.databaseInfo = String.format("%s %s",
                                      databaseInfo.getProductName(),
                                      databaseInfo.getProductVersion());

    this.title = title;
    crawlTimestamp = LocalDateTime.now();
  }

  @Override
  public LocalDateTime getCrawlTimestamp()
  {
    return crawlTimestamp;
  }

  @Override
  public String getDatabaseInfo()
  {
    return databaseInfo;
  }

  @Override
  public String getJdbcDriverInfo()
  {
    return jdbcDriverInfo;
  }

  @Override
  public String getSchemaCrawlerInfo()
  {
    return schemaCrawlerInfo;
  }

  @Override
  public String getTitle()
  {
    return title;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString()
  {
    final StringBuilder info = new StringBuilder(1024);
    info.append("-- generated by: ").append(schemaCrawlerInfo)
      .append(System.lineSeparator());
    info
      .append("-- generated on: ").append(DateTimeFormatter
        .ofPattern("yyyy-MM-dd HH:mm:ss").format(getCrawlTimestamp()))
      .append(System.lineSeparator());
    info.append("-- database: ").append(databaseInfo)
      .append(System.lineSeparator());
    info.append("-- driver: ").append(jdbcDriverInfo)
      .append(System.lineSeparator());
    return info.toString();
  }

}
